#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------






# File: fileselect.tcl
#
# Description:
#       fileselect -- display a file selection dialog and get the user's reply
#
# Global variables:
#c      fileselect_priv(command,$w)
#               Command to run when a file is selected in the dialog box $w
#c      fileselect_priv(dirname,$w)
#               Directory that is displayed in the dialog box $w
#c      fileselect_priv(dirvariable,$w)
#               Global variable in which $w's current directory name is
#               to be stored.
#c      fileselect_priv(filter,$w)
#               Function used to select only certain files for display in
#               dialog box $w.
#c      fileselect_priv(new,$w)
#               Boolean variable == 1 if new files may be created by file
#               selection in $w, 0 of only existing files are acceptable.
#c      fileselect_priv(pattern.$w)
#               Pattern restricting the files to select in $w.
#c      fileselect_priv(textvariable,$w)
#               Variable in which to store the name of the file selected in $w.


# Procedure: fileselect
#
# Synopsis:
#       Display a file selection dialog box.
#
# Usage:
#c      fileselect pathName ?-option value?...
#
# Parameters:
#c      pathName
#               Path name of the file selection dialog box, which will be
#               created as a transient window for its parent.
#
# Options:
#       Name:                   action
#       Class:                  Action
#       Command-line Switch:    -a, -action
#       Default:                `OK'
#               Name of the button that selects the file.
#       Name:                   command
#       Class:                  Command
#       Command-line Switch:    -c, -command
#       Default:                None.
#               If nonempty, `command' is the text of a Tcl command to run
#               when a file is selected.  The file name is appended to the
#               command.
#       Name:                   directory
#       Class:                  Directory
#       Command-line Switch:    -d, -directory
#       Default:                `.'
#               Directory at which to begin the search.
#       Name:                   dirOnly
#       Class:                  Directory
#       Command-line Switch:    -dirOnly
#       Default:                0
#               Restricts selectable values to be directories only.
#       Name:                   dirVariable
#       Class:                  DirVariable
#       Command-line switch:    -dv, -dirvariable
#       Default:                None.
#               Name of a global variable in which to store the name of the
#               working directory of the dialog.  If null, no global variable
#               is used.  If this option and `-directory' are both present,
#               and the variable is already set, the variable's value takes
#               precedence over the `-directory' option.  In this way, a user
#               can maintain a `working directory' across several invocations
#               of `fileselect'.
#       Name:                   filter
#       Class:                  Filter
#       Command-line switch:    -filter, -f
#       Default:                `fileselect:filter'
#               Tcl function that selects a subset of files to display.
#               The dialog will display only those files that cause the
#               specified function to return a true value when passed
#               the file name.  The default is `fileselect:filter', which
#               returns a true value if a file exists, and a false value
#               otherwise.
#       Name:                   message
#       Class:                  Message
#       Command-line switch:    -message, -m
#       Default:                `Select a file:'
#               Message to be displayed on the top line of the dialog box.
#       Name:                   new
#       Class:                  New
#       Command-line switch:    -new, -n
#       Default:                0
#               Boolean value. 1 if a new file is acceptable, and 0
#               if only a pre-existing file is OK.
#       Name:                   pattern
#       Class:                  Pattern
#       Command-line switch:    -pattern, -p
#       Default:                {*}
#               List of items to match against file names.  Only file names
#               that match at least one of the specified items will be
#               displayed.  The items are matched according to the rules
#               for `string match'.
#       Name:                   textVariable
#       Class:                  TextVariable
#       Command-line switch:    -textvariable, -textv
#       Default:                None.
#               Global variable in which to store the name of the selected
#               file.  If null, the file name is not stored.
#
# Return value:
#       Path name of the dialog box, suitable for packing, use with
#       `transient', and so on.
#
# Description:
#       `fileselect' displays a file selection dialog box, with a Motif-like
#       look and feel.
#
# Bugs:
#       - There is no way to configure a fileselect dialog to select
#         a directory.  This should be changed.
#       - There should be control over whether the selection is exported.
#       - There should be a `Rescan' operation.
#       - The `fileselect' is not a first class widget; it does not
#         respond to the `config' widget command.

 # Default option settings:

option add *Fileselect.action OK widgetDefault
option add *Fileselect.command {} widgetDefault
option add *Fileselect.directory . widgetDefault
option add *Fileselect.dirVariable {} widgetDefault
option add *Fileselect.dirOnly 0 widgetDefault
option add *Fileselect.filter fileselect:filter widgetDefault
option add *Fileselect.message "Select a file:" widgetDefault
option add *Fileselect.new 0 widgetDefault
option add *Fileselect.pattern "*" widgetDefault
option add *Fileselect.textVariable {} widgetDefault
        # Place button text in resources so that the user can customize it.
option add *Fileselect.b.c.b.text "Cancel" widgetDefault
option add *Fileselect.d.head.text "Directory:" widgetDefault
option add *Fileselect.patt.head.text "Filter:" widgetDefault
option add *Fileselect.patt.head.width 10 widgetDefault
option add *Fileselect.patt.head.anchor w widgetDefault
option add *Fileselect.patt.entry.relief sunken widgetDefault
option add *Fileselect.f.head.text "File name:" widgetDefault
option add *Fileselect.f.head.width 10 widgetDefault
option add *Fileselect.f.head.anchor w widgetDefault
option add *Fileselect.b.o.borderWidth 2 widgetDefault
option add *Fileselect.b.o.relief sunken widgetDefault
option add *Fileselect.f.entry.Font -*-courier-medium-r-*-120-* widgetDefault
option add *Fileselect.f.entry.relief sunken widgetDefault
option add *Fileselect.l.list.relief sunken widgetDefault
        # The indented geometry options are in the following list because the
        # packer appears to override the non-indented ones.
option add *Fileselect.e.height 20 widgetDefault
        option add *Fileselect.e.geometry 1x20 widgetDefault
option add *Fileselect.f.r.width 40 widgetDefault
        option add *Fileselect.f.r.geometry 40x1 widgetDefault
option add *Fileselect.patt.r.width 40 widgetDefault
        option add *Fileselect.patt.r.geometry 40x1 widgetDefault
option add *Fileselect.l.l.width 40 widgetDefault
        option add *Fileselect.l.l.geometry 40x1 widgetDefault
option add *Fileselect.l.r.width 40 widgetDefault
        option add *Fileselect.l.r.geometry 40x1 widgetDefault
option add *Fileselect.x.l.width 40 widgetDefault
        option add *Fileselect.x.l.geometry 40x1 widgetDefault
option add *Fileselect.x.r.width 60 widgetDefault
        option add *Fileselect.x.r.geometry 40x1 widgetDefault
        # Following option will be replaced someday with `width' and `height'
option add *Fileselect.l.list.geometry 32x10 widgetDefault
option add *Fileselect*info*background #ffe4c4  widgetDefault
option add *Fileselect*info*foreground black widgetDefault

proc fileselect {w args} {
        global fileselect_priv

        # Make the window

        frame $w -class Fileselect
        widget_addBinding $w Destroy "fileselect:destroy $w"

        # Defaults

        set action [option get $w action Action]
        set command [option get $w command Command]
        set dirname [option get $w directory Directory]
        set dirvar [option get $w dirVariable DirVariable]
        set dironly [option get $w dirOnly DirOnly]
        set filter  [option get $w filter Filter]
        set message [option get $w message Message]
        set newfile [option get $w new New]
        set pattern [option get $w pattern Pattern]
        set textvar [option get $w textVariable TextVariable]

        # Parse command line

        while {[llength $args] > 0} {
                set string [lindex $args 0]
                set args [lrange $args 1 end]
                case $string in {
                        {-a -action} {
                                set action [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-c -command} {
                                set command [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-d -directory} {
                                set dirname [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-dironly} {
                                set dironly [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-dv -dirvariable} {
                                set dirvar [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-f -filter} {
                                set filter [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-m -message} {
                                set message [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-n -new} {
                                set newfile [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-p -pattern} {
                                set pattern [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        {-textv -textvariable} {
                                set textvar [lindex $args 0]
                                set args [lrange $args 1 end]
                        }
                        default {
                                destroy $w
                                error "$string: unknown option"
                        }
                }
        }

        # Store options in globals

        set fileselect_priv(command,$w) $command
        set fileselect_priv(dirname,$w) $dirname
        set fileselect_priv(dirvariable,$w) $dirvar
        set fileselect_priv(dironly,$w) $dironly
        set fileselect_priv(filter,$w) $filter
        set fileselect_priv(new,$w) $newfile
        set fileselect_priv(pattern,$w) $pattern
        set fileselect_priv(textvariable,$w) $textvar

        # Create the subwindows

        # Prompt message

        global gui_flags
        fontcheck label $w.m -text $message -font $gui_flags(font,labels)

        # Directory name

        pack [frame $w.d]
        pack [fontcheck label $w.d.head -font $gui_flags(font,labels)] -side left
        pack [fontcheck label $w.d.name -width 40 -font $gui_flags(font,labels) -anchor w] -side left

        # Entry for file name

        frame $w.f
        pack [fontcheck label $w.f.head -font $gui_flags(font,labels)] -side left
        pack [fontcheck entry $w.f.entry  -font $gui_flags(font,textentry)] -side left -expand yes -fill x
        pack [frame $w.f.r] -side left
        
        frame $w.patt
        pack [fontcheck label $w.patt.head -font $gui_flags(font,labels)] -side left
        pack [fontcheck entry $w.patt.entry  -font $gui_flags(font,textentry)] -side left -expand yes -fill x
        pack [frame $w.patt.r] -side left
        
        bind $w.patt.entry <Return> "fileselect:select $w ."
        # Pad

        frame $w.e

        # Scrollbar for the file list

        frame $w.x
        frame $w.x.l
        scrollbar $w.x.s -command "$w.l.list xview" -orient horizontal
        frame $w.x.r
        pack append $w.x $w.x.l {left} $w.x.s {left expand fillx} $w.x.r {left}

        # Listbox to show files

        frame $w.l
        frame $w.l.l
        scrollbar $w.l.scroll -command "$w.l.list yview"
        fontcheck listbox $w.l.list -yscroll "$w.l.scroll set" -xscroll "$w.x.s set"  -font $gui_flags(font,listboxes)
        frame $w.l.c
        frame $w.l.r
        pack append $w.l \
                $w.l.l {left} \
                $w.l.list {left expand fill} \
                $w.l.scroll {left filly} \
                $w.l.r {left}

        tk_listboxSingleSelect $w.l.list

        bind $w.l.list <Any-ButtonRelease-1> {
                fileselect:copyEntry [winfo parent [winfo parent %W]] \
                                [fileselect_slash [%W get [%W nearest %y]]]
        }
        bind $w.l.list <Double-Button-1> {
                fileselect:select [winfo parent [winfo parent %W]] \
                                [fileselect_slash [%W get [%W nearest %y]]]
        }

        # Buttons

        pack [frame $w.b] -side bottom -pady 10
        pack [fontcheck focusable button $w.b.o -command "fileselect:ok $w" \
                -text $action  \
                -font $gui_flags(font,buttons)] \
            -side left -padx 10
        if $fileselect_priv(dironly,$w) {
            pack [fontcheck focusable button $w.b.open_dir \
                        -command "fileselect:open_dir $w" \
                        -text "Open Dir"  \
                        -font $gui_flags(font,buttons)] \
                -side left -padx 10
        }
        pack [fontcheck focusable button $w.b.filt -command "fileselect:select $w ." \
                -text "Filter"  \
                -font $gui_flags(font,buttons)] \
            -side left -padx 10
        pack [fontcheck focusable button $w.b.c \
                -command "fileselect:cancel $w" \
                -font $gui_flags(font,buttons)] \
            -side left -padx 10

        pack append $w \
                        $w.m {top expand frame w} \
                        $w.d {top expand fillx} \
                        $w.f {top expand fillx} \
                        $w.patt {top expand fillx} \
                        $w.e {top} \
                        $w.l {top expand fill} \
                        $w.x {top expand fillx} 

        # Fill in the default file name

        $w.patt.entry insert 0 $pattern
        if {$textvar != ""} {
                upvar #0 $textvar tv
                if {[info exists tv]} {
                        $w.f.entry delete 0 end
                        $w.f.entry insert 0 [file tail $tv]
                        entry_cursor $w.f.entry
                }
        }
        if {$dirvar != ""} {
                upvar #0 $dirvar dv
                if {[info exists dv] && $dv != "" 
                    && [file isdirectory $dv] && [file readable $dv]} {
                        set dirname $dv
                }
        }

        # Move into the specified directory
        wm protocol [winfo toplevel $w] WM_DELETE_WINDOW "catch {fileselect_priv(textvariable,$w) {}}"

        after 1 fileselect:dir $w [list $dirname/]

}

# Procedure:    fileselect:cancel
#
# Synopsis:
#       Cancel a file selection request.
#
# Usage:
#c      fileselect:cancel pathName
#
# Parameters:
#c      pathName
#               Path name of a file selecction dialog
#
# Return value:
#       None.
#
# Description:
#       `fileselect:cancel' is invoked when the user presses the `Cancel'
#       button in a file selection dialog.  It sets the text variable to
#       the null string, executes the selection command giving the null string
#       as an argument, and returns.

proc fileselect:cancel {w} {
        global fileselect_priv
        if {$fileselect_priv(textvariable,$w) != ""} {
                upvar #0 $fileselect_priv(textvariable,$w) tv
                set tv {}
        }
        if {$fileselect_priv(command,$w) != ""} {
                uplevel #0 $fileselect_priv(command,$w) {}
        }
}

# Procedure:    fileselect:copyEntry
#
# Synopsis:
#       Select an entry from the listbox in a file selection dialog.
#
# Usage:
#c      fileselect:copyEntry pathName text
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box
#c      text
#               File name.
#
# Return value:
#       None.
#
# Description:
#       `fileselect:copyEntry' is invoked in response to a change of the
#       selection within a file selection dialog box.  It copies
#       the current selection (given as the second parameter) to the
#       entry box so that the user can edit it if desired.

proc fileselect:copyEntry {w text} {
        catch {$w.f.entry delete 0 end}
        catch {$w.f.entry insert 0 $text ; entry_cursor $w}
}

# Procedure:    fileselect:destroy
#
# Synopsis:
#       Clean up when a file selection dialog is destroyed.
#
# Usage:
#c      fileselect:destroy pathName
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box.
#
# Return value:
#       None.
#
# Description:
#       `fileselect:destroy' is invoked when a file selection dialog box is
#       destroyed.  It removes all the private variables associated with the
#       dialog.

proc fileselect:destroy w {
        global fileselect_priv
        widget_unsetPriv fileselect_priv $w {
                command         dirname         dirvariable
                filter          new             pattern
                textvariable
        }
}

# Procedure:    fileselect:dir
#
# Synopsis:
#       Traverse to a new directory in a file selection dialog.
#
# Usage:
#c      fileselect:dir pathName directory
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box.
#c      directory
#               Directory name to read
#
# Return value:
#       None.
#
# Description:
#       `fileselect:dir' is invoked when the directory on display in a file
#       selection dialog box changes.  It scans the files in the new directory,
#       matches them against the pattern and filter, and rebuilds the list
#       box.  It locks the application in a modal dialog while scanning,
#       in order to avoid spurious selections.

proc fileselect:dir {w dirname} {
        global fileselect_priv

        set pattern [$w.patt.entry get]
        set filter $fileselect_priv(filter,$w)
        set status [catch {exec /bin/sh -cef "cd $dirname. && pwd"} dirname]
        set status [catch {glob $dirname} dirname]
        if {$status} {
                errormessage $w.error $dirname
                return
        }
        global busy
        incr busy
        global feedback
        set feedback [list $w "Scanning $dirname"]
#       modalDialog transient label $w.info -text "Scanning $dirname"
        set fileselect_priv(dirname,$w) $dirname/
        if {$fileselect_priv(dirvariable,$w) != ""} {
                upvar #0 $fileselect_priv(dirvariable,$w) dv
                set dv $dirname
        }

        if [string length $dirname]>40 {
           $w.d.name config -text "...[string range $dirname [expr [string length $dirname]-37] end]"
        } else {
           $w.d.name config -text $dirname
        }
        catch {$w.f.entry delete 0 end}
        catch {$w.l.list delete 0 end}
#       set status [catch {split [exec ls -a $dirname] \n} fileList]
        set status [catch {glob -nocomplain $dirname/*} fileList]
        set fileList [lsort $fileList]
        if {$status == 0} {
                set fileList [linsert $fileList 0 . ..]
                set toinsertA ""
                set toinsertB ""
                foreach filename $fileList {
                        set filename [file tail $filename]
                        if [file isdirectory $dirname/$filename] {
                                lappend toinsertA $filename/
                        } else {
                            if {!$fileselect_priv(dironly,$w) && [$filter $dirname/$filename]} {
                                set matches 0
                                foreach p $pattern {
                                    if [string match $p $filename] {
                                        set matches 1
                                        break
                                    }
                                }
                                if {$matches} {
                                        lappend toinsertB $filename
                                }
                            }
                        }
                }
                if {$toinsertA!=""} {
                        eval [list $w.l.list insert end] $toinsertA
                }
                if {$toinsertB!=""} {
                        eval [list $w.l.list insert end] $toinsertB
                }
        }
        incr busy -1
#       modalDialog.end $w.info
}               

# Procedure:    fileselect:filter
#
# Synopsis:
#       Default filter for a file selection dialog.
#
# Usage:
#c      fileselect:filter fileName
#
# Parameters:
#c      fileName
#               Path name of a file
#
# Return value:
#       0 if the file should be suppressed, 1 if it should be displayed.
#
# Description:
#       fileselect:filter is the default filter function in a file selection
#       dialog box if the user doesn't supply one.  It returns 1 if the
#       supplied file exists, and 0 otherwise.

proc fileselect:filter {filename} {
        return [file exists $filename]
}

# Procedure:    fileselect:ok
#
# Synopsis:
#       Handle the `OK' button in a file selection dialog.
#
# Usage:
#c      fileselect:ok pathName
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box.
#
# Return value:
#       None
#
# Description:
#       `fileselect:ok' is invoked when the user presses `OK' or double
#       clicks in a file selection dialog box.  It retrieves the file name
#       from the listbox or entry as appropriate, makes sure that a file
#       has been specified, and calls `fileselect:select' to select it.

proc fileselect:ok w {
        global fileselect_priv
        if $fileselect_priv(dironly,$w) {
            set f [$w.f.entry get]
            if {$f == ""} {
                set s [$w.l.list curselection]
                if {[llength $s] == 0} {
                        set f ""
                } elseif {[llength $s] > 1} {
                        errormessage $w.error "Please select only one file."
                        return
                } else {
                    set f [$w.l.list get $s]
                }
            }
            set f $fileselect_priv(dirname,$w)$f/
            set command $fileselect_priv(command,$w)
            if {$fileselect_priv(textvariable,$w) != ""} {
                upvar #0 $fileselect_priv(textvariable,$w) tv
                set tv $f
            }
            if {$command != ""} {
                uplevel #0 $command $f
            }
        } else {
            set f [$w.f.entry get]
            if {$f == ""} {
                set s [$w.l.list curselection]
                if {[llength $s] == 0} {
                        errormessage $w.error "Please select a file name"
                        return
                } elseif {[llength $s] > 1} {
                        errormessage $w.error "Please select only one file."
                        return
                } else {
                    set f [$w.l.list get $s]
                }
            }
            fileselect:select $w $f
        }
}

#used only with -dironly on. DRS.

proc fileselect:open_dir { w } {
        global fileselect_priv
            set f [$w.f.entry get]
            if {$f == ""} {
                set s [$w.l.list curselection]
                if {[llength $s] == 0} {
                        errormessage $w.error "Please select a file name"
                        return
                } elseif {[llength $s] > 1} {
                        errormessage $w.error "Please select only one file."
                        return
                } else {
                    set f [$w.l.list get $s]
                }
            }
            fileselect:select $w $f
}


# Procedure:    fileselect:select
#
# Synopsis:
#       Select a file or directory in a file selection dialog.
#
# Usage:
#c      fileselect:select pathName fileName
#
# Parameters:
#c      pathName
#               Path name of a file selection dialog box
#c      fileName
#               Name of a selected file or directory.
#
# Description:
#       `fileselect:select' is the general function that responds to the
#       `OK' button or to a double click in a file selection dialog box.
#       It is passed the file name being selected.  If the file is a
#       directory, it is scanned and opened.  If it is a plain file,
#       it is selected, the text variable is set, and the selection command
#       is executed.

proc fileselect:select {w filename} {
        global fileselect_priv
        set command $fileselect_priv(command,$w)
        set filter $fileselect_priv(filter,$w)
        set dir $fileselect_priv(dirname,$w)
        if {![regexp {^[/~]} $filename]} {
                set filename ${dir}$filename
        }
        if {!$fileselect_priv(new,$w) && ![file exists $filename]} {
                errormessage $w.error "Can't open ``$filename''."
        } else {
                catch {$w.f.entry delete 0 end}
                if [file isdirectory $filename] {
                        fileselect:dir $w $filename/
                } else {
                        if {$fileselect_priv(textvariable,$w) != ""} {
                                upvar #0 $fileselect_priv(textvariable,$w) tv
                                set tv $filename
                        }
                        if {$command != ""} {
                                uplevel #0 $command $filename
                        }
                }
        }
}

# Procedure:    fileselect_slash
#
# Synopsis:
#       Strip a trailing slash from a directory name
#
# Usage:
#c      fileselect_slash fileName
#
# Parameters:
#c      fileName
#               Name of a file or directory, optionally with a trailing slash.
#
# Return value:
#       File name, with the trailing slash removed.
#
# Description:
#       fileselect_slash strips trailing slashes from file names.

proc fileselect_slash f {
        if [regexp {^(.*)/$} $f rubbish leader] {
                set f $leader
        }
        return $f
}

# Procedure: selectfile
#
# Synopsis:
#       Modal file selection dialog.
#
# Usage:
#c      selectFile ?argument...?
#
# Parameters:
#       Parameters are the same as for fileselect, except that the
#       widget path name should NOT be supplied, and the `-command' and
#       `-textvariable' options should not be used.
#
# Return value:
#       Name of the selected file.
#
# Errors:
#c      operation cancelled
#               Self-explanatory.
#
# Description:
#       selectfile provides a simple interface to fileselect, performing the
#       selection in a transient modal dialog.
#
# See also:
#c      fileselect

proc selectfile args {
        global selectfile_priv
        set w [eval modalDialog transient fileselect .fileselect $args \
                        -textvariable selectfile_priv]
        widget_waitVariable selectfile_priv
        set selection $selectfile_priv
        unset selectfile_priv
        modalDialog.end $w
        return $selection
}



