#--------------------------------------------------------------------------
#                  Copyright (c) Donald Syme 1992                          
#                  All rights reserved                                     
#                                                                          
# Donald Syme, hereafter referred to as `the Author', retains the copyright
# and all other legal rights to the Software contained in this file,       
# hereafter referred to as `the Software'.                                 
#                                                                          
# The Software is made available free of charge on an `as is' basis. No    
# guarantee, either express or implied, of maintenance, reliability,       
# merchantability or suitability for any purpose is made by the Author.    
#                                                                          
# The user is granted the right to make personal or internal use of the    
# Software provided that both:                                             
# 1. The Software is not used for commercial gain.                         
# 2. The user shall not hold the Author liable for any consequences        
#    arising from use of the Software.                                     
#                                                                          
# The user is granted the right to further distribute the Software         
# provided that both:                                                      
# 1. The Software and this statement of rights are not modified.           
# 2. The Software does not form part or the whole of a system distributed  
#    for commercial gain.                                                  
#                                                                          
# The user is granted the right to modify the Software for personal or     
# internal use provided that all of the following conditions are observed: 
# 1. The user does not distribute the modified software.                   
# 2. The modified software is not used for commercial gain.                
# 3. The Author retains all rights to the modified software.               
#                                                                          
# Anyone seeking a licence to use this software for commercial purposes is 
# invited to contact the Author.                                           
#--------------------------------------------------------------------------





# File: collapsible.tcl
#
# Description:
#       Procedure to define a `collapsible' object.
#
# Global variables:
#c      collapsible_priv(visible,$w)
#               1 if the collapsible widget $w is visible, and 0 otherwise.


# Procedure: collapsible
#
# Synopsis:
#       Create a `collapsible' widget which may be added to or removed from
#       a larger display under user control.
#
# Usage:
#c      collapsible widgetType pathName ?option value?...
#
# Parameters:
#c      widgetType
#               A widget creation command such as `button' or `listbox'.
#
#c      pathName
#               Name of the frame in which the collapsible widget will be
#               created.
#
# Options:
#       Name:                   title
#       Class:                  Title
#       Command-Line Switch:    -title
#               Title of the collapsible widget.  Default is the null string.
#
#       Name:                   visible
#       Class:                  Visible
#       Command-Line Switch:    -visible
#               A Boolean value (0 or 1) giving the default state of the
#               visibility of the widget.  Default is 0.
#
#
#       Name:                   collapseCommand
#       Class:                  CollapseCommand
#       Command-Line Switch:    -collapseCommand
#               A command to execute each time the window is
#               collapsed/uncollapsed.  A value "0" or "1" will be
#               appended as an additional argument to the command.
#               Default is none.
#
#       Other options are those accepted by the `widgetType' command.
#
# Description:
#       A `collapsible' widget is one that is packed into its parent's frame
#       only on demand.  It appears on the screen as an arrow (pointing right
#       or downward) at upper left, a title at upper center, and the widget 
#       itself at lower right.  The arrow is a button. Invoking the button
#       toggles the state of the widget between being visible and invisible
#       by packing it or unpacking it.
#
# Bugs:
#       - If a widget becomes invisible while it or one of its components has
#       the keyboard focus, it doesn't relinquish the focus.  It ought to.
#
#       - The collabsible processor does not honor the `configure' widget
#       command.
#
#       - The collapsible processor does not use the widget creation
#       definitions to make itself a first-class widget.

option add *Collapsible.visible 0 widgetDefault
option add *Collapsible.title {} widgetDefault
option add *Collapsible.collapseCommand {} widgetDefault
option add *Collapsible.a.relief flat widgetDefault

proc collapsible {type w args} {
        global collapsible_priv

        global gui_flags

        frame $w -class Collapsible

        set title [option get $w title Title]
        set visible [option get $w visible Visible]
        set collapseCommand [option get $w collapseCommand CollapseCommand]
        set fargs {}

        while {[llength $args] >= 2} {
                set option [lindex $args 0]
                set value [lindex $args 1]
                set args [lrange $args 2 end]
                case $option in {
                        -title { set title $value }
                        -visible { set visible $value }
                        -collapseCommand { set collapseCommand $value }
                        default {
                                lappend fargs $option $value
                        }
                }
        }

        set collapsible_priv(visible,$w) $visible
        set collapsible_priv(collapseCommand,$w) $collapseCommand

        pack append $w \
                [button $w.a \
                        -command "collapsible_toggle $w" \
                        -borderwidth 0 \
                        -bitmap [icon_find triangler]] \
                                {left frame n} \
                [fontcheck label $w.t -text $title -font $gui_flags(font,labels)] \
                                {top frame w}

        eval [list $type $w.b] $fargs   

        widget_addBinding $w.a GainFocus "collapsible:focusIn.a $w.a"
        widget_addBinding $w.a LoseFocus "collapsible:focusOut.a $w.a"

        if {$visible} {
                collapsible_show $w
        }
        return $w
}

# Procedure:    collapsible_toggle
#
# Synopsis:
#       Toggle the state of a collapsible widget when a user requests the
#       change.
#
# Usage:
#c      collapsible_toggle pathName
#
# Parameters:
#c      pathName
#               Path name of a collapsible widget
#
# Return value:
#       None.
#
# Description:
#       collapsible_toggle toggles the state of a collapsible widget.
#       If it is invisible, it makes it visible, and vice versa.

proc collapsible_toggle w {
        global collapsible_priv
        if {$collapsible_priv(visible,$w)} {
                collapsible_hide $w
        } else {
                collapsible_show $w
        }
}

# Procedure:    collapsible_hide
#
# Synopsis:
#       Make a collapsible widget invisible.
#
# Usage:
#c      collapsible_hide pathName
#
# Parameters:
#c      pathName
#               Path name of a collapsible widget
#
# Return value:
#       None.
#
# Description:
#       collapsible_hide makes a collapsible widget invisible.

proc collapsible_hide w {
        global collapsible_priv
        set collapsible_priv(visible,$w) 0
        if {[string compare $w.a [focus]] == 0} {
                $w.a config -bitmap [icon_find filltriangler]
        } else {
                $w.a config -bitmap [icon_find triangler]
        }
        if {$collapsible_priv(collapseCommand,$w) != ""} {
            eval $collapsible_priv(collapseCommand,$w) 0
        }
        pack unpack $w.b
        # NOTE: Need to defocus any descendant of $w.b!
        
        
}

# Procedure: collapsible_show
#
# Synopsis:
#       Make a collapsible widget visible
#
# Usage:
#c      collapsible_show pathName
#
# Parameters:
#c      pathName
#               Path name of a collapsible widget.
#
# Return Value:
#       None.
#
# Description:
#       collapsible_show makes a collapsible widget visible.

proc collapsible_show w {
        global collapsible_priv
        set collapsible_priv(visible,$w) 1
        if {[string compare $w.a [focus]] == 0} {
                $w.a config -bitmap [icon_find filltriangled]
        } else {
                $w.a config -bitmap [icon_find triangled]
        }
        if {$collapsible_priv(collapseCommand,$w) != ""} {
            eval $collapsible_priv(collapseCommand,$w) 1
        }
        pack append $w $w.b {top expand fill}
}

# Procedure: collapsible:focusIn.a
#
# Synopsis:
#       Internal procedure that handles the <FocusIn> event on a collapsible
#       widget's control button.
#
# Usage:
#c      collapsible:focusIn.a pathName
#
# Parameters:
#c      pathName
#               Path name of the arrow button in a collapsible widget.
#
# Return value:
#       None.
#
# Description:
#       collapsible:focusIn.a is called when the arrow button in a collapsible
#       widget receives the keyboard focus.  It changes the butmap to a filled
#       one, to indicate the focus.

proc collapsible:focusIn.a w {
        set b [lindex [$w config -bitmap] 4]
        $w config -bitmap [file dirname $b]/fill[file tail $b]
}

# Procedure: collapsible:focusOut.a
#
# Synopsis:
#       Internal procedure that handles the <FocusOut> event on a collapsible
#       widget's control button.
#
# Usage:
#c      collapsible:focusOut.a pathName
#
# Parameters:
#c      pathName
#               Path name of the arrow button in a collapsible widget.
#
# Return value:
#       None.
#
# Description:
#       collapsible:focusIn.a is called when the arrow button in a collapsible
#       widget loses the keyboard focus.  It changes the butmap to an outline
#       one, to indicate the lost focus.

proc collapsible:focusOut.a w {
        set b [lindex [$w config -bitmap] 4]
        $w config -bitmap \
                [file dirname $b]/[string range [file tail $b] 4 end]
}

