(****************************************************************************)
(*                                                                          *)
(*               Copyright 1993, 1994 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : ppout.sml                                                *)
(* DESCRIPTION   : Pretty-printing to output streams.                       *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 7th December 1993                                        *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 5th December 1994                                        *)
(****************************************************************************)

signature LABEL_PPOUT =
sig
   type 'a ppbox
   val display_ppbox : 'a ppbox -> unit
   val output_ppbox  : string -> 'a ppbox -> unit
   type 'a text
   val display_text  : 'a text -> unit
   val output_text   : string -> 'a text -> unit
end;

functor LabelPPOutFun
   (LabelPPBoxes :
      sig
         structure PPStream : PPSTREAM
         type 'a ppbox
         val write_ppbox : 'a ppbox -> PPStream.ppstream -> unit
         type 'a text
         val write_text : 'a text -> PPStream.ppstream -> unit
      end) : LABEL_PPOUT =
struct

structure PPStream = LabelPPBoxes.PPStream;

type 'a ppbox = 'a LabelPPBoxes.ppbox
and 'a text = 'a LabelPPBoxes.text;

(*--------------------------------------------------------------------------*)
(* Wrappers for top-level pretty-printing.                                  *)
(* A final new line is added and the stream is flushed.                     *)
(*--------------------------------------------------------------------------*)

local

fun wrap ppf pps =
   let val {consumer,flush,...} = PPStream.dest_ppstream pps
   in  (ppf pps;
        PPStream.flush_ppstream pps;
        consumer "\n";
        flush ())
   end;

in

val wrap_ppbox = wrap o LabelPPBoxes.write_ppbox
and wrap_text = wrap o LabelPPBoxes.write_text;

end;

(*--------------------------------------------------------------------------*)
(* Widths of the display and files in number of characters.                 *)
(*--------------------------------------------------------------------------*)

val termwidth = 78
and filewidth = 78;

(*--------------------------------------------------------------------------*)
(* Function to make a pretty-printer stream from an output stream.          *)
(*--------------------------------------------------------------------------*)

fun out_pps width str =
   PPStream.mk_ppstream
      {consumer = (fn s => output (str,s)),
       flush = (fn () => ()),
       linewidth = width};

(*--------------------------------------------------------------------------*)
(* A pretty-printer stream for standard output.                             *)
(*--------------------------------------------------------------------------*)

val std_pps = out_pps termwidth std_out;

(*--------------------------------------------------------------------------*)
(* Pretty-print a ppbox/text to standard output.                            *)
(*--------------------------------------------------------------------------*)

fun display_ppbox box = wrap_ppbox box std_pps
and display_text text = wrap_text text std_pps;

(*--------------------------------------------------------------------------*)
(* Pretty-print a ppbox/text to a file.                                     *)
(*--------------------------------------------------------------------------*)

local

fun output filename ppf =
   let val str = open_out filename
       val pps = out_pps filewidth str
   in  (ppf pps; close_out str)
   end;

in

fun output_ppbox filename = output filename o wrap_ppbox
and output_text filename = output filename o wrap_text;

end;

end; (* LabelPPOutFun *)


signature PPOUT =
sig
   type ppbox
   val display_ppbox : ppbox -> unit
   val output_ppbox  : string -> ppbox -> unit
end;

functor PPOutFun
   (PPBoxes :
      sig
         structure PPStream : PPSTREAM
         type ppbox
         val write_ppbox : ppbox -> PPStream.ppstream -> unit
      end) : PPOUT =
let structure LabelPPOut =
       LabelPPOutFun
          (struct
              structure PPStream = PPBoxes.PPStream;
              type 'a ppbox = PPBoxes.ppbox;
              val write_ppbox = PPBoxes.write_ppbox;
              type 'a text = PPBoxes.ppbox;
              val write_text = PPBoxes.write_ppbox;
           end)
in  struct
       type ppbox = PPBoxes.ppbox
       val display_ppbox = LabelPPOut.display_ppbox
       val output_ppbox = LabelPPOut.output_ppbox
    end
end;
