%------------------------------------------------------------------------

  FILE		: mk_CPO.ml                                                

  A theory for upper bounds, lower bounds, complete partial orders, 
  and fixed points.	
 
  Defines orderings on types instead of sets by using type variables.   
  For example, the standard notation for representing a partial order   
  is a pair (D,<=), where D denotes some set of values, and <= is some  
  binary relation defined over elements in D. In this theory we do not	
  refer to D explicitly, but refer to it implicitly in the type of <=,	
  e.g. for <= : *->*->bool, D is implicitly represented as :*.		
 
  The final result in this theory is the proof of the Traski fixed point 
  theorem for continuos function which allows the definition of 
  recursive operators.           

  LOADS LIBRARY	: sets, tauts			       		        
  READS FILES	: NONE				          		
  WRITES FILES  : CPO.th					
 									
  AUTHOR	: Albert J Camilleri					
  AFFILIATION   : Hewlett-Packard Laboratories, Bristol			
  DATE		: 90.04.01						
------------------------------------------------------------------------%

%------------------------------------------------------------------------
  Re-written by Wishnu Prasetya on May 94.

  Let in the sequel r be a relation on type A, s a relation on type B, 
  f a function from A to B, and X a set on type A.
  Concepts/definitions supported:


Partial Orders:

  REFL r       : r is a reflexive 
  ANTISYM r    : r is anti-symmetric
  TRANS r      : r is transitive
  PO r         : r is a partial order 


Upper & lower bounds:

  isUB  r X a  : a is an upper bound of set X wrt ordering r
  isLB  r X a  : a is an lower bound of set X wrt ordering r
  isCUP r X a  : a is the least upper bound of set X wrt r
  isCAP r X a  : a is the greatest lower bound of set X wrt r
  CUP r X      : the least upper bound of X wrt r, if exists
  CAP r X      : the greatest lower bound of X wrt r, if exists
  Bot r        : the least element in type A according to 
  Top r        : the greatest element in type A according to r

Complete Partial Orders:

  Directed r X : X is a `directed` set wrt r
  CPO r        : r is a complete partial order
 
Monotonic, Continous functions & fix points:

  Mono r s f   : f is a monotonic function wrt r and s
  Cont r s f   : f is a continous function wrt r and s
  isLFP r f a  : a is the least fix point of f wrt r 
  LFP r f      : the least fix point of f wrt r, if exists
  ITER n f     : n times application of f

Main results supported by this theory:

  A continous function is monotonic.
  {f^n Bot | 0<=n} is the least fix point of f.


May 1994
Wishnu Prasetya
Dept. of Comp. Science, Utrecht University, the Netherlands
wishnu@cs.ruu.nl

------------------------------------------------------------------------%


new_theory `CPO`;;
load_library `sets`;;
load_library `taut` ;;

let XRULE_ASSUM rule = RULE_ASSUM_TAC (\thm. rule thm ? thm) ;;

letrec UNDISCH_ALL_TAC (asml,g) =
   if (asml=[]) then ALL_TAC (asml,g)
   else (UNDISCH_TAC (hd asml) THEN UNDISCH_ALL_TAC) (asml,g) ;;

%------------------------------------------------------------------------
  Definition of Reflexive, transitive and anti-symmetric
------------------------------------------------------------------------%

let REFL = new_definition 
  (`REFL`, "REFL (r:*->*->bool) = ! x:*. r x x");;

let TRANS = new_definition
  (`TRANS`,
   "TRANS (r:*->*->bool) = ! x y z:*. ((r x y) /\ (r y z)) ==> (r x z)");;

let ANTISYM = new_definition
  (`ANTISYM`,
   "ANTISYM (r:*->*->bool) = ! x y:*. ((r x y) /\ (r y x)) ==> (x = y)");;

%------------------------------------------------------------------------
  Here are extension theories for relations.  can be used to prove 
  equality of two elements in the domain.
------------------------------------------------------------------------%

let Low_EXTENSION = prove_thm
  (`Low_EXTENSION`,
   "!r x (y:*). REFL r /\ ANTISYM r ==> 
                ((x=y) = (!z. r z x = r z y))",
    REWRITE_TAC [REFL; ANTISYM]
    THEN REPEAT STRIP_TAC THEN EQ_TAC THEN REPEAT STRIP_TAC
    THENL [ ASM_REWRITE_TAC[] ; ALL_TAC ]
    THEN FIRST_ASSUM MATCH_MP_TAC THEN CONJ_TAC
    THENL
    [ XRULE_ASSUM (SPEC "x:*") THEN RES_TAC ;
      XRULE_ASSUM (SPEC "y:*") THEN RES_TAC ]) ;;

let High_EXTENSION = prove_thm
  (`High_EXTENSION`,
   "!r x (y:*). REFL r /\ ANTISYM r ==> 
                ((x=y) = (!z. r x z = r y z))",
    REWRITE_TAC [REFL; ANTISYM]
    THEN REPEAT STRIP_TAC THEN EQ_TAC THEN REPEAT STRIP_TAC
    THENL [ ASM_REWRITE_TAC[] ; ALL_TAC ]
    THEN FIRST_ASSUM MATCH_MP_TAC THEN CONJ_TAC
    THENL
    [ XRULE_ASSUM (SPEC "y:*") THEN RES_TAC ;
      XRULE_ASSUM (SPEC "x:*") THEN RES_TAC ]) ;;

%------------------------------------------------------------------------
   A partial order is a reflexive, transitive, and anti-symmetric
   relation.
------------------------------------------------------------------------%

let PO = new_definition
  (`PO`, "PO (r:*->*->bool) = (REFL r) /\ (TRANS r) /\ (ANTISYM r)");;

%------------------------------------------------------------------------
   Here is the definition of upper bound and lower bound.
------------------------------------------------------------------------%

let IS_UB =
    new_definition
       (`IS_UB`,
	"IS_UB r X (b:*) = (!a:*. (a IN X) ==> r a b)");;

let IS_LB =
    new_definition
       (`IS_LB`,
	"IS_LB r X (a:*) = (!b:*. (b IN X) ==> r a b)");;

%------------------------------------------------------------------------
  Here is the definition of least upper bound (CUP) and greatest 
  lower bound (CAP).
-------------------------------------------------------------------------%

let isCUP = new_definition
  (`isCUP`,
   "isCUP r X b =  (IS_UB r X b) /\ (!c:*. (IS_UB r X c) ==> r b c)");;

let isCAP = new_definition
  (`isCAP`,
   "isCAP r X c =  (IS_LB r X c) /\ (!b:*. (IS_LB r X b) ==> r b c)");;

%------------------------------------------------------------------------
  Define CUP r X as the least upper bound of X wrt r, if one exists,
  and CAP r X as the greatest lower bound of X wrt r, if one exists.
------------------------------------------------------------------------%

let lemma = prove
  ("?CUP. !(r:*->*->bool) X.
      (?a. isCUP r X a) ==> isCUP r X (CUP r X)",
    EXISTS_TAC "\(r:*->*->bool) X.
                  (?a. isCUP r X a) => (@a:*. isCUP r X a) | a"
    THEN REPEAT STRIP_TAC THEN BETA_TAC
    THEN COND_CASES_TAC
    THENL 
    [ CONV_TAC SELECT_CONV
      THEN EXISTS_TAC "a':*" THEN ASM_REWRITE_TAC[] ;
      UNDISCH_ALL_TAC
      THEN (CONV_TAC o DEPTH_CONV) NOT_EXISTS_CONV
      THEN REPEAT STRIP_TAC THEN RES_TAC] ) ;;

let CUP = new_specification `CUP` [`constant`,`CUP`] lemma ;;

let lemma = prove
  ("?CAP. !(r:*->*->bool) X.
      (?a. isCAP r X a) ==> isCAP r X (CAP r X)",
    EXISTS_TAC "\(r:*->*->bool) X.
                  (?a. isCAP r X a) => (@a:*. isCAP r X a) | a"
    THEN REPEAT STRIP_TAC THEN BETA_TAC
    THEN COND_CASES_TAC
    THENL 
    [ CONV_TAC SELECT_CONV
      THEN EXISTS_TAC "a':*" THEN ASM_REWRITE_TAC[] ;
      UNDISCH_ALL_TAC
      THEN (CONV_TAC o DEPTH_CONV) NOT_EXISTS_CONV
      THEN REPEAT STRIP_TAC THEN RES_TAC] ) ;;

let CAP = new_specification `CAP` [`constant`,`CAP`] lemma ;;

%------------------------------------------------------------------------
  Define Cup r a b as the CUP of {a,b} and Cap a b as the CAP of {a,b}.
  Define Bot r and Top r as the least upper bound and greatest lower
  bound of {}.
------------------------------------------------------------------------%

let Cup = new_definition
  (`Cup`, "Cup (r:*->*->bool) a b = CUP r {a,b}") ;;

let Cap = new_definition
  (`Cap`, "Cap (r:*->*->bool) a b = CAP r {a,b}") ;;

let Bot = new_definition
  (`Bot`, "Bot (r:*->*->bool) = CUP r {}") ;;

let Top = new_definition
  (`Top`, "Top (r:*->*->bool) = CAP r {}") ;;

%------------------------------------------------------------------------
   If exist, bottom is less than any x in the domain and top is greater
   than any x in the domain.
------------------------------------------------------------------------%

let Bot_THM = prove_thm
  (`Bot_THM`,
   "!a:*. isCUP r {} a ==> (!x. r (Bot r) x)",
    REWRITE_TAC [Bot] THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC CUP
    THEN UNDISCH_ALL_TAC
    THEN REWRITE_TAC [isCUP; IS_UB]
    THEN (CONV_TAC o DEPTH_CONV o IN_CONV) NO_CONV
    THEN REPEAT STRIP_TAC
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN REWRITE_TAC[]) ;;
  
let Top_THM = prove_thm
  (`Top_THM`,
   "!a:*. isCAP r {} a ==> (!x. r x (Top r))",
    REWRITE_TAC [Top] THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC CAP
    THEN UNDISCH_ALL_TAC
    THEN REWRITE_TAC [isCAP; IS_LB]
    THEN (CONV_TAC o DEPTH_CONV o IN_CONV) NO_CONV
    THEN REPEAT STRIP_TAC
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN REWRITE_TAC[]) ;;

%------------------------------------------------------------------------
   If r is anti-symmetric, then least upper bound and greatest lower
   bound are unique, if they exist.
------------------------------------------------------------------------%

let UNIQUE_CUP = prove_thm
  (`UNIQUE_CUP`,
   "!(r:*->*->bool) X x y. 
        ANTISYM r /\ isCUP r X x /\ isCUP r X y ==> (y = x)",
    REWRITE_TAC [isCUP; IS_UB; ANTISYM] 
    THEN REPEAT STRIP_TAC 
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN RES_TAC THEN ASM_REWRITE_TAC[]) ;;

let UNIQUE_CAP = prove_thm
  (`UNIQUE_CAP`,
   "!(r:*->*->bool) X x y. 
        ANTISYM r /\ isCAP r X x /\ isCAP r X y ==> (y = x)",
    REWRITE_TAC [isCAP; IS_LB; ANTISYM] 
    THEN REPEAT STRIP_TAC 
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN RES_TAC THEN ASM_REWRITE_TAC[]) ;;


let isCUP_CUP = prove_thm
  (`isCUP_CUP`,
   "!r X (a:*). ANTISYM r /\ isCUP r X a ==> (CUP r X = a)",
    REWRITE_TAC [CUP] THEN REPEAT STRIP_TAC
    THEN MATCH_MP_TAC UNIQUE_CUP
    THEN EXISTS_TAC "r:*->*->bool" 
    THEN EXISTS_TAC "X:(*)set"
    THEN IMP_RES_TAC CUP
    THEN ASM_REWRITE_TAC[]) ;;

let isCAP_CAP = prove_thm
  (`isCAP_CAP`,
   "!r X (a:*). ANTISYM r /\ isCAP r X a ==> (CAP r X = a)",
    REWRITE_TAC [CAP] THEN REPEAT STRIP_TAC
    THEN MATCH_MP_TAC UNIQUE_CAP
    THEN EXISTS_TAC "r:*->*->bool" 
    THEN EXISTS_TAC "X:(*)set"
    THEN IMP_RES_TAC CAP
    THEN ASM_REWRITE_TAC[]) ;;

%------------------------------------------------------------------------
  The following lemmas follows straight forward from the definition. 
  They may be handy in calculation.
  The least upper bound of X is greater than any element of X and less
  than any upper bound of X. Analog for greatest lower bound.
------------------------------------------------------------------------%

let CUP_greater = prove_thm
  (`CUP_greater`,
   "!r X (a:*) (x:*). isCUP r X a /\ x IN X ==> r x (CUP r X)",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC CUP 
    THEN UNDISCH_TAC "isCUP (r:*->*->bool) X (CUP r X)"
    THEN REWRITE_TAC [isCUP; IS_UB] THEN REPEAT STRIP_TAC
    THEN RES_TAC) ;;

let CUP_less = prove_thm
  (`CUP_less`,
   "!r X (a:*) (x:*). 
          PO r /\ isCUP r X a ==> (IS_UB r X x = r (CUP r X) x)",
    REWRITE_TAC [PO; TRANS] 
    THEN REPEAT STRIP_TAC THEN EQ_TAC THEN STRIP_TAC
    THENL
    [ IMP_RES_TAC CUP 
      THEN UNDISCH_TAC "isCUP (r:*->*->bool) X (CUP r X)"
      THEN REWRITE_TAC [isCUP] THEN REPEAT STRIP_TAC
      THEN RES_TAC ;
      IMP_RES_TAC CUP 
      THEN UNDISCH_TAC "isCUP (r:*->*->bool) X (CUP r X)"
      THEN REWRITE_TAC [isCUP; IS_UB] THEN REPEAT STRIP_TAC
      THEN RES_TAC ]) ;;

let CAP_less = prove_thm
  (`CAP_less`,
   "!r X (a:*) (x:*). isCAP r X a /\ x IN X ==> r (CAP r X) x",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC CAP 
    THEN UNDISCH_TAC "isCAP (r:*->*->bool) X (CAP r X)"
    THEN REWRITE_TAC [isCAP; IS_LB] THEN REPEAT STRIP_TAC
    THEN RES_TAC) ;;

let CAP_greater = prove_thm
  (`CAP_greater`,
   "!r X (a:*) (x:*). 
       PO r /\ isCAP r X a ==> (IS_LB r X x = r x (CAP r X))",
    REWRITE_TAC [PO; TRANS] 
    THEN REPEAT STRIP_TAC THEN EQ_TAC THEN STRIP_TAC
    THENL
    [ IMP_RES_TAC CAP 
      THEN UNDISCH_TAC "isCAP (r:*->*->bool) X (CAP r X)"
      THEN REWRITE_TAC [isCAP] THEN REPEAT STRIP_TAC
      THEN RES_TAC ;
      IMP_RES_TAC CAP 
      THEN UNDISCH_TAC "isCAP (r:*->*->bool) X (CAP r X)"
      THEN REWRITE_TAC [isCAP; IS_LB] THEN REPEAT STRIP_TAC
      THEN RES_TAC ]) ;;

%------------------------------------------------------------------------
  For any reflexive relation, saying that some element a is ordered below	
  some element b, is equivalent to saying that b is the least upper	
  bound of the set {a, b} or a is the greatest lower bound of {a,b}.
------------------------------------------------------------------------%

let pair_lemma = prove
  ("!x a (b:*). (x IN {a,b}) = ((x=a)\/(x=b))",
    REWRITE_TAC [INSERT_DEF]
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN (CONV_TAC o DEPTH_CONV o IN_CONV) NO_CONV
    THEN REWRITE_TAC[]) ;;
    
let isCUP_ORDER = prove_thm
  (`isCUP_ORDER`,
   "!r a (b:*). (REFL r) ==> (r a b = isCUP r {a,b} b)",
    REWRITE_TAC [REFL; isCUP; IS_UB; pair_lemma]
    THEN REPEAT STRIP_TAC THEN EQ_TAC 
    THEN REPEAT STRIP_TAC
    THENL 
    [ ASM_REWRITE_TAC[]; ASM_REWRITE_TAC[];
      FIRST_ASSUM MATCH_MP_TAC THEN REWRITE_TAC[];
      FIRST_ASSUM MATCH_MP_TAC THEN REWRITE_TAC[] ]) ;;

let isCAP_ORDER = prove_thm
  (`isCAP_ORDER`,
   "!r a (b:*). (REFL r) ==> (r a b = isCAP r {a,b} a)",
    REWRITE_TAC [REFL; isCAP; IS_LB; pair_lemma]
    THEN REPEAT STRIP_TAC THEN EQ_TAC 
    THEN REPEAT STRIP_TAC
    THENL 
    [ ASM_REWRITE_TAC[]; ASM_REWRITE_TAC[];
      FIRST_ASSUM MATCH_MP_TAC THEN REWRITE_TAC[];
      FIRST_ASSUM MATCH_MP_TAC THEN REWRITE_TAC[] ]) ;;

%------------------------------------------------------------------------
  For any reflexive relation, the CUP and CAP of {x} is x.
------------------------------------------------------------------------%

let lemma = prove
  ("{x:*} = {x,x}",
    REWRITE_TAC [EXTENSION; pair_lemma; INSERT_DEF; NOT_IN_EMPTY]
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN REWRITE_TAC[]) ;;

let CUP_SINGLETON = prove_thm
  (`CUP_SINGLETON`,
   "!r (a:*). REFL r ==> isCUP r {a} a",
     ONCE_REWRITE_TAC [lemma] THEN REPEAT STRIP_TAC
     THEN SUBGOAL_THEN "(r:*->*->bool) a a" ASSUME_TAC
     THENL [ XRULE_ASSUM (REWRITE_RULE [REFL]) 
             THEN ASM_REWRITE_TAC[] ; IMP_RES_TAC isCUP_ORDER ]) ;;

let CAP_SINGLETON = prove_thm
  (`CAP_SINGLETON`,
   "!r (a:*). REFL r ==> isCAP r {a} a",
     ONCE_REWRITE_TAC [lemma] THEN REPEAT STRIP_TAC
     THEN SUBGOAL_THEN "(r:*->*->bool) a a" ASSUME_TAC
     THENL [ XRULE_ASSUM (REWRITE_RULE [REFL]) 
             THEN ASM_REWRITE_TAC[] ; IMP_RES_TAC isCAP_ORDER ]) ;;

% This lemma just proves that |- isCUP b X r ==> (CUP X r = b)		%
% Comes in handy later.							%

let CUP_EQ = prove
  ("!(r:*->*->bool) X a. 
        ANTISYM r /\ isCUP r X a ==> (CUP r X = a)",
    REPEAT STRIP_TAC 
    THEN SUBGOAL_THEN "isCUP (r:*->*->bool) X (CUP r X)" ASSUME_TAC
    THENL [ MATCH_MP_TAC CUP
            THEN EXISTS_TAC "a:*" THEN ASM_REWRITE_TAC[] ;
            ALL_TAC ]
    THEN IMP_RES_TAC UNIQUE_CUP) ;;

%------------------------------------------------------------------------
  For any partial order, a set X is directed iff for any two  
  elements in X, there exists another which is ordered above the two.   
  (It follows that an empty set is directed)
------------------------------------------------------------------------%

let Directed = new_definition
  (`Directed`,
   "Directed r (X:(*)set) =
	  !a b. a IN X /\ b IN X ==> (?c. isCUP r {a,b} c)") ;;

let EMPTY_Directed = prove_thm
  (`EMPTY_Directed`,
   "!r:*->*->bool. Directed r {}",
    REWRITE_TAC [Directed; NOT_IN_EMPTY]) ;;

%------------------------------------------------------------------------
  A relation is a complete partial order iff:				
    it is a partial order, and						
    there exists a bottom element, and					
    all sets directed by the ordering has a least upper bound.		 
------------------------------------------------------------------------%

let CPO = new_definition
  (`CPO`,
   "CPO (r:*->*->bool) =
	 ((PO r) /\
	  (? a. isCUP r {} a) /\
	  (!X. (Directed r X) ==> ?a. isCUP r X a))");;

%------------------------------------------------------------------------
  In a CPO, Bot is indeed the least element and CUP X of a directed
  set X is indeed the least upperbound of X.
------------------------------------------------------------------------%

let CPO_Bot = prove_thm
  (`CPO_Bot`,
   "!r (x:*). CPO r ==> r (Bot r) x",
    REWRITE_TAC [CPO] THEN REPEAT STRIP_TAC
    THEN IMP_RES_TAC Bot_THM THEN ASM_REWRITE_TAC[]) ;;

let CPO_CUP = prove_thm
  (`CPO_CUP`,
   "!r (X:(*)set). CPO r /\ Directed r X ==> isCUP r X (CUP r X)",
    REWRITE_TAC [CPO] THEN REPEAT STRIP_TAC
    THEN RES_TAC THEN IMP_RES_TAC CUP) ;;

%------------------------------------------------------------------------
   Here is the definition of the least fix point.
------------------------------------------------------------------------%

let isLFP = new_definition
  (`isLFP`,
   "isLFP r fun (x:*) = (fun x = x) /\ !y. (fun y = y) ==> (r x y)");;


let lemma = prove
  ("?lfp. !r f. (?x:*. isLFP r f x) ==> isLFP r f (lfp r f)",
    EXISTS_TAC "\(r:*->*->bool) f.
                  (?a. isLFP r f a) => (@a:*. isLFP r f a) | a"
    THEN REPEAT STRIP_TAC THEN BETA_TAC
    THEN COND_CASES_TAC
    THENL 
    [ CONV_TAC SELECT_CONV
      THEN EXISTS_TAC "x:*" THEN ASM_REWRITE_TAC[] ;
      UNDISCH_ALL_TAC
      THEN (CONV_TAC o DEPTH_CONV) NOT_EXISTS_CONV
      THEN REPEAT STRIP_TAC THEN RES_TAC] ) ;;

let LFP = new_specification `LFP` [`constant`,`LFP`] lemma ;;

%------------------------------------------------------------------------
   If exists, lfp wrt an anti-symmetric relation is unique.
------------------------------------------------------------------------%

let UNIQUE_LFP = prove_thm
  (`UNIQUE_LFP`,
   "!r f x (y:*). 
       ANTISYM r /\ isLFP r f x /\ isLFP r f y ==> (x = y)",
    REWRITE_TAC [isLFP; ANTISYM] 
    THEN REPEAT STRIP_TAC 
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN RES_TAC
    THEN ASM_REWRITE_TAC[]) ;;

%------------------------------------------------------------------------
  Define ITER f n x as n times application of f on x
------------------------------------------------------------------------%

let ITER = new_prim_rec_definition
  (`ITER`,
   "(ITER 0       f x = (x:*)) /\
    (ITER (SUC n) f x = f(ITER n f x))");;

%------------------------------------------------------------------------
  Here is the definition of monotonic and continous function
------------------------------------------------------------------------%  

let Mono = new_definition
  (`Mono`,
   "Mono r s (f:*->**) = !x y. r x y ==> s (f x) (f y)") ;;

let Cont = new_definition
  (`Cont`,
   "Cont r s (f:*->**) =
      !X. ~(X={}) /\ Directed r X ==>
          Directed s (IMAGE f X) /\
          (f (CUP r X) = CUP s (IMAGE f X))") ;;

%------------------------------------------------------------------------
  The composition of monotonic/continous functions is monotonic/cont.
------------------------------------------------------------------------%

let Mono_o = prove_thm
  (`Mono_o`,
   "!r s t (f:*A->*B) (g:*B->*C).
       Mono r s f /\ Mono s t g ==> Mono r t (g o f)",
    REWRITE_TAC [o_THM; Mono]
    THEN REPEAT STRIP_TAC THEN RES_TAC THEN RES_TAC) ;;

let Cont_o = prove_thm
  (`Cont_o`,
   "!r s t (f:*A->*B) (g:*B->*C).
       Cont r s f /\ Cont s t g ==> Cont r t (g o f)",
    REWRITE_TAC [Cont; o_THM; IMAGE_COMPOSE]
    THEN REPEAT STRIP_TAC THEN RES_TAC THEN RES_TAC
    THENL
    [ XRULE_ASSUM (REWRITE_RULE [IMAGE_EQ_EMPTY])
      THEN RES_TAC ;
      ASM_REWRITE_TAC[] 
      THEN XRULE_ASSUM (REWRITE_RULE [IMAGE_EQ_EMPTY])
      THEN RES_TAC ]) ;;

%------------------------------------------------------------------------
   A continous function is monotonic.
------------------------------------------------------------------------%

let CUP_SING = prove
  ("!r (a:*). REFL r ==> isCUP r {a,a} a",
     REPEAT STRIP_TAC
     THEN SUBGOAL_THEN "(r:*->*->bool) a a" ASSUME_TAC
     THENL [ XRULE_ASSUM (REWRITE_RULE [REFL]) 
             THEN ASM_REWRITE_TAC[] ; IMP_RES_TAC isCUP_ORDER ]) ;;

let lemma1 = prove
  ("!r a (b:*). REFL r /\  r a b ==> Directed r {a,b}",
    REWRITE_TAC [Directed]
    THEN REPEAT STRIP_TAC
    THEN XRULE_ASSUM (REWRITE_RULE [INSERT_DEF])
    THEN XRULE_ASSUM ((CONV_RULE o DEPTH_CONV) SET_SPEC_CONV)
    THEN XRULE_ASSUM (REWRITE_RULE [NOT_IN_EMPTY])
    THEN UNDISCH_ALL_TAC THEN REPEAT STRIP_TAC
    THEN ASM_REWRITE_TAC []
    THENL
    [ EXISTS_TAC "a:*" THEN IMP_RES_TAC CUP_SING
      THEN ASM_REWRITE_TAC[] ;
      EXISTS_TAC "b:*" THEN IMP_RES_TAC isCUP_ORDER ;
      ONCE_REWRITE_TAC [INSERT_COMM]
      THEN EXISTS_TAC "b:*" THEN IMP_RES_TAC isCUP_ORDER ;
      EXISTS_TAC "b:*" THEN IMP_RES_TAC CUP_SING
      THEN ASM_REWRITE_TAC[] ]) ;;

let lemma2 = prove
  ("!r X (a:*). (?b. isCUP r X b) /\ (a=CUP r X) ==> isCUP r X a",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC CUP
    THEN ASM_REWRITE_TAC[]) ;;

let lemma3 = prove
   ("!a (b:*). ~({a,b}={})",
    REWRITE_TAC [SYM (SPEC_ALL MEMBER_NOT_EMPTY); pair_lemma]
    THEN REPEAT GEN_TAC THEN EXISTS_TAC "a:*"
    THEN REWRITE_TAC[]) ;;

let Cont_IMP_Mono = prove_thm
  (`Cont_IMP_Mono`,
   "!r s (f:*->**). 
       CPO r /\ CPO s /\ Cont r s f ==> Mono r s f",
    REWRITE_TAC [CPO; PO; Cont; Mono]
    THEN REPEAT STRIP_TAC
    THEN SUBGOAL_THEN "s ((f:*->**) x) (f y) = isCUP s {f x, f y} (f y)"
         SUBST1_TAC
    THENL [ MATCH_MP_TAC isCUP_ORDER THEN ASM_REWRITE_TAC[] ; ALL_TAC ]
    THEN SUBGOAL_THEN "{(f:*->**) x, f y}= IMAGE f {x,y}"
         SUBST1_TAC
    THENL [ REWRITE_TAC [IMAGE_INSERT; IMAGE_EMPTY] ; ALL_TAC ]
    THEN MATCH_MP_TAC  lemma2
    THEN IMP_RES_TAC lemma1 
    THEN ASSUME_TAC (SPECL ["x:*"; "y:*"] lemma3) 
    THEN RES_TAC THEN CONJ_TAC
    THENL
    [ RES_TAC THEN EXISTS_TAC "a''':**"
      THEN ASM_REWRITE_TAC[] ;
      EVERY_ASSUM (\thm. SUBST1_TAC (SYM thm) ? ALL_TAC)
      THEN SUBGOAL_THEN "CUP r {x,y:*} = y" SUBST1_TAC
      THENL [ ALL_TAC ; REFL_TAC]
      THEN MATCH_MP_TAC isCUP_CUP 
      THEN SUBGOAL_THEN "r x y = isCUP r {x,y} (y:*)"
           (\thm. ASM_REWRITE_TAC [SYM thm])
      THEN IMP_RES_TAC isCUP_ORDER THEN ASM_REWRITE_TAC[] ]) ;;

%-------------------------------------------------------------------------
  Here begins the proof of the KNASTER-TARSKY fixed point theorem.      
  Several intermediate lemmas are proved.				
-------------------------------------------------------------------------%

%-------------------------------------------------------------------------
  lemma0 : for a continous f and cpo <:
           
           m<n ==> f^m Bot < f^n Bot
-------------------------------------------------------------------------%

let lemma0 = prove
  ("!(r:*->*->bool) f.
     CPO r /\ Cont r r f ==> 
     (!n n'. n<n' ==> r (ITER n f (Bot r)) (ITER n' f (Bot r)))",
     REPEAT GEN_TAC THEN STRIP_TAC
     THEN IMP_RES_TAC Cont_IMP_Mono
     THEN UNDISCH_ALL_TAC THEN REWRITE_TAC [CPO; Mono]
     THEN STRIP_TAC THEN STRIP_TAC THEN STRIP_TAC 
     THEN IMP_RES_TAC Bot_THM
     THEN INDUCT_TAC
     THEN ASM_REWRITE_TAC [ITER]
     THEN INDUCT_TAC
     THENL [REWRITE_TAC [NOT_LESS_0]; ASM_REWRITE_TAC [ITER] ]
     THEN DISCH_TAC
     THEN XRULE_ASSUM (REWRITE_RULE [LESS_MONO_EQ])
     THEN RES_TAC THEN RES_TAC) ;;

%-------------------------------------------------------------------------
  lemma1 : for a continous f and cpo <, {f^n Bot | 0<=n} is directed
-------------------------------------------------------------------------%

let lemma = prove
   ("!r a (b:*). REFL r /\ r a b ==> isCUP r {a,b} b",
     REPEAT STRIP_TAC
     THEN IMP_RES_TAC isCUP_ORDER) ;;

let lemma1 = prove
   ("!r (f:*->*). CPO r /\ Cont r r f ==>
	          Directed r {x | ?n. x = ITER n f (Bot r)}",
     REPEAT STRIP_TAC
     THEN IMP_RES_TAC lemma0
     THEN UNDISCH_ALL_TAC THEN REWRITE_TAC [CPO; PO]
     THEN REPEAT STRIP_TAC
     THEN REWRITE_TAC [Directed] 
     THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
     THEN REPEAT STRIP_TAC
     THEN ASM_REWRITE_TAC[] 
     THEN DISJ_CASES_TAC (SPECL ["n:num"; "n':num"] LESS_CASES)
     THENL 
     [ EXISTS_TAC "ITER n' (f:*->*) (Bot r)"
       THEN MATCH_MP_TAC lemma THEN RES_TAC
       THEN ASM_REWRITE_TAC[] ;
       UNDISCH_TAC "n'<=n"
       THEN REWRITE_TAC [LESS_OR_EQ]
       THEN STRIP_TAC
       THENL
       [ EXISTS_TAC "ITER n (f:*->*) (Bot r)"
         THEN ONCE_REWRITE_TAC [INSERT_COMM]
         THEN MATCH_MP_TAC lemma THEN RES_TAC
         THEN ASM_REWRITE_TAC[] ;
         EXISTS_TAC "ITER n (f:*->*) (Bot r)"
         THEN ASM_REWRITE_TAC[]  
         THEN MATCH_MP_TAC lemma 
         THEN XRULE_ASSUM (REWRITE_RULE [REFL]) 
         THEN ASM_REWRITE_TAC[REFL]] 
        ]) ;;

%-------------------------------------------------------------------------
  lemma2 : for a continous f and cpo <, f*{f^n Bot | 0<=n} is directed
-------------------------------------------------------------------------%

let lemma_huh = prove
  ("!r (f:*->*). ~({x | ?n. x = ITER n f (Bot r)} = {})",
    REWRITE_TAC [SYM (SPEC_ALL MEMBER_NOT_EMPTY)]
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN REPEAT GEN_TAC
    THEN EXISTS_TAC "Bot (r:*->*->bool)"
    THEN EXISTS_TAC "0"
    THEN REWRITE_TAC [ITER]) ;;

let lemma2 = prove
   ("!r (f:*->*). CPO r /\ Cont r r f ==>
	          Directed r (IMAGE f {x | ?n. x = ITER n f (Bot r)})",
     REPEAT STRIP_TAC
     THEN IMP_RES_TAC lemma1
     THEN ASSUME_TAC (SPECL ["r:*->*->bool"; "f:*->*"] lemma_huh)
     THEN XRULE_ASSUM (REWRITE_RULE [Cont])
     THEN RES_TAC) ;;

%-------------------------------------------------------------------------
  lemma3 : for a cpo <, Bot is the bottom element.
-------------------------------------------------------------------------%

let lemma3 = prove
    ("!r. CPO r ==> (!x:*. r (Bot r) x)",
      REWRITE_TAC [CPO] THEN REPEAT STRIP_TAC
      THEN IMP_RES_TAC Bot_THM
      THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
  lemma4 : for a cpo < and cont. f: 
           CUP of {f^n Bot | 0<=n} exists.
-------------------------------------------------------------------------%

let lemma4 = prove
   ("!r (f:*->*). CPO r /\ Cont r r f ==> 
       isCUP r {x | ?n. x = ITER n f (Bot r)} 
                (CUP r {x | ?n. x = ITER n f (Bot r)})",
     REPEAT STRIP_TAC THEN IMP_RES_TAC lemma1
     THEN MATCH_MP_TAC CUP
     THEN UNDISCH_ALL_TAC THEN REWRITE_TAC [CPO]
     THEN REPEAT STRIP_TAC 
     THEN RES_TAC
     THEN EXISTS_TAC "a':*" THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
  lemma5 : for a cpo < and cont. f:
           CUP of f*{f^n Bot | 0<=n} exists.
-------------------------------------------------------------------------%

let lemma5 = prove
   ("!r (f:*->*). CPO r /\ Cont r r f ==> 
       isCUP r (IMAGE f {x | ?n. x = ITER n f (Bot r)})
                (CUP r (IMAGE f {x | ?n. x = ITER n f (Bot r)}))",
     REPEAT STRIP_TAC THEN IMP_RES_TAC lemma2
     THEN MATCH_MP_TAC CUP
     THEN UNDISCH_ALL_TAC THEN REWRITE_TAC [CPO]
     THEN REPEAT STRIP_TAC 
     THEN RES_TAC
     THEN EXISTS_TAC "a':*" THEN ASM_REWRITE_TAC[]) ;;

%-------------------------------------------------------------------------
  lemma6 : for a cpo < and cont. f:
            CUP f*{f^n Bot | 0<=n} is also the CUP of
           {f^n Bot | 0<=n}
-------------------------------------------------------------------------%

let lemma6 = prove
   ("!r (f:*->*). CPO r /\ Cont r r f ==> 
       isCUP r {x | ?n. x = ITER n f (Bot r)}
                (CUP r (IMAGE f {x | ?n. x = ITER n f (Bot r)}))",
     REPEAT STRIP_TAC THEN IMP_RES_TAC lemma5
     THEN UNDISCH_ALL_TAC THEN REWRITE_TAC [isCUP; IS_UB; IN_IMAGE]
     THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
     THEN REPEAT STRIP_TAC 
     THENL
     [ DISJ_CASES_TAC (SPEC "n:num" num_CASES)
       THENL
       [ ASM_REWRITE_TAC [ITER] 
         THEN IMP_RES_TAC lemma3 THEN ASM_REWRITE_TAC[] ;
         UNDISCH_ALL_TAC THEN REPEAT STRIP_TAC
         THEN FIRST_ASSUM MATCH_MP_TAC
         THEN ASM_REWRITE_TAC [ITER]    
         THEN EXISTS_TAC "ITER n' (f:*->*) (Bot r)"     
         THEN ASM_REWRITE_TAC []
         THEN EXISTS_TAC "n':num" THEN ASM_REWRITE_TAC [] ] ;
       SUBGOAL_THEN 
       "!a:*. (?x. (a = f x) /\ (?n. x = ITER n f(Bot r))) ==> r a c"
       ASSUME_TAC
       THENL [ ALL_TAC; RES_TAC]
       THEN REPEAT STRIP_TAC 
       THEN FIRST_ASSUM MATCH_MP_TAC
       THEN ASM_REWRITE_TAC[]
       THEN EXISTS_TAC "SUC n" THEN REWRITE_TAC [ITER] 
     ]) ;;

%-------------------------------------------------------------------------
  lemma7 : for a cpo < and cont. f:
           CUP {f^n Bot | 0<=n} = CUP f*{f^n Bot | 0<=n}
-------------------------------------------------------------------------%

let lemma7 = prove
  ("!r (f:*->*). CPO r /\ Cont r r f ==>
                (CUP r {x | ?n. x = ITER n f(Bot r)} =
                 CUP r (IMAGE f {x | ?n. x = ITER n f(Bot r)}))",
    REPEAT STRIP_TAC
    THEN MATCH_MP_TAC UNIQUE_CUP
    THEN EXISTS_TAC "r:*->*->bool"
    THEN EXISTS_TAC "{x:* | ?n. x = ITER n f(Bot r)}"
    THEN IMP_RES_TAC lemma4
    THEN IMP_RES_TAC lemma6
    THEN XRULE_ASSUM (REWRITE_RULE [CPO; PO])
    THEN ASM_REWRITE_TAC[]) ;;
 
%-------------------------------------------------------------------------
  lemma8 : for a cpo < and cont. f:
           CUP f^n Bot | 0<=n} is a fix point
-------------------------------------------------------------------------%

let lemma8 = prove
  ("!r (f:*->*). CPO r /\ Cont r r f ==>
                (f (CUP r {x | ?n. x = ITER n f(Bot r)}) =
                    CUP r {x | ?n. x = ITER n f(Bot r)})",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC lemma7
    THEN XRULE_ASSUM SYM
    THEN IMP_RES_TAC lemma1
    THEN XRULE_ASSUM (REWRITE_RULE [Cont])
    THEN ASSUME_TAC (SPECL ["r:*->*->bool"; "f:*->*"] lemma_huh)
    THEN RES_TAC
    THEN ASM_REWRITE_TAC[]) ;;
 
%-------------------------------------------------------------------------
  lemma9 : for a cpo < and cont. f:
           CUP f^n Bot | 0<=n} less than any other fix point
-------------------------------------------------------------------------%

let lemma9 = prove
  ("!r (f:*->*) x. CPO r /\ Cont r r f /\ (f x = x) 
                   ==>
                   (r (CUP r {x | ?n. x = ITER n f(Bot r)}) x)",
    REPEAT STRIP_TAC
    THEN IMP_RES_TAC lemma4
    THEN UNDISCH_ALL_TAC
    THEN REWRITE_TAC [isCUP; IS_UB]
    THEN (CONV_TAC o DEPTH_CONV) SET_SPEC_CONV
    THEN REPEAT STRIP_TAC
    THEN FIRST_ASSUM MATCH_MP_TAC
    THEN REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[]
    THEN POP_ASSUM (\thm. ALL_TAC)
    THEN SPEC_TAC ("n:num","n:num")
    THEN INDUCT_TAC
    THEN REWRITE_TAC [ITER] THEN REPEAT STRIP_TAC
    THENL 
    [ IMP_RES_TAC lemma3 THEN ASM_REWRITE_TAC[] ;
      ONCE_REWRITE_TAC [SYM (ASSUME "f x = (x:*)")] 
      THEN IMP_RES_TAC Cont_IMP_Mono
      THEN XRULE_ASSUM (REWRITE_RULE [Mono])
      THEN RES_TAC ]) ;;

%-------------------------------------------------------------------------
   Tarksi fix point theorem for CPO:
   For CPO r and continuous f, CUP {f^n Bot | 0<=n} is the least fix
   point of f.
-------------------------------------------------------------------------%
                  
let TARSKI_CPO = prove_thm
  (`TARSKI_CPO`,
   "!r (f:*->*).
       CPO r /\ Cont r r f ==> 
       isLFP r f (CUP r {x | ?n. x = ITER n f (Bot r)})",
    REWRITE_TAC [isLFP]
    THEN REPEAT STRIP_TAC
    THENL
    [ IMP_RES_TAC lemma8 ; IMP_RES_TAC lemma9]) ;;


close_theory ();;


