(****************************************************************************)
(* FILE          : rationals.sml                                            *)
(* DESCRIPTION   : Type of rational numbers.                                *)
(*                                                                          *)
(* AUTHOR (HOL88): R.J.Boulton, University of Cambridge                     *)
(* DATE          : 4th March 1991                                           *)
(*                                                                          *)
(* TRANSLATOR    : R.J.Boulton, University of Cambridge                     *)
(* DATE          : 16th February 1993                                       *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton, University of Edinburgh                     *)
(* DATE          : 17th February 1998                                       *)
(****************************************************************************)

signature RATIONAL =
sig
   exception Rational
   exception Inverse
   exception Div
   eqtype rat
   val rational : int * int -> rat
   val numerator : rat -> int
   val denominator : rat -> int
   val inverse : rat -> rat
   val rat_of_int : int -> rat
   val lower_int_of_rat : rat -> int
   val upper_int_of_rat : rat -> int
   val zero : rat
   val one : rat
   val ~ : rat -> rat
   val + : rat * rat -> rat
   val - : rat * rat -> rat
   val * : rat * rat -> rat
   val / : rat * rat -> rat
   val < : rat * rat -> bool
   val print : rat -> unit
end;

structure Rational : RATIONAL =
struct

local

(*--------------------------------------------------------------------------*)
(* Function to compute the Greatest Common Divisor of two integers.         *)
(*--------------------------------------------------------------------------*)

exception Gcd;

fun gcd (i,j) =
   let fun gcd' (i,j) =
          let val r = (i mod j)
          in  if (r = 0)
              then j
              else gcd' (j,r)
          end
   in  (if ((i < 0) orelse (j < 0))
        then raise Gcd
        else if (i < j) then gcd' (j,i) else gcd' (i,j)
       ) handle Mod => raise Gcd
   end;

in

(*==========================================================================*)
(* Rational arithmetic                                                      *)
(*==========================================================================*)

exception Rational;
exception Inverse;
exception Div;

(*--------------------------------------------------------------------------*)
(* Abstract datatype for rational numbers.                                  *)
(*                                                                          *)
(* rational    : int * int -> rat                                           *)
(* numerator   : rat -> int                                                 *)
(* denominator : rat -> int                                                 *)
(* inverse     : rat -> rat                                                 *)
(* rat_plus    : rat * rat -> rat                                           *)
(* rat_minus   : rat * rat -> rat                                           *)
(* rat_mult    : rat * rat -> rat                                           *)
(* rat_div     : rat * rat -> rat                                           *)
(* print       : rat -> unit                                                *)
(*--------------------------------------------------------------------------*)

datatype rat = Rat of int * int;

fun rational (i,j) =
   if (i = 0) then Rat (0,1)
   else if (j = 1) then Rat (i,1)
   else let val g = gcd (abs i,abs j)
            val i' = i div g
            and j' = j div g
        in  if (j' < 0)
            then Rat (~i',~j')
            else Rat (i',j')
        end
        handle _ => raise Rational;

fun numerator (Rat (i,_)) = i;

fun denominator (Rat (_,j)) = j;

fun inverse (Rat (i,j)) =
   if (i < 0) then Rat ((~j),(~i))
   else if (i > 0) then Rat (j,i)
   else raise Inverse;

fun rat_plus (Rat (i1,j1),Rat (i2,j2)) =
   let val g = gcd (j1,j2)
       val d1 = j1 div g
       and d2 = j2 div g
       val (i,j) = ((i1 * d2) + (i2 * d1),(j1 * d2))
   in  if (i = 0) then Rat (0,1) else Rat (i,j)
   end;

fun rat_minus (Rat (i1,j1),Rat (i2,j2)) =
   let val g = gcd (j1,j2)
       val d1 = j1 div g
       and d2 = j2 div g
       val (i,j) = ((i1 * d2) - (i2 * d1),(j1 * d2))
   in  if (i = 0) then Rat (0,1) else Rat (i,j)
   end;

fun rat_mult (Rat (i1,j1),Rat (i2,j2)) =
   if ((i1 = 0) orelse (i2 = 0))
   then Rat (0,1)
   else let val g = gcd (abs i1,j2)
            and h = gcd (abs i2,j1)
            val i = (i1 div g) * (i2 div h)
            and j = (j1 div h) * (j2 div g)
        in  Rat (i,j)
        end;

fun rat_div (Rat (i1,j1),Rat (i2,j2)) =
   if (i2 = 0) then raise Div
   else if (i1 = 0) then Rat (0,1)
   else let val g = gcd (abs i1,abs i2)
            and h = gcd (j1,j2)
            val i = (i1 div g) * (j2 div h)
            and j = (j1 div h) * (i2 div g)
        in  if (j < 0) then Rat ((~i),(~j)) else Rat (i,j)
        end;

fun print (Rat (i,j)) =
   let val print = TextIO.print o Int.toString
   in  if (j = 1)
       then print i
       else (print i; TextIO.print "/"; print j)
   end;

(*--------------------------------------------------------------------------*)
(* rat_of_int : int -> rat                                                  *)
(*                                                                          *)
(* Conversion from integers to rationals.                                   *)
(*--------------------------------------------------------------------------*)

fun rat_of_int i = rational (i,1);

(*--------------------------------------------------------------------------*)
(* lower_int_of_rat : rat -> int                                            *)
(*                                                                          *)
(* Conversion from rationals to integers.                                   *)
(*                                                                          *)
(* Computes the largest integer less than or equal to the rational.         *)
(*--------------------------------------------------------------------------*)

fun lower_int_of_rat r =
   let val n = numerator r
       and d = denominator r
   in  if (n < 0)
       then let val p = (n * d) in (((n - p) div d) + n) end
       else (n div d)
   end;

(*--------------------------------------------------------------------------*)
(* upper_int_of_rat : rat -> int                                            *)
(*                                                                          *)
(* Conversion from rationals to integers.                                   *)
(*                                                                          *)
(* Computes the smallest integer greater than or equal to the rational.     *)
(*--------------------------------------------------------------------------*)

fun upper_int_of_rat r =
   let val n = numerator r
       and d = denominator r
   in  if (n > 0)
       then let val p = (n * d) in (((n - p) div d) + n) end
       else (n div d)
   end;

(*--------------------------------------------------------------------------*)
(* The rational number zero.                                                *)
(*--------------------------------------------------------------------------*)

val zero = rat_of_int 0;

(*--------------------------------------------------------------------------*)
(* The rational number one.                                                 *)
(*--------------------------------------------------------------------------*)

val one = rat_of_int 1;

(*--------------------------------------------------------------------------*)
(* Rational arithmetic operators.                                           *)
(*--------------------------------------------------------------------------*)

fun ~ r = rat_minus (zero,r);
val op + = rat_plus
and op - = rat_minus
and op * = rat_mult
and op / = rat_div;
fun r1 < r2 = Int.< (numerator (r1 - r2),0);

end;

end; (* Rational *)
