(****************************************************************************)
(*                                                                          *)
(*               Copyright 1994, 1996 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : ppboxes.sml                                              *)
(* DESCRIPTION   : Box abstraction for pretty-printing.                     *)
(*                                                                          *)
(* AUTHOR (HOL88): R.J.Boulton                                              *)
(* DATE          : 1989                                                     *)
(*                                                                          *)
(* TRANSLATED BY : R.J.Boulton                                              *)
(* DATE          : 14th June 1994                                           *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 1st July 1996                                            *)
(****************************************************************************)

functor LabelPPBoxesFun (Scale : sig val char_size : int end) : LABEL_PPBOXES =
struct

open Portable;

structure Scale =
struct

open Scale;

fun num_of_chars i =
   let val scaled = i div Scale.char_size
       val i' = scaled * Scale.char_size
   in  if (i' = i) then scaled else scaled + 1
   end;

end;

open Scale;

val one_char = char_size;

(*--------------------------------------------------------------------------*)
(* Auxiliary functions.                                                     *)
(*                                                                          *)
(* replicate : int -> string -> string                                      *)
(* itlist : ('a -> 'b -> 'b) -> 'a list -> 'b -> 'b                         *)
(*--------------------------------------------------------------------------*)

fun replicate n s =
   if (n < 1) then "" else s ^ (replicate (n - 1) s);

fun itlist f [] y = y
  | itlist f (x::xs) y = f x (itlist f xs y);

(*--------------------------------------------------------------------------*)
(* Datatype for boxes of text.                                              *)
(*                                                                          *)
(* A box looks like this:                                                   *)
(*                                                                          *)
(*        <-io-->@__________________                                        *)
(*         ______|                  |   |                                   *)
(*        |                         | height                                *)
(*        |                _________|   |                                   *)
(*        |_______________|             |                                   *)
(*        <------fo------->                                                 *)
(*        <----------width---------->                                       *)
(*                                                                          *)
(*                                                                          *)
(* `N_box' (null box) is a box with dimensions of length zero.              *)
(* `A_box ((width,s),_)' (atomic box) is a box of height 1, width the       *)
(*    length of the string s (so width is redundant, but useful for         *)
(*    efficiency), io = 0 and fo = width.                                   *)
(* `L_box ((width,separation,pb1,pb2),_)' (linear box) is a box of height   *)
(*    1, io = 0 and fo = width. separation is the number of spaces between  *)
(*    the back of pb1 and the front of pb2. A linear box is a special case  *)
(*    of a compound box. The advantage of using a linear box when possible  *)
(*    is that it takes up less memory.                                      *)
(* `C_box (((io,width,fo),height,(x,y),pb1,pb2),_)' (compound box) is a box *)
(*    made from two other boxes, pb1 and pb2. The dimensions of the         *)
(*    compound box are included. (x,y) are the horizontal (to the right)    *)
(*    and vertical (downwards) offset of pb2 within the compound box. The   *)
(*    offsets are measured between the origins of the boxes (labelled by @  *)
(*    in the diagram). The offsets of pb1 are (0,0).                        *)
(*--------------------------------------------------------------------------*)

datatype 'a print_box
   = N_box
   | A_box of (int * string) * 'a
   | L_box of (int * int * 'a print_box * 'a print_box) * 'a
   | C_box of ((int * int * int) * int * (int * int) *
                  'a print_box * 'a print_box) * 'a;

(*--------------------------------------------------------------------------*)
(* print_box_io : 'a print_box -> int                                       *)
(* print_box_width : 'a print_box -> int                                    *)
(* print_box_fo : 'a print_box -> int                                       *)
(* print_box_height : 'a print_box -> int                                   *)
(* print_box_sizes : 'a print_box -> (int * int * int) * int                *)
(*                                                                          *)
(* Functions to extract the dimensions of a box.                            *)
(*--------------------------------------------------------------------------*)

fun print_box_io N_box = 0
  | print_box_io (A_box _) = 0
  | print_box_io (L_box _) = 0
  | print_box_io (C_box (((io,_,_),_,_,_,_),_)) = io;

fun print_box_width N_box = 0
  | print_box_width (A_box ((width,_),_)) = width
  | print_box_width (L_box ((width,_,_,_),_)) = width
  | print_box_width (C_box (((_,width,_),_,_,_,_),_)) = width;

fun print_box_fo N_box = 0
  | print_box_fo (A_box ((width,_),_)) = width
  | print_box_fo (L_box ((width,_,_,_),_)) = width
  | print_box_fo (C_box (((_,_,fo),_,_,_,_),_)) = fo;

fun print_box_height N_box = 0
  | print_box_height (A_box _) = one_char
  | print_box_height (L_box _) = one_char
  | print_box_height (C_box ((_,height,_,_,_),_)) = height;

fun print_box_sizes N_box = ((0,0,0),0)
  | print_box_sizes (A_box ((w,_),_)) = ((0,w,w),one_char)
  | print_box_sizes (L_box ((w,_,_,_),_)) = ((0,w,w),one_char)
  | print_box_sizes (C_box (((io,w,fo),h,_,_,_),_)) = ((io,w,fo),h);

(*--------------------------------------------------------------------------*)
(* join_boxes                                                               *)
(*    : int -> int -> 'a print_box -> 'a print_box -> 'a -> 'a print_box    *)
(*                                                                          *)
(* Function for joining two boxes together.                                 *)
(*                                                                          *)
(* `x' and `y' have rather strange definitions which allow the one function *)
(* to be used for joining boxes both horizontally and vertically. Note that *)
(* `join_boxes' does not work properly with boxes of zero height.           *)
(*                                                                          *)
(* The intermediate values `lo' and `ro' are illustrated (both with         *)
(* positive values) in the diagram below:                                   *)
(*                                                                          *)
(*                         _______ <-ro->                                   *)
(*                    ____|  _____|_____                                    *)
(*                   |______|_|      ___|                                   *)
(*                       |__________|                                       *)
(*                   <lo>                                                   *)
(*                                                                          *)
(* The composition of the two boxes looks like this:                        *)
(*                                                                          *)
(*                         _____________                                    *)
(*                    ____|             |                                   *)
(*                   |               ___|                                   *)
(*                   |______________|                                       *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

fun join_boxes x y pb1 pb2 v =
   let val ((io1,w1,fo1),h1) = print_box_sizes pb1
       and ((io2,w2,fo2),h2) = print_box_sizes pb2
       val lo = x - io2
       and ro = (w2 - io2) - (w1 - x)
       val io = if (lo < 0) then (io1 - lo) else io1
       and w  = if (lo < 0)
                then if (ro < 0) then (w1 - lo) else w2
                else if (ro < 0) then w1 else (w2 + lo)
       and fo = if (lo < 0) then fo2 else (fo2 + lo)
       and h  = h1 + h2 + y
       and x2 = x - io1
       and y2 = h1 + y
   in  if (h = one_char)
       then L_box ((w,x2 - w1,pb1,pb2),v)
       else C_box (((io,w,fo),h,(x2,y2),pb1,pb2),v)
   end;

(*--------------------------------------------------------------------------*)
(* join_h_boxes : int -> 'a print_box -> 'a print_box -> 'a -> 'a print_box *)
(*                                                                          *)
(* Function to join boxes horizontally with separation `dx'.                *)
(*                                                                          *)
(* Composition with an `N_box' leaves the other box unchanged.              *)
(*                                                                          *)
(* Composing two boxes horizontally:                                        *)
(*                                                                          *)
(*                          |dx|                                            *)
(*                         _______                                          *)
(*                    ____|  _____|_____                                    *)
(*                   |______|__|     ___|   | -y                            *)
(*                       |__________|                                       *)
(*                   <----x---->                                            *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

fun join_h_boxes _ N_box pb2 _ = pb2
  | join_h_boxes _ pb1 N_box _ = pb1
  | join_h_boxes dx pb1 pb2 v =
   join_boxes ((print_box_fo pb1) + dx) (~ one_char) pb1 pb2 v;

(*--------------------------------------------------------------------------*)
(* join_v_boxes                                                             *)
(*    : int -> int -> 'a print_box -> 'a print_box -> 'a -> 'a print_box    *)
(*                                                                          *)
(* Function to join boxes vertically with separation `dh' and indentation   *)
(* `di'.                                                                    *)
(*                                                                          *)
(* Composition with an `N_box' leaves the other box unchanged.              *)
(*                                                                          *)
(* Composing two boxes vertically:                                          *)
(*                                                                          *)
(*                         _______                                          *)
(*                    ____|     __|                                         *)
(*                   |_________|                                            *)
(*                        <di>_______    | y = dh                           *)
(*                      _____|     __|                                      *)
(*                     |__________|                                         *)
(*                   <---x--->                                              *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

fun join_v_boxes _ _ N_box pb2 _ = pb2
  | join_v_boxes _ _ pb1 N_box _ = pb1
  | join_v_boxes di dh pb1 pb2 v =
   join_boxes ((print_box_io pb1) + di) dh pb1 pb2 v;

(*--------------------------------------------------------------------------*)
(* join_strings                                                             *)
(*    : (int * string) * int ->                                             *)
(*      (int * string) * int ->                                             *)
(*      (int * string) * int                                                *)
(*                                                                          *)
(* Function to join two strings with specified x-coordinates into one       *)
(* string. The function fails if the strings are overlapping.               *)
(*--------------------------------------------------------------------------*)

fun join_strings ((w1,s1),x1) ((w2,s2),x2) =
   if (x1 = x2)
   then if ((w1 = 0) orelse (w2 = 0))
        then ((w1 + w2,s1 ^ s2),x1)
        else raise Fail "join_strings -- overlapping strings"
   else if (x1 < x2)
        then let val sep = x2 - (x1 + w1)
             in  if (sep < 0)
                 then raise Fail "join_strings -- overlapping strings"
                 else ((w1 + sep + w2,
                        s1 ^ (replicate (num_of_chars sep) " ") ^ s2),
                       x1)
             end
        else let val sep = x1 - (x2 + w2)
             in  if (sep < 0)
                 then raise Fail "join_strings -- overlapping strings"
                 else ((w2 + sep + w1,
                        s2 ^ (replicate (num_of_chars sep) " ") ^ s1),
                       x2)
             end;

(*--------------------------------------------------------------------------*)
(* merge_string_lists                                                       *)
(*    : ((int * string) * int * int) list ->                                *)
(*      ((int * string) * int * int) list ->                                *)
(*      ((int * string) * int * int) list                                   *)
(*                                                                          *)
(* Function to merge two lists of strings with x and y-coordinates.         *)
(*                                                                          *)
(* The two input lists should be in increasing order of y coordinate and    *)
(* contain at most one string for each value of y.  The resulting list has  *)
(* the same properties.                                                     *)
(*--------------------------------------------------------------------------*)

fun merge_string_lists sl1 sl2 =
   if (null sl1)
   then sl2
   else if (null sl2)
        then sl1
        else let val (ws1,x1,y1:int) = hd sl1
                 and (ws2,x2,y2:int) = hd sl2
             in  if (y1 = y2) then
                    let val (ws,x) = join_strings (ws1,x1) (ws2,x2)
                    in  (ws,x,y1)::(merge_string_lists (tl sl1) (tl sl2))
                    end
                 else if (y1 < y2) then
                    (hd sl1)::(merge_string_lists (tl sl1) sl2)
                 else if (y1 > y2) then
                    (hd sl2)::(merge_string_lists sl1 (tl sl2))
                 else raise Fail "merge_string_lists"
             end;

(*--------------------------------------------------------------------------*)
(* stringify_print_box                                                      *)
(*    : int -> int -> 'a print_box -> ((int * string) * int * int) list     *)
(*                                                                          *)
(* Function to convert a box of text into a list of strings.                *)
(*                                                                          *)
(* The x,y coordinates of the origin of the box must be specified.          *)
(* The `labels' in the box are discarded.                                   *)
(*--------------------------------------------------------------------------*)

fun stringify_print_box _ _ N_box = []
  | stringify_print_box x y (A_box (ws,_)) = [(ws,x,y)]
  | stringify_print_box x y (L_box ((_,sep,pb1,pb2),_)) =
   merge_string_lists (stringify_print_box x y pb1)
      (stringify_print_box (x + (print_box_width pb1) + sep) y pb2)
  | stringify_print_box x y (C_box ((_,_,(x2,y2),pb1,pb2),_)) =
   merge_string_lists (stringify_print_box x y pb1)
      (stringify_print_box (x + x2) (y + y2) pb2);

(*--------------------------------------------------------------------------*)
(* fill_in_strings : bool -> int -> int ->                                  *)
(*                   ((int * string) * int * int) list -> string list       *)
(*                                                                          *)
(* Function to convert a list of strings (with coordinates) into a list of  *)
(* strings suitable for use as output.                                      *)
(*                                                                          *)
(* The y coordinates of the top and bottom of the block of text must be     *)
(* specified. If any of the strings in the input list are out of the text   *)
(* region specified, an error occurs. This error will only reach top-level  *)
(* if debugging is set to true. Otherwise, the string `*error*' is inserted *)
(* in the text produced.                                                    *)
(*--------------------------------------------------------------------------*)

fun fill_in_strings debug t b sl =
   if (t >= b)
   then if (null sl)
        then []
        else if debug
             then raise Fail "fill_in_strings -- string below specified region"
             else ["*error*"]
   else if (null sl)
        then ""::(fill_in_strings debug (t + one_char) b sl)
        else let val ((_,s),x,y) = hd sl
             in  if (x < 0)
                 then if debug
                      then raise
                              Fail ("fill_in_strings -- " ^
                                    "string to the left of specified region")
                      else fill_in_strings debug t b
                              (((7 * one_char,"*error*"),0,y)::(tl sl))
                 else if (y < t)
                      then if debug
                           then raise Fail ("fill_in_strings -- " ^
                                            "string above specified region")
                           else "*error*"::(fill_in_strings debug t b (tl sl))
                      else if (y = t)
                           then ((replicate (num_of_chars x) " ") ^ s)::
                                   (fill_in_strings debug (t + one_char) b
                                       (tl sl))
                           else ""::(fill_in_strings debug (t + one_char) b sl)
             end;

(*--------------------------------------------------------------------------*)
(* print_box_to_strings : bool -> int -> 'a print_box -> string list        *)
(*                                                                          *)
(* Function to convert a box of text into a list of strings suitable for    *)
(* output.                                                                  *)
(*                                                                          *)
(* An indentation from the left margin must be specified. The `debug'       *)
(* argument determines whether or not errors reach top-level.               *)
(*                                                                          *)
(* The `labels' in the box are discarded.                                   *)
(*--------------------------------------------------------------------------*)

fun print_box_to_strings debug i pb =
   fill_in_strings debug 0 (print_box_height pb) (stringify_print_box i 0 pb);

(*--------------------------------------------------------------------------*)
(* Type for box abstraction.                                                *)
(*--------------------------------------------------------------------------*)

type 'a ppbox = int -> int -> 'a print_box;

(*--------------------------------------------------------------------------*)
(* Datatype for indentation values.                                         *)
(*                                                                          *)
(* `ABSOLUTE' is absolute indentation (relative to first sub-box).          *)
(* `RELATIVE' is incremental indentation (relative to previous sub-box).    *)
(*--------------------------------------------------------------------------*)

datatype ppindent = ABSOLUTE of int | RELATIVE of int;

(*--------------------------------------------------------------------------*)
(* pp_empty_box : int -> int -> 'a print_box                                *)
(*                                                                          *)
(* Empty box.                                                               *)
(*--------------------------------------------------------------------------*)

val pp_empty_box = fn (m:int) => fn (i:int) => N_box;

(*--------------------------------------------------------------------------*)
(* pp_string_box : 'a -> string * int -> (int -> int -> 'a print_box)       *)
(*                                                                          *)
(* Make a box from a string. The integer argument is the width of the box.  *)
(*--------------------------------------------------------------------------*)

fun pp_string_box label (s,w) =
   fn (m:int) => fn (i:int) => A_box ((w,s),label);

(*--------------------------------------------------------------------------*)
(* pp_h_sepbox                                                              *)
(*    : 'a ->                                                               *)
(*      (int -> int -> 'a print_box) ->                                     *)
(*      (int * (int -> int -> 'a print_box)) list ->                        *)
(*      (int -> int -> 'a print_box)                                        *)
(*                                                                          *)
(* Function to build a horizontal (h) box.                                  *)
(*                                                                          *)
(* The sub-function `gaps' is used to compute the total separation between  *)
(* the sub-boxes. To this is added the number of sub-boxes (less the        *)
(* first). The available width (m) is then reduced by this total to give    *)
(* the new available width (m'). This is an attempt to guess how much space *)
(* to leave on the line for the remainder of the sub-boxes. The effective   *)
(* width of each sub-box is assumed to be one. In fact it could be any      *)
(* value, even negative. The heuristic seems to work well in practice       *)
(* though, probably because most horizontal boxes that are large enough to  *)
(* spread over more than one line are of the form parenthesis - large block *)
(* - parenthesis, or in place of the parentheses, some other single         *)
(* character.                                                               *)
(*                                                                          *)
(* As each sub-box is built, the gap between it and the previous sub-box    *)
(* plus one is added back on to the available width, and the indentation    *)
(* from the left margin is changed by the genuine amount. In fact, the      *)
(* indentation is computed each time from the original indentation, the     *)
(* effective width of the box built so far, and the effective width of the  *)
(* latest sub-box.                                                          *)
(*--------------------------------------------------------------------------*)

fun pp_h_sepbox label box boxl m i =
   let fun f pb m' boxl' =
          if (null boxl')
          then pb
          else let val (dx,pbfn) = hd boxl'
                   val m'' = m' + one_char + dx
                   and i' = i + ((print_box_fo pb) - (print_box_io pb)) + dx
               in  f (join_h_boxes dx pb (pbfn m'' i') label) m'' (tl boxl')
               end
       fun gaps boxl' = itlist (fn (x,_) => fn n => x + n) boxl' 0
       val m' = m - ((gaps boxl) + (length boxl))
   in  f (box m' i) m' boxl
   end;

(*--------------------------------------------------------------------------*)
(* pp_v_sepbox                                                              *)
(*    : 'a ->                                                               *)
(*      (int -> int -> 'a print_box) ->                                     *)
(*      ((ppindent * int) * (int -> int -> 'a print_box)) list ->           *)
(*      (int -> int -> 'a print_box)                                        *)
(*                                                                          *)
(* Function to build a vertical (v) box.                                    *)
(*                                                                          *)
(* The sub-boxes are composed vertically. The indentation from the left     *)
(* margin is modified as each sub-box is added.                             *)
(*--------------------------------------------------------------------------*)

fun pp_v_sepbox label box boxl (m:int) i =
   let fun f pb i' boxl' =
          if (null boxl')
          then pb
          else let val ((pi,dh),pbfn) = hd boxl'
                   val di = case pi
                            of (ABSOLUTE n) => n
                             | (RELATIVE n) => (n + i' - i)
               in  f (join_v_boxes di dh pb (pbfn m (i + di)) label)
                                                (i + di) (tl boxl')
               end
   in  f (box m i) i boxl
   end;

(*--------------------------------------------------------------------------*)
(* pp_hv_sepbox                                                             *)
(*    : 'a ->                                                               *)
(*      (int -> int -> 'a print_box) ->                                     *)
(*      ((int * ppindent * int) * (int -> int -> 'a print_box)) list ->     *)
(*      (int -> int -> 'a print_box)                                        *)
(*                                                                          *)
(* Function to build a horizontal/vertical (hv) box.                        *)
(*                                                                          *)
(* The sub-function `fh' generates a list of boxes to be composed           *)
(* vertically where each box has been made up by composing one or more of   *)
(* the original sub-boxes horizontally. The list generated is in reverse    *)
(* order and the indentations for the vertical composition are offsets from *)
(* the first box. Note that the function used with `itlist' reverses its    *)
(* arguments. Consideration of the call to `itlist' should reveal the       *)
(* rather delicate nature of the composition occurring. The order in which  *)
(* the composition is done is crucially linked with whether the             *)
(* indentations are absolute or relative.                                   *)
(*                                                                          *)
(* The sub-function builds horizontal boxes until they are too big, and     *)
(* then adds them to a list of boxes built so far. When trying to add a     *)
(* sub-box to the current horizontal box, the function evaluates by how     *)
(* much the offset from the left margin (i') will be increased if a         *)
(* line-break is not used. If this is less than or equal to the increase    *)
(* that will occur with a line-break, the sub-box is added to the           *)
(* horizontal box regardless.                                               *)
(*                                                                          *)
(* The function uses two criteria for determining when to break. If the new *)
(* box is wider than the available space, a break must occur. There must    *)
(* also be a break if the right-hand edge of the box exceeds the right-hand *)
(* margin. The two criteria are not necessarily the same because the        *)
(* indentation may force the box further to the right. Since the            *)
(* indentation can also be negative, it could pull the box to the left,     *)
(* giving a false result. For this reason negative indentations are taken   *)
(* to be zero.                                                              *)
(*                                                                          *)
(* The vertical composition parameters of the first sub-box of a horizontal *)
(* box are remembered when it is started, so that they become the           *)
(* parameters for the composite horizontal box.                             *)
(*--------------------------------------------------------------------------*)

fun pp_hv_sepbox label box boxl m i =
   let fun fh newboxl newbox i' boxl' =
          if (null boxl')
          then newbox::newboxl
          else let val ((dx,pi,dh),pbfn) = hd boxl'
                   and (newdi,newdh,pb) = newbox
                   val di = case pi
                            of (ABSOLUTE n) => n
                             | (RELATIVE n) => (n + i' - i)
                   and no_break_indent =
                          dx + (print_box_fo pb) - (print_box_io pb)
               in  if ((di - (i' - i)) < no_break_indent)
                   then let val newb = pbfn m (i + di)
                            val newhb = join_h_boxes dx pb newb label
                        in  if ((print_box_width newhb > m) orelse
                                (print_box_width newhb - print_box_io newhb >
                                 m - max (i',0)))
                            then fh (newbox::newboxl) (di,dh,newb)
                                                  (i + di) (tl boxl')
                            else fh newboxl (newdi,newdh,newhb) i' (tl boxl')
                        end
                   else let val newhb =
                               join_h_boxes dx pb
                                  (pbfn m (i' + no_break_indent)) label
                        in  fh newboxl (newdi,newdh,newhb) i' (tl boxl')
                        end
               end
       val newboxl = fh [] (0,0,box m i) i boxl
   in  itlist (fn (di,dh,pb2) => fn pb1 => join_v_boxes di dh pb1 pb2 label)
          newboxl N_box
   end;

(*--------------------------------------------------------------------------*)
(* pp_hov_sepbox                                                            *)
(*    : 'a ->                                                               *)
(*      (int -> int -> 'a print_box) ->                                     *)
(*      ((int * ppindent * int) * (int -> int -> 'a print_box)) list ->     *)
(*      (int -> int -> 'a print_box)                                        *)
(*                                                                          *)
(* Function to build a horizontal-or-vertical (hov) box.                    *)
(*                                                                          *)
(* The sub-function `f' computes the indentations for each box and builds   *)
(* the sub-boxes under the assumption that each will go on a new line.      *)
(*                                                                          *)
(* The body of the main function composes the boxes horizontally. If the    *)
(* resulting box is too big (see comments for `pp_hv_sepbox'), the boxes    *)
(* are composed vertically. The narrower of the two compositions is then    *)
(* used. See comments for `pp_hv_sepbox' regarding use of `itlist' for      *)
(* composing.                                                               *)
(*--------------------------------------------------------------------------*)

fun pp_hov_sepbox label box boxl m i =
   let fun f newboxl i' boxl' =
          if (null boxl')
          then newboxl
          else let val ((dx,pi,dh),pbfn) = hd boxl'
                   val di = case pi
                            of (ABSOLUTE n) => n
                             | (RELATIVE n) => (n + i' - i)
               in  f ((dx,di,dh,pbfn m (i + di))::newboxl) (i + di) (tl boxl')
               end
       val newb = box m i
       and newboxl = f [] i boxl
       val newhb =
          itlist (fn (dx,di,dh,pb2) => fn pb1 => join_h_boxes dx pb1 pb2 label)
             newboxl newb
       val hw = print_box_width newhb
       and hio = print_box_io newhb
   in  if ((hw > m) orelse (hw - hio > (m - max (i,0))))
       then let val newvb =
                   itlist (fn (dx,di,dh,pb2) => fn pb1 =>
                              join_v_boxes di dh pb1 pb2 label) newboxl newb
                val vw = print_box_width newvb
                and vio = print_box_io newvb
            in  if ((hw > vw) orelse (hw - hio > vw - vio))
                then newvb
                else newhb
            end
       else newhb
   end;

(*--------------------------------------------------------------------------*)
(* pp_h_box : 'a -> int -> ppbox list -> (int -> int -> 'a print_box)       *)
(*                                                                          *)
(* Horizontally formatted box with h spaces between subboxes.               *)
(*--------------------------------------------------------------------------*)

fun pp_h_box _ _ [] = pp_empty_box
  | pp_h_box label h (box::boxes) =
   pp_h_sepbox label box (map (fn box => (h,box)) boxes);

(*--------------------------------------------------------------------------*)
(* pp_v_box : 'a -> ppindent * int -> (int -> int -> 'a print_box) list ->  *)
(*            (int -> int -> 'a print_box)                                  *)
(*                                                                          *)
(* Vertically formatted box with indentation i and vertical separation v.   *)
(*--------------------------------------------------------------------------*)

fun pp_v_box _ (_,_) [] = pp_empty_box
  | pp_v_box label iv (box::boxes) =
   pp_v_sepbox label box (map (fn box => (iv,box)) boxes);

(*--------------------------------------------------------------------------*)
(* pp_hv_box                                                                *)
(*    : 'a -> int * ppindent * int -> (int -> int -> 'a print_box) list ->  *)
(*      (int -> int -> 'a print_box)                                        *)
(*                                                                          *)
(* Format subboxes horizontally if sufficient space, otherwise break        *)
(* inconsistently.                                                          *)
(*--------------------------------------------------------------------------*)

fun pp_hv_box _ (_,_,_) [] = pp_empty_box
  | pp_hv_box label hiv (box::boxes) =
   pp_hv_sepbox label box (map (fn box => (hiv,box)) boxes);

(*--------------------------------------------------------------------------*)
(* pp_hov_box                                                               *)
(*    : 'a -> int * ppindent * int -> (int -> int -> 'a print_box) list ->  *)
(*      (int -> int -> 'a print_box)                                        *)
(*                                                                          *)
(* Format subboxes horizontally if sufficient space, otherwise format the   *)
(* subboxes vertically.                                                     *)
(*--------------------------------------------------------------------------*)

fun pp_hov_box _ (_,_,_) [] = pp_empty_box
  | pp_hov_box label hiv (box::boxes) =
   pp_hov_sepbox label box (map (fn box => (hiv,box)) boxes);

(*--------------------------------------------------------------------------*)
(* Type of boxes committed to a particular line width.                      *)
(*--------------------------------------------------------------------------*)

type 'a text = 'a print_box * int;

(*--------------------------------------------------------------------------*)
(* commit_ppbox : int -> (int -> int -> 'a print_box) -> 'a print_box * int *)
(*                                                                          *)
(* Commit a ppbox to a specified line width.                                *)
(*--------------------------------------------------------------------------*)

fun commit_ppbox m box = (box m 0,m);

(*--------------------------------------------------------------------------*)
(* width_of_text : 'a print_box * int -> int                                *)
(*                                                                          *)
(* Determine the line width to which a ppbox has been committed.            *)
(*--------------------------------------------------------------------------*)

fun width_of_text (_,width) = width;

(*--------------------------------------------------------------------------*)
(* Structure for pretty-printer streams.                                    *)
(*--------------------------------------------------------------------------*)

structure PPStream : PPSTREAM =
struct

type ppstream = {consumer:string->unit,flush:unit->unit,linewidth:int};

fun mk_ppstream x = x
and dest_ppstream x = x
and clear_ppstream _ = ()
and flush_ppstream ({flush,...} : ppstream) = flush ();

end;

(*--------------------------------------------------------------------------*)
(* write_text : 'a print_box * int -> (ppstream -> unit)                    *)
(*                                                                          *)
(* Write a commited box to a ppstream. The box should have been committed   *)
(* with the width of the ppstream.                                          *)
(*--------------------------------------------------------------------------*)

fun write_text (pb,width : int) pps =
   let val {consumer,...} = PPStream.dest_ppstream pps
       fun print_strings ss =
          if (null ss)
          then ()
          else if (null (tl ss))
               then consumer (hd ss)
               else (consumer (hd ss);
                     consumer "\n";
                     print_strings (tl ss))
       val ss = print_box_to_strings false 0 pb
   in  print_strings ss
   end;

(*--------------------------------------------------------------------------*)
(* write_ppbox : (int -> int -> 'a print_box) -> (ppstream -> unit)         *)
(*                                                                          *)
(* Write a box to a ppstream.                                               *)
(*--------------------------------------------------------------------------*)

fun write_ppbox box pps =
   let val {linewidth,...} = PPStream.dest_ppstream pps
   in  write_text (commit_ppbox (linewidth * one_char) box) pps
   end;

exception Label_at;

local

(*--------------------------------------------------------------------------*)
(* is_in_text : (int * int) -> ('a print_box * int * int) -> bool           *)
(*                                                                          *)
(* Function to determine whether a specified pair of (x,y) coordinates lie  *)
(* within a print_box which has its origin at coordinates (x',y').          *)
(*--------------------------------------------------------------------------*)

fun is_in_text (x,y) (pb,x',y') =
   let val ((io,w,fo),h) = print_box_sizes pb
       val xi = x' - io
       and xw = x' + w - io
       and xf = x' + fo - io
       and yh = y' + h
   in  ((y = y') andalso ((x > x') orelse (x = x')) andalso (x < xw))
       orelse
       ((y = yh - one_char) andalso
        ((x > xi) orelse (x = xi)) andalso (x < xf))
       orelse
       ((y > y') andalso (y < yh - one_char) andalso
        ((x > xi) orelse (x = xi)) andalso (x < xw))
   end;

(*--------------------------------------------------------------------------*)
(* label_at : (int * int) -> 'a print_box -> 'a                             *)
(*                                                                          *)
(* Function to obtain the label of the smallest box containing the given    *)
(* (x,y) coordinates (the top left corner of the display for the print box  *)
(* is taken as the origin (0,0)).                                           *)
(*                                                                          *)
(* If there is no box at the specified coordinates, the function fails.     *)
(*--------------------------------------------------------------------------*)

fun label_at (x,y) N_box = raise Label_at
  | label_at (x,y) (pb as A_box (_,label)) =
   if (is_in_text (x,y) (pb,0,0))
   then label
   else raise Label_at
  | label_at (x,y) (L_box ((_,sep,pb1,pb2),label)) =
   let val x2 = (print_box_width pb1) + sep
   in  if (is_in_text (x,y) (pb1,0,0)) then
          label_at (x,y) pb1 handle Label_at => label
       else if (is_in_text (x,y) (pb2,x2,0)) then
          label_at (x - x2,y) pb2 handle Label_at => label
       else raise Label_at
   end
  | label_at (x,y) (C_box ((_,_,(x2,y2),pb1,pb2),label)) =
   if (is_in_text (x,y) (pb1,0,0)) then
      label_at (x,y) pb1 handle Label_at => label
   else if (is_in_text (x,y) (pb2,x2,y2)) then
      label_at (x - x2,y - y2) pb2 handle Label_at => label
   else raise Label_at;

(*--------------------------------------------------------------------------*)
(* all_labels                                                               *)
(*    : ('a print_box * int * int) -> ((int * int) * (int * int) * 'a) list *)
(*                                                                          *)
(* Function to obtain all the labels in a print_box, together with the      *)
(* (x,y) coordinates of the top-left and bottom-right of the subbox that is *)
(* associated with each label. The integers in the argument are the x and y *)
(* coordinates of the origin of the print_box.                              *)
(*--------------------------------------------------------------------------*)

fun all_labels (N_box,_,_) = []
  | all_labels (A_box ((w,_),label),x,y) = [((x,y),(x + w,y + one_char),label)]
  | all_labels (L_box ((w,sep,pb1,pb2),label),x,y) =
   ((x,y),(x + w,y + one_char),label) ::
   (all_labels (pb1,x,y)) @
   (all_labels (pb2,x + print_box_width pb1 + sep,y))
  | all_labels (C_box (((io,_,fo),h,(x2,y2),pb1,pb2),label),x,y) =
   ((x,y),(x + fo - io,y + h),label) ::
   (all_labels (pb1,x,y)) @
   (all_labels (pb2,x + x2,y + y2));

(*--------------------------------------------------------------------------*)
(* leaf_nodes_p : ('a print_box * int * int) ->                             *)
(*                (string * (int * int) * (int * int) * 'a) list            *)
(*                                                                          *)
(* Function to obtain all the atomic boxes in a print_box with the (x,y)    *)
(* coordinates of the top-left and bottom-right corners. The integers in    *)
(* the argument are the x and y coordinates of the origin of the print_box. *)
(*--------------------------------------------------------------------------*)

fun leaf_nodes_p (N_box,_,_) = []
  | leaf_nodes_p (A_box ((w,s),label),x,y) =
   [(s,(x,y),(x + w,y + one_char),label)]
  | leaf_nodes_p (L_box ((_,sep,pb1,pb2),_),x,y) =
   (leaf_nodes_p (pb1,x,y)) @
   (leaf_nodes_p (pb2,x + print_box_width pb1 + sep,y))
  | leaf_nodes_p (C_box ((_,_,(x2,y2),pb1,pb2),_),x,y) =
   (leaf_nodes_p (pb1,x,y)) @
   (leaf_nodes_p (pb2,x + x2,y + y2));

(*--------------------------------------------------------------------------*)
(* leaf_nodes_c : ('a print_box * int * int) ->                             *)
(*                (string * (int * int) * 'a) list                          *)
(*                                                                          *)
(* Function to obtain all the atomic boxes in a print_box with the          *)
(* horizontal character position (column number) and line number of the     *)
(* first character in the box. The integers in the argument are the         *)
(* horizontal character position and line number of the origin of the       *)
(* print_box. The atomic boxes are listed in the order of a raster scan.    *)
(*--------------------------------------------------------------------------*)

fun leaf_nodes_c (N_box,_,_) = []
  | leaf_nodes_c (A_box ((_,s),label),c,l) = [(s,(c,l),label)]
  | leaf_nodes_c (L_box ((_,sep,pb1,pb2),_),c,l) =
   (leaf_nodes_c (pb1,c,l)) @
   (leaf_nodes_c (pb2,c + num_of_chars (print_box_width pb1 + sep),l))
  | leaf_nodes_c (C_box ((_,_,(x2,y2),pb1,pb2),_),c,l) =
   (leaf_nodes_c (pb1,c,l)) @
   (leaf_nodes_c (pb2,c + num_of_chars x2,l + num_of_chars y2));

in

val is_in_text = fn (x,y) => fn ((pb,_),x',y') => is_in_text (x,y) (pb,x,y)
and label_at = fn (x,y) => fn (pb,_) => label_at (x,y) pb
and all_labels = fn ((pb,_),x,y) => all_labels (pb,x,y)
and leaf_nodes_p = fn ((pb,_),x,y) => leaf_nodes_p (pb,x,y)
and leaf_nodes_c = fn ((pb,_),x,y) => leaf_nodes_c (pb,x,y);

end;

end; (* LabelPPBoxesFun *)
