(****************************************************************************)
(*                                                                          *)
(*               Copyright 1994, 1996 University of Cambridge               *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : pretty.support.sml                                       *)
(* DESCRIPTION   : Declarations used by compiled PP language code.          *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 11th February 1994                                       *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 22nd May 1996                                            *)
(****************************************************************************)

structure PrettySupport =
struct

local
   open Portable
in

exception StringTable;

type 'a string_table =
   {text_and_width : string * 'a -> (string * int) * 'a,
    attrib : string -> string -> string,
    is : string -> string -> bool};

exception PPpattern;

fun string_of_int i =
   let fun digits 0 = ""
         | digits i = digits (i div 10) ^ chr (ord("0") + (i mod 10))
   in  if (i = 0)
       then "0"
       else if (i < 0)
            then "~" ^ digits (~i)
            else digits i
   end;

exception IntOfString;
fun int_of_string s =
   let val zero = ord("0")
       fun digits result [] = result
         | digits result (c::cs) =
          if ("0" <= c) andalso (c <= "9")
          then digits (ord(c) - zero + 10 * result) cs
          else raise IntOfString
   in  case (explode s)
       of ["~"] => raise IntOfString
        | [] => raise IntOfString
        | "~"::cs => ~(digits 0 cs)
        | cs => digits 0 cs
   end;

datatype precedence = MINPREC | PREC of int | MAXPREC;

fun prec_test (_,MAXPREC) = false
  | prec_test (_,MINPREC) = true
  | prec_test (MAXPREC,PREC _) = true
  | prec_test (MINPREC,PREC _) = false
  | prec_test (PREC outer,PREC inner) = inner <= outer;

fun upto from to =
   if (from > to)
   then []
   else from::(upto (from + 1) to);

exception Max;

fun max [] = raise Max
  | max (x::xs) =
   let fun max' m [] = (m:int)
         | max' m (x::xs) = max' (if (x > m) then x else m) xs
   in  max' x xs
   end;

fun el default f [] _ = default
  | el default f (x::xs) n =
   if (n = 1)
   then f x
   else el default f xs (n - 1);

val string_el = el "" (fn x => x);

end;

end; (* PrettySupport *)


structure DefaultStringTable =
struct

val string_table : 'a PrettySupport.string_table =
   {text_and_width = fn (s,label) => ((s,1000 * size s),label),
    attrib = fn s => fn att => raise PrettySupport.StringTable,
    is = fn s => fn att => raise PrettySupport.StringTable};

end; (* DefaultStringTable *)
