(****************************************************************************)
(*                                                                          *)
(*            Copyright 1994, 1995, 1996 University of Cambridge            *)
(*                  Copyright 1998 University of Edinburgh                  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : hol_type_support.sml                                     *)
(* DESCRIPTION   : Support code for HOL type definitions generated from Syn.*)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 24th May 1994                                            *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 19th February 1998                                       *)
(****************************************************************************)

fun mk_option {opt = NONE,ty} =
   mk_const {Name = "NONE",Ty = mk_type {Args = [ty],Tyop = "option"}}
  | mk_option {opt = SOME tm,ty} =
   let val const =
          mk_const
             {Name = "SOME",
              Ty = mk_type {Args = [ty,mk_type {Args = [ty],Tyop = "option"}],
                            Tyop = "fun"}}
   in  mk_comb {Rator = const,Rand = tm}
   end
   handle _ =>
   raise HOL_ERR
            {origin_structure = "",origin_function = "mk_option",message = ""};

fun dest_option tm =
   let val {Rator,Rand} = dest_comb tm
   in  if (#Name (dest_const Rator) = "SOME")
       then {opt = SOME Rand,ty = type_of Rand}
       else raise Fail ""
   end
   handle _ =>
   let val {Name,Ty} = dest_const tm
   in  if (Name = "NONE")
       then {opt = NONE,ty = hd (#Args (dest_type Ty))}
       else raise Fail ""
   end
   handle _ =>
   raise HOL_ERR {origin_structure = "",
                  origin_function = "dest_option",
                  message = ""};

structure SimpleASTtoHOL =
struct

fun term_for_string s =
   mk_const {Name = "\"" ^ s ^ "\"",Ty = mk_type {Args = [],Tyop = "string"}};

fun term_for_boolean b =
   mk_const {Name = if b then "T" else "F",
             Ty = mk_type {Args = [],Tyop = "bool"}};

fun term_for_natural n =
   mk_const {Name = Portable.string_of_int (Natural.int_of_nat n),
             Ty = mk_type {Args = [],Tyop = "num"}};

fun term_for_integer i =
   let val num = mk_type {Args = [],Tyop = "num"}
       val int = mk_type {Args = [],Tyop = "integer"}
       val numintfun = mk_type {Args = [num,int],Tyop = "fun"}
       val intintfun = mk_type {Args = [int,int],Tyop = "fun"}
       val n = Portable.string_of_int (if (i < 0) then (~i) else i)
       val tm = mk_comb {Rator = mk_const {Name = "INT",Ty = numintfun},
                         Rand = mk_const {Name = n,Ty = num}}
   in  if (i < 0)
       then mk_comb {Rator = mk_const {Name = "neg",Ty = intintfun},Rand = tm}
       else tm
   end;

fun term_for_rational n =
   raise HOL_ERR {origin_structure = "ASTtoHOL",
                  origin_function = "term_for_rational",
                  message = "type `rational' not currently supported"};

fun term_for_real r =
   raise HOL_ERR {origin_structure = "ASTtoHOL",
                  origin_function = "term_for_real",
                  message = "type `real' not currently supported"};

end; (* SimpleASTtoHOL *)

structure HOLtoSimpleAST =
struct

fun ML_for_string tm =
   let val {Name,Ty} = dest_const tm
   in  if (#Tyop (dest_type Ty) = "string")
       then substring (Name,1,size Name - 2)
       else raise Fail ""
   end
   handle _ => raise HOL_ERR {origin_structure = "HOLtoAST",
                              origin_function = "ML_for_string",
                              message = "term is not a string constant"};

fun ML_for_boolean tm =
   let val {Name,Ty} = dest_const tm
   in  case (Name,#Tyop (dest_type Ty))
       of ("T","bool") => true
        | ("F","bool") => false
        | _ => raise Fail ""
   end
   handle _ => raise HOL_ERR {origin_structure = "HOLtoAST",
                              origin_function = "ML_for_boolean",
                              message = "term is not a boolean constant"};

fun ML_for_natural tm =
   let val {Name,Ty} = dest_const tm
   in  if (#Tyop (dest_type Ty) = "num")
       then Natural.nat_of_int (Portable.int_of_string Name)
       else raise Fail ""
   end
   handle _ => raise HOL_ERR {origin_structure = "HOLtoAST",
                              origin_function = "ML_for_natural",
                              message = "term is not a natural constant"};

fun ML_for_integer tm : int =
   let val RR as {Rator,Rand} = dest_comb tm
       val {Name,Ty = _} = dest_const Rator
       val (negative,{Rator,Rand}) =
          if (Name = "neg") then (true,dest_comb Rand) else (false,RR)
       val {Name,Ty = _} = dest_const Rator
       and {Name = n,Ty = ty} = dest_const Rand
   in  if (Name = "INT") andalso (#Tyop (dest_type ty) = "num")
       then (if negative then ~1 else 1) * Portable.int_of_string n
       else raise Fail ""
   end
   handle _ => raise HOL_ERR {origin_structure = "HOLtoAST",
                              origin_function = "ML_for_integer",
                              message = "term is not an integer constant"};

fun ML_for_rational tm =
   raise HOL_ERR {origin_structure = "HOLtoAST",
                  origin_function = "ML_for_rational",
                  message = "type `rational' not currently supported"};

fun ML_for_real tm =
   raise HOL_ERR {origin_structure = "HOLtoAST",
                  origin_function = "ML_for_real",
                  message = "type `real' not currently supported"};

end; (* HOLtoSimpleAST *)

structure ASTtoHOL = SimpleASTtoHOL;
structure HOLtoAST = HOLtoSimpleAST;
