(****************************************************************************)
(*                                                                          *)
(*            Copyright 1994, 1995, 1996 University of Cambridge            *)
(*                  Copyright 1998 University of Edinburgh                  *)
(*                                                                          *)
(*                           All rights reserved.                           *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(* FILE          : hol.pp.support.sml                                       *)
(* DESCRIPTION   : Support code for HOL pretty-printer.                     *)
(*                                                                          *)
(* AUTHOR        : R.J.Boulton                                              *)
(* DATE          : 4th May 1994                                             *)
(*                                                                          *)
(* LAST MODIFIED : R.J.Boulton                                              *)
(* DATE          : 19th February 1998                                       *)
(****************************************************************************)

structure HOLPPConstructors = struct end;

structure HOLPPDestructors =
struct

exception HOLDestruct;

datatype branch =
   HYP of int | CONCL | RATOR | RAND | BODY | (* TY | *) TYARG of int;

local
   fun zip ([],[]) = []
     | zip (x::xs,y::ys) = (x,y)::(zip (xs,ys))
     | zip _ = raise Fail "zip";
   fun upto from to =
      if (from > to)
      then []
      else from::(upto (from + 1) to);
in

fun Vartype (ty,_) = dest_vartype ty
and Type (ty,add) =
   case dest_type ty
   of {Tyop,Args} => (Tyop,map (fn (arg,i) => (arg,(TYARG i)::add))
                               (zip (Args,upto 1 (length Args))));

fun Var (tm,add) =
   case dest_var tm of {Name,Ty} =>
   (Name,if !Globals.show_types then SOME (Ty,(* TY:: *)add) else NONE)
and Const (tm,add) =
   case dest_const tm of {Name,Ty} =>
   (Name,if !Globals.show_types then SOME (Ty,(* TY:: *)add) else NONE)
and Comb (tm,add) =
   case dest_comb tm of {Rator,Rand} => ((Rator,RATOR::add),(Rand,RAND::add))
and Abs (tm,add) =
   case dest_abs tm of {Bvar,Body} => ((Bvar,add),(Body,BODY::add));

fun Pabs (tm,add) =
   (case dest_pabs tm
    of {varstruct,body} => ((varstruct,add),(body,BODY::add)))
   handle _ => Abs (tm,add);

fun Thm (th,add) =
   let val (hs,c) = dest_thm th
   in  (map (fn (h,i) => (h,(HYP i)::add)) (zip (hs,upto 1 (length hs))),
        (c,CONCL::add))
   end;

fun Goal (gl:goal,add) =
   let val (hs,c) = gl
   in  (map (fn (h,i) => (h,(HYP i)::add)) (zip (hs,upto 1 (length hs))),
        (c,CONCL::add))
   end;

local

fun OPTION_NONE tmadd =
   case (Const tmadd)
   of ("NONE",_) => ()
    | _ => raise HOLDestruct;

fun OPTION_SOME tmadd =
   let val (f,x) = Comb tmadd
   in  case (Const f)
       of ("SOME",_) => x
        | _ => raise HOLDestruct
   end;

in

fun OPTION tmadd =
   SOME (OPTION_SOME tmadd)
   handle _ =>
   let val _ = OPTION_NONE tmadd in NONE end;

end;

local

fun LIST_NIL tmadd =
   case (Const tmadd)
   of ("NIL",_) => ()
    | _ => raise HOLDestruct;

fun LIST_CONS tmadd =
   let val (f1,xs) = Comb tmadd
       val (f0,x) = Comb f1
   in  case (Const f0)
       of ("CONS",_) => (x,xs)
        | _ => raise HOLDestruct
   end;

in

fun LIST tmadd =
   let val (x,xs) = LIST_CONS tmadd in x :: LIST xs end
   handle _ =>
   let val _ = LIST_NIL tmadd in [] end;

end;

fun String (tm,add) = case (HOLtoAST.ML_for_string tm) of s => s;

exception Option;
fun None NONE = ()
  | None _ = raise Option
and Some (SOME tmadd) = tmadd
  | Some _ = raise Option;

end;

end;

structure HOLPPExternals =
struct

val is_infix = fn name => is_constant name andalso is_infix name
and is_binder = is_binder
and is_res_quan = fn name => not (assoc2 name (binder_restrictions()) = NONE);

fun restriction_name name =
   case (assoc2 name (binder_restrictions()))
   of NONE => name
    | SOME (binder,_) => binder;

val pad_binder =
   let val symbols = explode "#?+*/\\=<>&%@!,:;_|~-"
       val symbolic = all (fn c => mem c symbols) o explode
   in  fn s => if (symbolic s) then s else (s ^ " ")
   end;

val fixity = fn s => case (fixity s) of Infix prec => prec | _ => 0;

fun show_types () = !Globals.show_types
and show_assums () = !Globals.show_assums
and show_restrictions () = !Globals.show_restrict;

fun replicate (s,0) = ""
  | replicate (s,n) = s ^ replicate (s,n - 1);

fun common_frees ((varstruct,_),(t1,_),(t2,_)) =
   set_eq (strip_pair varstruct) (intersect (free_vars t1) (free_vars t2));

fun goal_line () = !Globals.goal_line;

end;

structure HOLLexSupport =
struct

fun string_of_string (tm,_) = HOLtoAST.ML_for_string tm;

(*
fun string_of_character (tm,_) =
   LexSupport.string_of_character (HOLtoAST.ML_for_character tm);
*)

fun string_of_boolean (tm,_) =
   LexSupport.string_of_boolean (HOLtoAST.ML_for_boolean tm);

fun string_of_natural (tm,_) =
   LexSupport.string_of_natural (HOLtoAST.ML_for_natural tm);

fun string_of_integer (tm,_) =
   LexSupport.string_of_integer (HOLtoAST.ML_for_integer tm);

fun string_of_rational (tm,_) =
   LexSupport.string_of_rational (HOLtoAST.ML_for_rational tm);

fun string_of_real (tm,_) =
   LexSupport.string_of_real (HOLtoAST.ML_for_real tm);

end;
