structure TerminalStringTable =
struct

local
   fun text_and_width (s,label) = ((s,1000 * size s),label)
   fun attrib _ = (fn _ => raise PrettySupport.StringTable)
   fun is _ = (fn _ => raise PrettySupport.StringTable)
in

val string_table:'a PrettySupport.string_table =
   {text_and_width = text_and_width,attrib = attrib,is = is};

end;

end;

structure HOLPP =
struct

structure Const = HOLPPConstructors
and Dest = HOLPPDestructors
and Ext = HOLPPExternals;

structure PP =
struct

structure LabelPPBoxes = LabelPPBoxesFun (struct val char_size = 1000; end);

open LabelPPBoxes;

fun pp (st:'a PrettySupport.string_table) label s =
   let val (text_and_width,label') = #text_and_width st (s,label)
   in pp_string_box label' text_and_width
   end;

end;

structure Local =
struct

open PrettySupport;

fun precedence ":prod" = 300
  | precedence ":sum" = 200
  | precedence ":fun" = 100
  | precedence "#appl" = 2000
  | precedence "COND" = 300
  | precedence "LET" = 200
  | precedence "#binder" = 100
  | precedence ":" = 0
  | precedence s' = if Ext.is_infix s' then Ext.fixity s' + 1000 else 0;

fun attrib (st:'a PrettySupport.string_table) s att =
   let fun attrib s' = if Ext.is_infix s' then (fn _ => "") else (fn _ => "")
   in #attrib st s att handle PrettySupport.StringTable => attrib s att
   end;

fun is (st:'a PrettySupport.string_table) s att =
   let fun is ":prod" = (fn "infix_tycon" => true | _ => false)
         | is ":sum" = (fn "infix_tycon" => true | _ => false)
         | is ":fun" = (fn "infix_tycon" => true | _ => false)
         | is s' =
          if Ext.is_infix s' then (fn _ => false) else (fn _ => false)
   in #is st s att handle PrettySupport.StringTable => is s att
   end;

fun box_el f l = el PP.pp_empty_box f l;

fun lower_prec s = PREC (2 * precedence s - 1);

fun upper_prec s = PREC (2 * precedence s);

fun lpar st label prec s =
   if prec_test (prec,upper_prec s)
   then PP.pp st label "("
   else PP.pp_empty_box;

fun rpar st label prec s =
   if prec_test (prec,upper_prec s)
   then PP.pp st label ")"
   else PP.pp_empty_box;

end;

val initial_precedence = Local.MINPREC;

val error = "";

val elision = " ... ";

val default_h = 0
and default_v = (PP.ABSOLUTE 0,0)
and default_hv = (0,PP.ABSOLUTE 0,0)
and default_hov = (0,PP.ABSOLUTE 0,0);

val initial_depth = ~1;

val initial_params = {context = ""};

structure D =
struct

open Dest;

val UNOP =
   fn tr =>
      let val (tr1,tr2) = Comb tr
          val (tr11,tr12) = Const tr1
          val op' = tr11
          val arg' = tr2
      in (op',arg')
      end;

val BINOP =
   fn tr =>
      let val (tr1,tr2) = Comb tr
          val (tr11,tr12) = Comb tr1
          val (tr111,tr112) = Const tr11
          val op' = tr111
          val arg1' = tr12
          val arg2' = tr2
      in (op',arg1',arg2')
      end;

val TUPLE =
   fn tr =>
      let fun loop_1 n result =
             let val tr = #tr result
                 val (tr1,tr2,tr3) = BINOP tr
                 val _ = if tr1 = "," then () else raise Local.PPpattern
                 val comps' = tr2
                 val link = tr3
             in loop_1 (n + 1) {comps' = #comps' result @ [comps'],tr = link}
             end
             handle e => if n >= 1 then result else raise e
          val {comps',tr = tr1} = loop_1 0 {comps' = [],tr = tr}
          val comps' = comps' @ [tr1]
      in comps'
      end;

val BINDER =
   fn tr =>
      let val (tr1,tr2) = Comb tr
          val (tr11,tr12) = Const tr1
          val op' = tr11
          val (tr21,tr22) = Pabs tr2
          val varstruct' = tr21
          val body' = tr22
          val _ = if Ext.is_binder op' then () else raise Local.PPpattern
      in (op',varstruct',body')
      end;

val RESQUAN =
   fn tr =>
      let val (tr1,tr2) = Comb tr
          val (tr11,tr12) = Comb tr1
          val (tr111,tr112) = Const tr11
          val op' = tr111
          val pred' = tr12
          val (tr21,tr22) = Pabs tr2
          val varstruct' = tr21
          val body' = tr22
          val _ = if Ext.is_res_quan op' then () else raise Local.PPpattern
      in (op',pred',varstruct',body')
      end;

val DEC =
   fn tr =>
      let fun loop_1 n result =
             let val tr = #tr result
                 val (tr1,tr2) = Pabs tr
                 val link = tr2
             in loop_1 (n + 1) {tr = link}
             end
             handle _ => result
          val {tr = tr1} = loop_1 0 {tr = tr}
          val letbody' = tr1
          val args' = tr
      in (args',letbody')
      end;

val LETS =
   fn tr =>
      let fun loop_1 n result =
             let val tr = #tr result
                 val (tr1,tr2,tr3) = BINOP tr
                 val _ = if tr1 = "LET" then () else raise Local.PPpattern
                 val link = tr2
                 val (tr31,tr32) = DEC tr3
                 val argsl' = tr31
                 val letbodyl' = tr32
             in loop_1 (n + 1)
                   {argsl' = #argsl' result @ [argsl'],
                    letbodyl' = #letbodyl' result @ [letbodyl'],tr = link}
             end
             handle e => if n >= 1 then result else raise e
          val {argsl',letbodyl',tr = tr1} =
             loop_1 0 {argsl' = [],letbodyl' = [],tr = tr}
          fun loop_2 n result =
             if n >= length argsl'
             then result
             else let val tr = #tr result
                      val (tr1,tr2) = Pabs tr
                      val bvl' = tr1
                      val link = tr2
                  in loop_2 (n + 1) {bvl' = #bvl' result @ [bvl'],tr = link}
                  end
                  handle e => if n >= length argsl' then result else raise e
          val {bvl',tr = tr11} = loop_2 0 {bvl' = [],tr = tr1}
          val body' = tr11
      in (bvl',rev argsl',rev letbodyl',body')
      end;

val CONS =
   fn tr =>
      let val (tr1,tr2) = Comb tr
          val (tr11,tr12) = Comb tr1
          val (tr111,tr112) = Const tr11
          val _ = if tr111 = "CONS" then () else raise Local.PPpattern
          val x' = tr12
          val xs' = tr2
      in (x',xs')
      end;

val LIST =
   fn tr =>
      let fun loop_1 n result =
             let val tr = #tr result
                 val (tr1,tr2) = CONS tr
                 val elems' = tr1
                 val link = tr2
             in loop_1 (n + 1) {elems' = #elems' result @ [elems'],tr = link}
             end
             handle _ => result
          val {elems',tr = tr1} = loop_1 0 {elems' = [],tr = tr}
          val (tr11,tr12) = Const tr1
          val _ = if tr11 = "NIL" then () else raise Local.PPpattern
      in elems'
      end;

val INSERT =
   fn tr =>
      let val (tr1,tr2) = Comb tr
          val (tr11,tr12) = Comb tr1
          val (tr111,tr112) = Const tr11
          val _ = if tr111 = "INSERT" then () else raise Local.PPpattern
          val x' = tr12
          val xs' = tr2
      in (x',xs')
      end;

val SET =
   fn tr =>
      let fun loop_1 n result =
             let val tr = #tr result
                 val (tr1,tr2) = INSERT tr
                 val elems' = tr1
                 val link = tr2
             in loop_1 (n + 1) {elems' = #elems' result @ [elems'],tr = link}
             end
             handle _ => result
          val {elems',tr = tr1} = loop_1 0 {elems' = [],tr = tr}
          val (tr11,tr12) = Const tr1
          val _ = if tr11 = "EMPTY" then () else raise Local.PPpattern
      in elems'
      end;

open Dest;

end;

local
   open Portable Const
in

fun pp_type st {outer_prec,depth} (params as {context = context'})
       (tr as (_,label)) =
   (fn (pp_type,pp_term) =>
       if depth = 0
       then PP.pp st label elision
       else let val tr1 = D.Vartype tr
                val op' = tr1
            in PP.pp st label op'
            end
            handle _ =>
            let val (tr1,tr2) = D.Type tr
                val op' = tr1
                val tr21 = hd tr2
                and tr22 = hd (tl tr2)
                val _ =
                   if null (tl (tl tr2)) then () else raise Local.PPpattern
                val type1' = tr21
                val type2' = tr22
                val _ =
                   if Local.is st (":" ^ op') "infix_tycon"
                   then ()
                   else raise Local.PPpattern
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec (":" ^ op'),
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      [PP.pp_h_box label 1000
                          [pp_type st
                              {outer_prec = Local.upper_prec (":" ^ op'),
                               depth = depth - 1} params type1',
                           if op' = "fun"
                           then PP.pp st label "->"
                           else if op' = "prod"
                                then PP.pp st label "#"
                                else if op' = "sum"
                                     then PP.pp st label "+"
                                     else PP.pp st label op'],
                       pp_type st
                          {outer_prec = Local.upper_prec (":" ^ op'),
                           depth = depth - 1} params type2'],
                   Local.rpar st label outer_prec (":" ^ op')]
            end
            handle _ =>
            let val (tr1,tr2) = D.Type tr
                val op' = tr1
                val _ = if null tr2 then () else raise Local.PPpattern
            in PP.pp st label op'
            end
            handle _ =>
            let val (tr1,tr2) = D.Type tr
                val op' = tr1
                val tr2rev = rev tr2
                val tr21 = rev (tl tr2rev)
                and tr22 = hd tr2rev
                val types' = tr21
                val type' = tr22
            in PP.pp_hv_box label (0,PP.ABSOLUTE 3000,0)
                  [PP.pp_h_box label default_h
                      [PP.pp st label "(",
                       PP.pp_hv_box label (0,PP.RELATIVE 3000,0)
                          (map
                              (fn el =>
                                  PP.pp_h_box label 0
                                     [Local.box_el
                                         (pp_type st
                                             {outer_prec = Local.MINPREC,
                                              depth = depth - 1} params)
                                         types' el,PP.pp st label ","])
                              (Local.upto 1 (length types')) @
                           [pp_type st
                               {outer_prec = Local.MINPREC,depth = depth - 1}
                               params type']),PP.pp st label ")"],
                   PP.pp st label op']
            end
            handle _ => raise Local.PPpattern)
and pp_term st {outer_prec,depth} (params as {context = context'})
       (tr as (_,label)) =
   (fn (pp_type,pp_term) =>
       if depth = 0
       then PP.pp st label elision
       else let fun loop_1 n result =
                   let val tr = #tr result
                       val (tr1,tr2) = D.Pabs tr
                       val args' = tr1
                       val link = tr2
                   in loop_1 (n + 1)
                         {args' = #args' result @ [args'],tr = link}
                   end
                   handle _ => result
                val {args',tr = tr1} = loop_1 0 {args' = [],tr = tr}
                val _ =
                   if context' = "in_let_args"
                   then ()
                   else raise Local.PPpattern
            in PP.pp_h_box label 1000
                  ([PP.pp_empty_box] @
                   map
                      (fn el =>
                          Local.box_el
                             (pp_term st
                                 {outer_prec = Local.MINPREC,
                                  depth = depth - 1} {context = ""}) args' el)
                      (Local.upto 1 (length args')))
            end
            handle _ =>
            let val (tr1,tr2) = D.Var tr
                val var' = tr1
                val _ = D.None tr2
            in PP.pp st label var'
            end
            handle _ =>
            let val (tr1,tr2) = D.Var tr
                val var' = tr1
                val tr21 = D.Some tr2
                val type' = tr21
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec ":",
                   PP.pp_hv_box label (0,PP.ABSOLUTE 0,0)
                      [PP.pp st label var',
                       PP.pp_h_box label 0
                          [PP.pp st label ":",
                           pp_type st
                              {outer_prec = Local.MINPREC,depth = depth - 1}
                              params type']],
                   Local.rpar st label outer_prec ":"]
            end
            handle _ =>
            let val (tr1,tr2) = D.Const tr
                val _ = if tr1 = "NIL" then () else raise Local.PPpattern
                val _ = D.None tr2
            in PP.pp st label "[]"
            end
            handle _ =>
            let val (tr1,tr2) = D.Const tr
                val _ = if tr1 = "NIL" then () else raise Local.PPpattern
                val tr21 = D.Some tr2
                val type' = tr21
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec ":",
                   PP.pp_hv_box label (0,PP.ABSOLUTE 0,0)
                      [PP.pp st label "[]",
                       PP.pp_h_box label 0
                          [PP.pp st label ":",
                           pp_type st
                              {outer_prec = Local.MINPREC,depth = depth - 1}
                              params type']],
                   Local.rpar st label outer_prec ":"]
            end
            handle _ =>
            let val (tr1,tr2) = D.Const tr
                val _ = if tr1 = "EMPTY" then () else raise Local.PPpattern
                val _ = D.None tr2
            in PP.pp st label "{}"
            end
            handle _ =>
            let val (tr1,tr2) = D.Const tr
                val _ = if tr1 = "EMPTY" then () else raise Local.PPpattern
                val tr21 = D.Some tr2
                val type' = tr21
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec ":",
                   PP.pp_hv_box label (0,PP.ABSOLUTE 0,0)
                      [PP.pp st label "{}",
                       PP.pp_h_box label 0
                          [PP.pp st label ":",
                           pp_type st
                              {outer_prec = Local.MINPREC,depth = depth - 1}
                              params type']],
                   Local.rpar st label outer_prec ":"]
            end
            handle _ =>
            let val (tr1,tr2) = D.Const tr
                val const' = tr1
                val _ = D.None tr2
            in if Ext.is_infix const' orelse Ext.is_binder const' orelse
                  const' = "~"
               then PP.pp_h_box label default_h
                       [PP.pp st label "$",PP.pp st label const']
               else PP.pp st label const'
            end
            handle _ =>
            let val (tr1,tr2) = D.Const tr
                val const' = tr1
                val tr21 = D.Some tr2
                val type' = tr21
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec ":",
                   PP.pp_hv_box label (0,PP.ABSOLUTE 0,0)
                      [if Ext.is_infix const' orelse Ext.is_binder const'
                          orelse const' = "~"
                       then PP.pp_h_box label default_h
                               [PP.pp st label "$",PP.pp st label const']
                       else PP.pp st label const',
                       PP.pp_h_box label 0
                          [PP.pp st label ":",
                           pp_type st
                              {outer_prec = Local.MINPREC,depth = depth - 1}
                              params type']],
                   Local.rpar st label outer_prec ":"]
            end
            handle _ =>
            let val tr1 = D.TUPLE tr
                val tr1rev = rev tr1
                val tr11 = rev (tl tr1rev)
                and tr12 = hd tr1rev
                val comps' = tr11
                val comp' = tr12
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec ",",
                   PP.pp_hv_box label (0,PP.ABSOLUTE 0,0)
                      (map
                          (fn el =>
                              PP.pp_h_box label 0
                                 [Local.box_el
                                     (pp_term st
                                         {outer_prec = Local.upper_prec ",",
                                          depth = depth - 1} params) comps'
                                     el,PP.pp st label ","])
                          (Local.upto 1 (length comps')) @
                       [pp_term st
                           {outer_prec = Local.upper_prec ",",
                            depth = depth - 1} params comp']),
                   Local.rpar st label outer_prec ","]
            end
            handle _ =>
            let val tr1 = D.LIST tr
                val tr1rev = rev tr1
                val tr11 = rev (tl tr1rev)
                and tr12 = hd tr1rev
                val elems' = tr11
                val elem' = tr12
            in PP.pp_h_box label default_h
                  [PP.pp st label "[",
                   PP.pp_hov_box label (0,PP.ABSOLUTE 0,0)
                      (map
                          (fn el =>
                              PP.pp_h_box label 0
                                 [Local.box_el
                                     (pp_term st
                                         {outer_prec = Local.MINPREC,
                                          depth = depth - 1} params) elems'
                                     el,PP.pp st label ";"])
                          (Local.upto 1 (length elems')) @
                       [pp_term st
                           {outer_prec = Local.MINPREC,depth = depth - 1}
                           params elem']),PP.pp st label "]"]
            end
            handle _ =>
            let val tr1 = D.SET tr
                val tr1rev = rev tr1
                val tr11 = rev (tl tr1rev)
                and tr12 = hd tr1rev
                val elems' = tr11
                val elem' = tr12
            in PP.pp_h_box label default_h
                  [PP.pp st label "{",
                   PP.pp_hov_box label (0,PP.ABSOLUTE 0,0)
                      (map
                          (fn el =>
                              PP.pp_h_box label 0
                                 [Local.box_el
                                     (pp_term st
                                         {outer_prec = Local.MINPREC,
                                          depth = depth - 1} params) elems'
                                     el,PP.pp st label ";"])
                          (Local.upto 1 (length elems')) @
                       [pp_term st
                           {outer_prec = Local.MINPREC,depth = depth - 1}
                           params elem']),PP.pp st label "}"]
            end
            handle _ =>
            let val (tr1,tr2) = D.Comb tr
                val (tr11,tr12) = D.Const tr1
                val _ = if tr11 = "GSPEC" then () else raise Local.PPpattern
                val (tr21,tr22) = D.Pabs tr2
                val varstruct' = tr21
                val (tr221,tr222,tr223) = D.BINOP tr22
                val _ = if tr221 = "," then () else raise Local.PPpattern
                val elem' = tr222
                val spec' = tr223
                val _ =
                   if Ext.common_frees (varstruct',elem',spec')
                   then ()
                   else raise Local.PPpattern
            in PP.pp_h_box label default_h
                  [PP.pp st label "{",
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      [PP.pp_h_box label 1000
                          [pp_term st
                              {outer_prec = Local.MINPREC,depth = depth - 1}
                              params elem',PP.pp st label "|"],
                       pp_term st
                          {outer_prec = Local.MINPREC,depth = depth - 1}
                          params spec'],PP.pp st label "}"]
            end
            handle _ =>
            let fun loop_1 n result =
                   let val tr = #tr result
                       val (tr1,tr2,tr3) = D.BINOP tr
                       val op' = tr1
                       val args' = tr2
                       val link = tr3
                   in if (case #op' result
                          of NONE => true | SOME x => op' = x)
                      then loop_1 (n + 1)
                              {op' = SOME op',args' = #args' result @ [args'],
                               tr = link}
                      else raise Local.PPpattern
                   end
                   handle e => if n >= 1 then result else raise e
                val (op',args',tr1) =
                   case loop_1 0 {op' = NONE,args' = [],tr = tr}
                   of {op' = SOME op',args',tr = tr1} => (op',args',tr1)
                    | _ => raise Local.PPpattern
                val arg' = tr1
                val _ = if Ext.is_infix op' then () else raise Local.PPpattern
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec op',
                   PP.pp_hov_box label (1000,PP.ABSOLUTE 0,0)
                      (map
                          (fn el =>
                              PP.pp_hv_box label (1000,PP.ABSOLUTE 0,0)
                                 [Local.box_el
                                     (pp_term st
                                         {outer_prec = Local.upper_prec op',
                                          depth = depth - 1} params) args' el,
                                  PP.pp st label op'])
                          (Local.upto 1 (length args')) @
                       [pp_term st
                           {outer_prec = Local.upper_prec op',
                            depth = depth - 1} params arg']),
                   Local.rpar st label outer_prec op']
            end
            handle _ =>
            let val (tr1,tr2,tr3) = D.BINOP tr
                val op' = tr1
                val arg1' = tr2
                val arg2' = tr3
                val _ = if Ext.is_infix op' then () else raise Local.PPpattern
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec op',
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      [PP.pp_h_box label 1000
                          [pp_term st
                              {outer_prec = Local.upper_prec op',
                               depth = depth - 1} params arg1',
                           PP.pp st label op'],
                       pp_term st
                          {outer_prec = Local.upper_prec op',
                           depth = depth - 1} params arg2'],
                   Local.rpar st label outer_prec op']
            end
            handle _ =>
            let val (tr1,tr2) = D.UNOP tr
                val _ = if tr1 = "~" then () else raise Local.PPpattern
                val arg' = tr2
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "~",PP.pp st label "~",
                   pp_term st
                      {outer_prec = Local.upper_prec "~",depth = depth - 1}
                      params arg',Local.rpar st label outer_prec "~"]
            end
            handle _ =>
            let fun loop_1 n result =
                   let val tr = #tr result
                       val (tr1,tr2,tr3) = D.BINDER tr
                       val op' = tr1
                       val bvs' = tr2
                       val link = tr3
                   in if (case #op' result
                          of NONE => true | SOME x => op' = x)
                      then loop_1 (n + 1)
                              {op' = SOME op',bvs' = #bvs' result @ [bvs'],
                               tr = link}
                      else raise Local.PPpattern
                   end
                   handle e => if n >= 1 then result else raise e
                val (op',bvs',tr1) =
                   case loop_1 0 {op' = NONE,bvs' = [],tr = tr}
                   of {op' = SOME op',bvs',tr = tr1} => (op',bvs',tr1)
                    | _ => raise Local.PPpattern
                val body' = tr1
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "#binder",
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      [PP.pp_h_box label 0
                          [PP.pp st label (Ext.pad_binder op'),
                           PP.pp_hv_box label (1000,PP.ABSOLUTE 0,0)
                              (map
                                  (fn el =>
                                      Local.box_el
                                         (pp_term st
                                             {outer_prec = Local.MAXPREC,
                                              depth = depth - 1} params) bvs'
                                         el) (Local.upto 1 (length bvs'))),
                           PP.pp st label "."],
                       pp_term st
                          {outer_prec = Local.lower_prec "#binder",
                           depth = depth - 1} params body'],
                   Local.rpar st label outer_prec "#binder"]
            end
            handle _ =>
            let fun loop_1 n result =
                   let val tr = #tr result
                       val (tr1,tr2,tr3,tr4) = D.RESQUAN tr
                       val op' = tr1
                       val pred' = tr2
                       val bvs' = tr3
                       val link = tr4
                   in if (case #pred' result
                          of NONE => true | SOME x => #1 pred' = #1 x) andalso
                         (case #op' result
                          of NONE => true | SOME x => op' = x)
                      then loop_1 (n + 1)
                              {op' = SOME op',pred' = SOME pred',
                               bvs' = #bvs' result @ [bvs'],tr = link}
                      else raise Local.PPpattern
                   end
                   handle e => if n >= 1 then result else raise e
                val (op',pred',bvs',tr1) =
                   case loop_1 0 {op' = NONE,pred' = NONE,bvs' = [],tr = tr}
                   of {op' = SOME op',pred' = SOME pred',bvs',tr = tr1} =>
                      (op',pred',bvs',tr1)
                    | _ => raise Local.PPpattern
                val body' = tr1
                val _ =
                   if Ext.show_restrictions ()
                   then ()
                   else raise Local.PPpattern
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "#binder",
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      [PP.pp_h_sepbox label
                          (PP.pp st label
                              (Ext.pad_binder (Ext.restriction_name op')))
                          [(0,
                            PP.pp_hv_box label (1000,PP.ABSOLUTE 0,0)
                               (map
                                   (fn el =>
                                       Local.box_el
                                          (pp_term st
                                              {outer_prec = Local.MAXPREC,
                                               depth = depth - 1} params) bvs'
                                          el) (Local.upto 1 (length bvs')))),
                           (1000,PP.pp st label "::")],
                       PP.pp_h_box label 0
                          [pp_term st
                              {outer_prec = Local.MAXPREC,depth = depth - 1}
                              params pred',PP.pp st label "."],
                       pp_term st
                          {outer_prec = Local.lower_prec "#binder",
                           depth = depth - 1} params body'],
                   Local.rpar st label outer_prec "#binder"]
            end
            handle _ =>
            let fun loop_1 n result =
                   let val tr = #tr result
                       val (tr1,tr2) = D.Pabs tr
                       val bvs' = tr1
                       val link = tr2
                   in loop_1 (n + 1) {bvs' = #bvs' result @ [bvs'],tr = link}
                   end
                   handle e => if n >= 1 then result else raise e
                val {bvs',tr = tr1} = loop_1 0 {bvs' = [],tr = tr}
                val body' = tr1
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "#binder",
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      [PP.pp_h_box label 0
                          [PP.pp st label "\\",
                           PP.pp_hv_box label (1000,PP.ABSOLUTE 0,0)
                              (map
                                  (fn el =>
                                      Local.box_el
                                         (pp_term st
                                             {outer_prec = Local.MAXPREC,
                                              depth = depth - 1} params) bvs'
                                         el) (Local.upto 1 (length bvs'))),
                           PP.pp st label "."],
                       pp_term st
                          {outer_prec = Local.lower_prec "#binder",
                           depth = depth - 1} params body'],
                   Local.rpar st label outer_prec "#binder"]
            end
            handle _ =>
            let val (tr1,tr2) = D.Comb tr
                val (tr11,tr12) = D.Comb tr1
                val (tr111,tr112) = D.Comb tr11
                val (tr1111,tr1112) = D.Const tr111
                val _ = if tr1111 = "COND" then () else raise Local.PPpattern
                val cond' = tr112
                val x' = tr12
                val y' = tr2
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "COND",
                   PP.pp_hov_box label (1000,PP.ABSOLUTE 0,0)
                      [PP.pp_hv_box label (1000,PP.ABSOLUTE 0,0)
                          [pp_term st
                              {outer_prec = Local.upper_prec "COND",
                               depth = depth - 1} params cond',
                           PP.pp st label "=>"],
                       PP.pp_hv_box label (1000,PP.ABSOLUTE 0,0)
                          [pp_term st
                              {outer_prec = Local.upper_prec "COND",
                               depth = depth - 1} params x',
                           PP.pp st label "|"],
                       pp_term st
                          {outer_prec = Local.upper_prec "COND",
                           depth = depth - 1} params y'],
                   Local.rpar st label outer_prec "COND"]
            end
            handle _ =>
            let val (tr1,tr2,tr3,tr4) = D.LETS tr
                val tr11 = hd tr1
                and tr12 = tl tr1
                val bv' = tr11
                val bvl' = tr12
                val tr21 = hd tr2
                and tr22 = tl tr2
                val args' = tr21
                val argsl' = tr22
                val tr31 = hd tr3
                and tr32 = tl tr3
                val letbody' = tr31
                val letbodyl' = tr32
                val body' = tr4
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "LET",
                   PP.pp_hov_box label (1000,PP.ABSOLUTE 0,0)
                      ([PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                           [PP.pp_h_sepbox label (PP.pp st label "let")
                               [(1000,
                                 pp_term st
                                    {outer_prec = Local.MAXPREC,
                                     depth = depth - 1} params bv'),
                                (0,
                                 pp_term st
                                    {outer_prec = Local.MAXPREC,
                                     depth = depth - 1}
                                    {context = "in_let_args"} args'),
                                (1000,PP.pp st label "=")],
                            pp_term st
                               {outer_prec = Local.upper_prec "LET",
                                depth = depth - 1} params letbody']] @
                       map
                          (fn el =>
                              PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                                 [PP.pp_h_sepbox label (PP.pp st label "and")
                                     [(1000,
                                       Local.box_el
                                          (pp_term st
                                              {outer_prec = Local.MAXPREC,
                                               depth = depth - 1} params) bvl'
                                          el),
                                      (0,
                                       Local.box_el
                                          (pp_term st
                                              {outer_prec = Local.MAXPREC,
                                               depth = depth - 1}
                                              {context = "in_let_args"})
                                          argsl' el),
                                      (1000,PP.pp st label "=")],
                                  Local.box_el
                                     (pp_term st
                                         {outer_prec = Local.upper_prec "LET",
                                          depth = depth - 1} params) letbodyl'
                                     el])
                          (Local.upto 1
                              (Local.max
                                  [length bvl',length argsl',
                                   length letbodyl'])) @
                       [PP.pp_h_box label 1000
                           [PP.pp st label "in",
                            pp_term st
                               {outer_prec = Local.upper_prec "LET",
                                depth = depth - 1} params body']]),
                   Local.rpar st label outer_prec "LET"]
            end
            handle _ =>
            let fun loop_1 n result =
                   let val tr = #tr result
                       val (tr1,tr2) = D.Comb tr
                       val link = tr1
                       val rands' = tr2
                   in loop_1 (n + 1)
                         {rands' = #rands' result @ [rands'],tr = link}
                   end
                   handle e => if n >= 1 then result else raise e
                val {rands',tr = tr1} = loop_1 0 {rands' = [],tr = tr}
                val rator' = tr1
            in PP.pp_h_box label default_h
                  [Local.lpar st label outer_prec "#appl",
                   PP.pp_hv_box label (1000,PP.ABSOLUTE 3000,0)
                      ([pp_term st
                           {outer_prec = Local.upper_prec "#appl",
                            depth = depth - 1} params rator'] @
                       map
                          (fn el =>
                              Local.box_el
                                 (pp_term st
                                     {outer_prec = Local.upper_prec "#appl",
                                      depth = depth - 1} params) (rev rands')
                                 el) (Local.upto 1 (length rands'))),
                   Local.rpar st label outer_prec "#appl"]
            end
            handle _ => raise Local.PPpattern)
and pp_thm st {outer_prec,depth} (params as {context = context'})
       (tr as (_,label)) =
   (fn (pp_type,pp_term) =>
       if depth = 0
       then PP.pp st label elision
       else let val (tr1,tr2) = D.Thm tr
                val tr1rev = rev tr1
                val tr11 = rev (tl tr1rev)
                and tr12 = hd tr1rev
                val hyps' = tr11
                val hyp' = tr12
                val concl' = tr2
            in if Ext.show_assums ()
               then PP.pp_hov_box label (1000,PP.ABSOLUTE 0,0)
                       (map
                           (fn el =>
                               PP.pp_h_box label 0
                                  [Local.box_el
                                      (pp_term st
                                          {outer_prec = Local.MINPREC,
                                           depth = depth - 1} params) hyps'
                                      el,PP.pp st label ","])
                           (Local.upto 1 (length hyps')) @
                        [pp_term st
                            {outer_prec = Local.MINPREC,depth = depth - 1}
                            params hyp',
                         PP.pp_h_box label 1000
                            [PP.pp st label "|-",
                             pp_term st
                                {outer_prec = Local.MINPREC,depth = depth - 1}
                                params concl']])
               else PP.pp_h_box label 1000
                       [PP.pp st label (Ext.replicate (".",length hyps' + 1)),
                        PP.pp st label "|-",
                        pp_term st
                           {outer_prec = Local.MINPREC,depth = depth - 1}
                           params concl']
            end
            handle _ =>
            let val (tr1,tr2) = D.Thm tr
                val _ = if null tr1 then () else raise Local.PPpattern
                val concl' = tr2
            in PP.pp_h_box label 1000
                  [PP.pp st label "|-",
                   pp_term st {outer_prec = Local.MINPREC,depth = depth - 1}
                      params concl']
            end
            handle _ => raise Local.PPpattern)
and pp_goal st {outer_prec,depth} (params as {context = context'})
       (tr as (_,label)) =
   (fn (pp_type,pp_term) =>
       if depth = 0
       then PP.pp st label elision
       else let val (tr1,tr2) = D.Goal tr
                val hyps' = tr1
                val concl' = tr2
            in let val boxes =
                      [((PP.ABSOLUTE 4000,0),
                        PP.pp_h_box label 0
                           [PP.pp st label "(--`",
                            pp_term st
                               {outer_prec = Local.MINPREC,depth = depth - 1}
                               params concl',PP.pp st label "`--)"])] @
                      [((PP.ABSOLUTE 0,0),
                        PP.pp st label (Ext.goal_line ()))] @
                      map
                         (fn el =>
                             ((PP.ABSOLUTE 4000,0),
                              PP.pp_h_box label 0
                                 [PP.pp st label "(--`",
                                  Local.box_el
                                     (pp_term st
                                         {outer_prec = Local.MINPREC,
                                          depth = depth - 1} params)
                                     (rev hyps') el,PP.pp st label "`--)"]))
                         (Local.upto 1 (length hyps')) @
                      [((PP.ABSOLUTE 4000,0),PP.pp_empty_box)]
               in PP.pp_v_sepbox label (#2 (hd boxes)) (tl boxes)
               end
            end
            handle _ =>
            let val (tr1,tr2) = D.Goal tr
                val _ = if null tr1 then () else raise Local.PPpattern
                val concl' = tr2
            in PP.pp_v_box label (PP.ABSOLUTE 0,0)
                  [PP.pp_h_box label 0
                      [PP.pp st label "(--`",
                       pp_term st
                          {outer_prec = Local.MINPREC,depth = depth - 1}
                          params concl',PP.pp st label "`--)"],
                   PP.pp_empty_box,PP.pp_empty_box]
            end
            handle _ => raise Local.PPpattern);

end;

end;
