; Dragondictate utilities for emacs v. 0.8
; Written by Thomas Rene Nielsen (trn@imada.ou.dk)
; Notice that this is not the final version

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Definition of modes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(setq c-modes '(c-mode))
(setq lisp-modes '(lisp-mode emacs-lisp-mode))
(setq tex-modes '(ams-latex-mode latex-mode plain-tex-mode))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Internal stuff
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(global-set-key [down-mouse-1] 'mouse-1-down-action)
(global-set-key "\r" 'return-key)
(global-set-key "\n" 'return-key)

(setq main-frame (selected-frame))


; Sets up a couple of hooks
(defun mouse-1-down-action (event)
  "Executes the normal 'mouse-drag-region' command and runs the hook 'mouse-1-down-hook'"
  (interactive "e")
  (progn (mouse-drag-region event)
	 (run-hooks 'mouse-1-down-hook))
)


(defun return-key ()
  "Executes the 'newline' as normal and calls the 'return-key-hook'."
  (interactive)
  (progn (newline-and-indent)
	 (run-hooks 'return))
  )



;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Polymorph functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun let-selection ()
  "A polymorph function. Will make a let form in lisp-mode and just
write 'let ' in all other modes."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-make-let)
    (insert "let ")) 
  )


(defun make-while-do ()
  "Inserts a while-do form, depending on EMACS mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-while-do)
    (if (member major-mode c-modes)
      (c-while-do)))    
  )


(defun make-do-while ()
  "Inserts a do-while form, depending on EMACS mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-do-while)
    (if (member major-mode c-modes)
      (c-do-while)))    
  )


(defun make-conditional ()
  "Inserts a conditional form (if), depending om EMACS mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-conditional)
    (if (member major-mode c-modes)
      (c-conditional)))    
  )  


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Pairs of brackets, parens, quotes ..
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun paren-body ()
  "make paranteses body."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-paren)
    (progn (insert "()")
	   (backward-char 1)))
  )


(defun normal-quotes ()
  "make quote body."
  (interactive)
  (insert-char ?\" 2)
  (backward-char)
)

(defun single-quotes ()
  "make single quote body."
  (interactive)
  (insert-char ?\' 2)
  (backward-char)
)


(defun quote-body ()
  "Inserts normal or tex quote pair, depending on major-mode."
  (interactive)
  (if (member major-mode tex-modes)
      (tex-quotes)
    (if (member major-mode lisp-modes)
	(lisp-quote)
      (normal-quotes))) 
  )


(defun brace-body ()
  "make brace pair."
  (interactive)
  (insert "{}")
  (backward-char)
)


(defun bracket-body ()
  "make bracket body."
  (interactive)
  (insert "[]")
  (backward-char)
)


(defun insert-remark ()
  "Inserts C, Lisp or tex remark, depending on major-mode."
  (interactive)
  (if (member major-mode tex-modes)
      (tex-remark))
  (if (member major-mode c-modes)
      (c-remark))
  (if (member major-mode lisp-modes)
      (lisp-remark))
  )



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Internal used functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun position-of-mouse ()
  "Returns the buffer position of the mousepointer in selected window."
  (car (compute-motion (window-start)
		       (cons (car(window-edges)) (car(cdr(window-edges))))
		       (point-max)
		       (cons (car(cdr(mouse-position))) (cdr(cdr(mouse-position))))
		       ; window width minus one because one coloum is used
		       ; to show '\' or '$' in continuation lines.
		       (- (window-width) 1)
		       (cons (window-hscroll) 0)
		       (selected-window)))
  )


(defun position-of-cursor ()
  "Returns the screen coordinates of the cursor."
  (let ((temp (compute-motion (window-start)
		       (cons (car(window-edges)) (car(cdr(window-edges))))
		       (point)
		       '(1000 . 1000)
		       ; window width minus one because one coloum is used
		       ; to show '\' or '$' in continuation lines.
		       (- (window-width) 1)
		       (cons (window-hscroll) 0)
		       (selected-window))))
    (cons (car (cdr temp)) (car (cdr (cdr temp)))))
  
  )

(defun get-start-pos ()
  (let ((this-pos (point)))
    (beginning-of-line)
    (let ((start-pos (point)))
      (goto-char this-pos)
      start-pos))
  )


(defun get-end-pos ()
  (let ((this-pos (point)))
    (end-of-line)
    (let ((end-pos (point)))
      (goto-char this-pos)
      end-pos))
  )


(defun looking-at-num (num exp)
  "Like 'looking-at', but using num as offset from point."
  (forward-char num)
  (let ((bool (looking-at exp)))
    (backward-char num)
    bool)
  )


(defun start-of-word ()
  "Moves cursor to start of word. Returns true if cursor has been moved."
  (interactive)
    (while (and (>= (- (point) 1) (point-min)) (or (and (<= ?A (char-after (- (point) 1))) (>= ?Z (char-after (- (point) 1)))) (and (<= ?a (char-after (- (point) 1))) (>= ?z (char-after (- (point) 1)))))) 
      (backward-char 1))
  )

 

;;;;;;;;;;;;;;;;;;;;;;;
;; Selection
;;;;;;;;;;;;;;;;;;;;;;;

(defun cursor-list-select ()
  "Selects the list entry at cursor. For use with the File-list,
Identifier-list Latex-list...."
  (interactive)
  (set-mouse-position (selected-frame) (car (position-of-cursor)) (cdr (position-of-cursor)))
  (defun-list-mouse-action)
  (var-list-mouse-action)
  (c-function-list-mouse-action)
  (c-var-list-mouse-action)
  (file-list-mouse-action)
  (latex-main-mouse-action)
  (latex-sub-mouse-action)
  (word-pad-mouse-action)
  (c-main-mouse-action)
  (c-sub-mouse-action)
  (lisp-main-mouse-action)
  (lisp-sub-mouse-action)
  )


(defun get-list-entry (num)
  "Selects the 'num'th entry in current window."
  (interactive "p")
  (let ((this-pos (point)))
    (move-to-window-line num)
    (set-mouse-position (selected-frame) (car (position-of-cursor)) (cdr (position-of-cursor)))
    (defun-list-mouse-action)
    (var-list-mouse-action)
    (c-function-list-mouse-action)
    (c-var-list-mouse-action)
    (file-list-mouse-action)
    (latex-main-mouse-action)
    (latex-sub-mouse-action)
    (word-pad-mouse-action)
    (c-main-mouse-action)
    (c-sub-mouse-action)
    (lisp-main-mouse-action)
    (lisp-sub-mouse-action)
  ;  (beginning-of-line)
  ;  (let ((start-pos (point)))
  ;    (end-of-line)
  ;    (let ((end-pos (point)))
  ;	(copy-region-as-kill start-pos end-pos)))
    (goto-char this-pos))
  )
	

;;;;;;;;;;;;;;;;;;;;;;;;
;; Cursor movement
;;;;;;;;;;;;;;;;;;;;;;;;

(defun several-lines-up ()
  "Asks for number of lines to go up, and executes if possible."
  (interactive)
  (previous-line (string-to-number (read-string "Lines to go : ")))
  )


(defun several-lines-down ()
  "Asks for number of lines to go down, and executes if possible."
  (interactive)
  (next-line (string-to-number (read-string "Lines to go : ")))
  )


(defun start-of-buffer ()
  "Moves cursor to start of buffer"
  (interactive)
  (goto-char (point-min))
  )



(defun start-of-line ()
  "Moves cursor to start of line"
  (interactive)
  (goto-char (get-start-pos))
  )


(defun buffer-part (num)
  "Takes a number (0 - 10) as input. 0 is first line and 10 is last line in
buffer. Imagine the buffer is divided in 10 parts. buffer-part wil go to the
first line in of the choosen part."
  (interactive "p")
  (let ((lines (count-lines (point-min) (point-max))))
    (goto-line (* (/ lines 10) num)))
  )


(defun skip-forward ()
  "Skips forward to the next logical place to write, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-skip-forward)
    (if (member major-mode c-modes)
	(c-skip-forward)
      (if (member major-mode tex-modes)
	  (latex-skip-forward)
	(text-skip-forward))))
  )


(defun skip-backward ()
  "Skips backward to the previous logical place to write, depending on
current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-skip-backward)
    (if (member major-mode c-modes)
	(c-skip-backward)
      (if (member major-mode tex-modes)
	  (latex-skip-backward)
	(text-skip-backward))))
  )


(defun nice-newline ()
  "Moves to new line, below current line, without moving the rest of the
current line to the new line."
  (interactive)
  (next-line 1)
  (beginning-of-line)
  (newline 1)
  (previous-line 1)
)


(defun go-mainframe ()
  "Jumps to the main frame. That is, the frame first activated."
  (interactive)
  (activate-frame main-frame)
  )


(defun go-var-frame ()
  "Jumps to var frame. Lisp or C depending on EMACS mode."
  (interactive)
  (if (member major-mode c-modes)
      (go-c-var-frame)
    (if (member major-mode lisp-modes)
	(go-lisp-var-frame)))
  )


(defun go-function-frame ()
  "Jumps to function frame. Lisp or C depending on EMACS mode."
  (interactive)
  (if (member major-mode c-modes)
      (go-c-function-frame)
    (if (member major-mode lisp-modes)
	(go-lisp-defun-frame)))
  )


(defun go-main-menu-frame ()
  "Jumps to main menu frame in Lisp, Latex or C depending on EMACS mode."
  (interactive)
  (if (member major-mode c-modes)
      (go-c-main-frame)
    (if (member major-mode lisp-modes)
	(go-lisp-main-frame)
      (if (member major-mode tex-modes)
	  (go-latex-main-frame))))
  )


(defun go-sub-menu-frame ()
  "Jumps to sub menu frame in Lisp, Latex or C depending on EMACS mode."
  (interactive)
  (if (member major-mode c-modes)
      (go-c-sub-frame)
    (if (member major-mode lisp-modes)
	(go-lisp-sub-frame)
      (if (member major-mode tex-modes)
	  (go-latex-sub-frame))))
  )






(defun goto-next-frame ()
  "Jumps to the next frame."
  (interactive)
  (activate-frame (next-frame))
  )


(defun goto-previous-frame ()
  "Jumps to the previous frame."
  (interactive)
  (activate-frame (previous-frame))
  )


(defun activate-frame (frame)
  (if (frame-live-p frame)
      (progn (set-mouse-position frame 0 0)
	     (raise-frame frame)))
  )


(defun get-frame-position (frame)
  "Returns the (left . top) position of frame."
  (cons (cdr (assoc 'left (frame-parameters frame))) (cdr (assoc 'top (frame-parameters frame))))
  )

(defun get-frame-middle (frame)
  "returns the middle of frame (x . y)."
  (cons (/ (+ (cdr (assoc 'left (frame-parameters frame)))
	      (frame-pixel-width frame)) 2)
	(/ (+ (cdr (assoc 'top (frame-parameters frame)))
	      (frame-pixel-height frame)) 2))
  )


(defun frame-west ()
  "Activates the frame most to the west."
  (interactive)
  (let ((frame-list (visible-frame-list)) (left 100000) (choosen-frame nil))
    (while (car frame-list)
      (if (< (car (get-frame-position (car frame-list))) left)
	  (progn (setq choosen-frame (car frame-list))
		 (setq left (car (get-frame-position choosen-frame)))))
      (setq frame-list (cdr frame-list)))
    (activate-frame choosen-frame))
  )


(defun frame-north ()
  "Activates the frame most to the north."
  (interactive)
  (let ((frame-list (visible-frame-list)) (north 100000) (choosen-frame nil))
    (while (car frame-list)
      (if (< (cdr (get-frame-position (car frame-list))) north)
	  (progn (setq choosen-frame (car frame-list))
		 (setq north (cdr (get-frame-position choosen-frame)))))
      (setq frame-list (cdr frame-list)))
    (activate-frame choosen-frame))
  )



(defun frame-east ()
  "Activates the frame most to the east."
  (interactive)
  (let ((frame-list (visible-frame-list)) (right -1) (choosen-frame nil))
    (while (car frame-list)
      (if (> (+ (car (get-frame-position (car frame-list)))
		(frame-pixel-width (car (frame-list))))
	     right)
	  (progn (setq choosen-frame (car frame-list))
		 (setq right (+ (car (get-frame-position choosen-frame))
				(frame-pixel-width choosen-frame)))))
      (setq frame-list (cdr frame-list)))
    (activate-frame choosen-frame))
  )


(defun frame-south ()
  "Activates the frame most to the south."
  (interactive)
  (let ((frame-list (visible-frame-list)) (south -1) (choosen-frame nil))
    (while (car frame-list)
      (if (> (+ (cdr (get-frame-position (car frame-list)))
		(frame-pixel-height (car (frame-list))))
	     south)
	  (progn (setq choosen-frame (car frame-list))
		 (setq south (+ (cdr (get-frame-position choosen-frame))
				(frame-pixel-height choosen-frame)))))
      (setq frame-list (cdr frame-list)))
    (activate-frame choosen-frame))
  )


(defun get-frame (num)
  "Selection of frames. Takes a two-digit number as input. Each digit can go from 1 to 3. Imagine the screen is divided into a 3x3 grid. with gridpart (x,y) 11 upper left and grid-part 33 lower right. If multiple frames in same grid-part, the other frames can be reached be calling get-frame for that part, repeatedly."
  (interactive "p")
  (let ((x (/ num 10)) (y (% num 10)))
    (let ((x-min (/ (* (- x 1) (x-display-pixel-width)) 3))
	  (x-max (/ (* x (x-display-pixel-width)) 3))
	  (y-min (/ (* (- y 1) (x-display-pixel-height)) 3))
	  (y-max (/ (* y (x-display-pixel-height)) 3)))
      (let ((cur-frame (next-frame (selected-frame) 'visible))
	    (choosen-frame nil))
	(while (not (equal cur-frame (selected-frame)))
	  (if (and
	       (or (and (<= (car (get-frame-position cur-frame)) x-max)
			(>= (+ (frame-pixel-width cur-frame) (car (get-frame-position cur-frame))) x-max))
		   (and (<= (car (get-frame-position cur-frame)) x-min)
			(>= (+ (frame-pixel-width cur-frame) (car (get-frame-position cur-frame))) x-min))
		   (and (<= (car (get-frame-position cur-frame)) x-max)
			(>= (car (get-frame-position cur-frame)) x-min))
		   (and (<= (+ (frame-pixel-width cur-frame) (car (get-frame-position cur-frame))) x-max)
			(>= (+ (frame-pixel-width cur-frame) (car (get-frame-position cur-frame))) x-min)))
	       (or (and (<= (cdr (get-frame-position cur-frame)) y-max)
			(>= (+ (frame-pixel-height cur-frame) (cdr (get-frame-position cur-frame))) y-max))
		   (and (<= (cdr (get-frame-position cur-frame)) y-min)
			(>= (+ (frame-pixel-height cur-frame) (cdr (get-frame-position cur-frame))) y-min))
		   (and (<= (cdr (get-frame-position cur-frame)) y-max)
			(>= (cdr (get-frame-position cur-frame)) y-min))
		   (and (<= (+ (frame-pixel-height cur-frame) (cdr (get-frame-position cur-frame))) y-max)
			(>= (+ (frame-pixel-height cur-frame) (cdr (get-frame-position cur-frame))) y-min))))
	      (setq choosen-frame cur-frame))
	  
	  (setq cur-frame (next-frame cur-frame 'visible)))

	(if choosen-frame
	    (activate-frame choosen-frame)))))
  )





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; binaery 'search' of lists/buffers
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq binary-list nil)

(defun middle-of-buffer ()
  "Moves to the middle of the buffer, regarding lines.
Used to reset the binary 'search' commands :
'upper-part' and 'lower-part'."
  (interactive)
  (goto-line (/ (count-lines (point-min) (point-max)) 2))
  (insert-binary-list (point-min) (point-max))
  )


(defun upper-part ()
  "Makes binary 'search' on the upper part of buffer.
Will go to the line in the middle between the current cursor
position and the upper bound of the last binary 'search'."
  (interactive)
  (if (assoc (current-buffer) binary-list)
      (let ((pos1 (car (cdr (assoc (current-buffer) binary-list)))))
	(let ((cursor-pos (point)))
	  (previous-line (/ (count-lines pos1 (point)) 2))
	  (insert-binary-list pos1 cursor-pos)))
    (progn (insert-binary-list (point-min) (point))
	   ((previous-line (/ (count-lines (point-min) (point)) 2)))))
  )


(defun lower-part ()
  "Makes binary 'search' on the lower part of buffer.
Will go to the line in the middle between the current cursor
position and the lower bound of the last binary 'search'."
  (interactive)
  (if (assoc (current-buffer) binary-list)
      (let ((pos2 (cdr (cdr (assoc (current-buffer) binary-list)))))
	(let ((cursor-pos (point)))
	  (next-line (/ (count-lines (point) pos2) 2))
	  (insert-binary-list cursor-pos pos2)))
    (progn (insert-binary-list (point) (point-max))
	   ((next-line (/ (count-lines (point) (point-max)) 2)))))
  )  


(defun insert-binary-list (pos1 pos2)
  (setq binary-list (delete (assoc (current-buffer) binary-list)
			    binary-list))
  (setq binary-list (append binary-list (list (cons (current-buffer)
					      (cons pos1 pos2)))))
  )




;;;;;;;;;;;;;;;;;;;;;;;
;; Parsing functions
;;;;;;;;;;;;;;;;;;;;;;;

(defun collect-function ()
  "collects the identifiers in the function, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-collect-defun-only)
    (if (member major-mode c-modes)
	(collect-c-identifiers)))
  )


(defun collect-identifiers ()
  "collects the identifiers, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-collect-identifiers)
    (if (member major-mode c-modes)
	(collect-c-identifiers)))
  )


(defun clear-identifiers ()
  "Clears the identifiers, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-clear-identifiers)
    (if (member major-mode c-modes)
	(clear-c-identifiers)))
  )


(defun open-identifier-list ()
  "Opens the identifier lists, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (progn (open-var-list)
	     (open-defun-list))
    (if (member major-mode c-modes)
	(progn (open-c-var-list)
	       (open-c-function-list))))
  )


(defun open-var-identifier-list ()
  "Opens the variable list, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (open-var-list)
    (if (member major-mode c-modes)
	(open-c-var-list)))
  )


(defun open-function-identifier-list ()
  "Opens the function list, depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (open-defun-list)
    (if (member major-mode c-modes)
	(open-c-function-list)))
  )



(defun collect-all-identifiers-buffer ()
  "collects all identifiers in the buffer,depending on current mode."
  (interactive)
  (if (member major-mode lisp-modes)
      (lisp-collect-all-identifiers-buffer)
    (if (member major-mode c-modes)
	(collect-all-c-identifiers-buffer)))
  )


;;;;;;;;;;;;;;;;;;;;;;;;
;; Copy, paste, delete 
;;;;;;;;;;;;;;;;;;;;;;;;


(defun delete-word ()
  "Deletes the word under the cursor or word before cursor if cursor is placed
on space behind word. If several spaces follows word, 'delete-word' deletes
following word instead. Deleted word is places in kill-ring."
  (interactive)
  (start-of-word)
  (if (not (looking-at "[\n ]"))
	   (progn (kill-word 1)
  
  ; kill empty space
		  (if (looking-at " ")
		      (progn (delete-char 1)
			     (backward-char 1)))))
   )


(defun copy-word ()
  "Copies the word at mousepoint to point. Only coping the word directly under
the mousepoint in words connected with '-' or '_'."
  (interactive)
  (let ((this-pos (point)))
    (let ((this-window (selected-window)))
      (select-window (window-at (car(cdr(mouse-position))) (cdr(cdr(mouse-position)))))
      (goto-char (position-of-mouse))
      (start-of-word)
      (let ((start-pos (point)))
	(forward-word 1)
	(if (member major-mode lisp-modes)
	    (while (looking-at "-")
	      (forward-word 1)))
	(if (member major-mode c-modes)
	    (while (looking-at "_")
	      (forward-word 1)))
	(let ((end-pos (point)))
	  (copy-region-as-kill start-pos end-pos)))
      (select-window this-window))
    (goto-char this-pos))
  (yank)
  )



(defun copy-between ()
  "Copies region between mouse-point and cursor to kill-ring."
  (interactive)
  (kill-ring-save (position-of-mouse) (point))
  )

 
(defun copy-region ()
  "Copies region between mark and cursor to kill-ring."
  (interactive)
  (if mark-active
      (kill-ring-save (mark) (point))
    (message "Mark is not set!"))
  )


(defun kill-mark-region ()
 "Kills region between mark and cursor, coping region to kill-ring."
 (interactive)
 (if mark-active
     (kill-region (mark) (point))
   (message "Mark is not set!"))
  )


(defun kill-between ()
  "Kills region between mouse-point and cursor, coping region to kill-ring."
  (interactive)
  (kill-region (position-of-mouse) (point))
  )


(defun paste ()
  "Pastes with 'yank', but wil run functions in tex, C and lisp too.
See 'function-paste'."
  (interactive)
  (if (or (member major-mode c-modes) (member major-mode tex-modes) (member major-mode lisp-modes))
      (function-paste)
    (yank))
  )


(defun function-paste ()
  "Yanks normal text, but text starting with '@\' will be run as a command."
  (let ((start-pos (point)))
    (yank)
    (let ((end-pos (point)))
      (goto-char start-pos)
      (if (looking-at "@")
	  (if (looking-at-num 1 "[\\]")
	      (let ((command-name (buffer-substring (+ start-pos 2) end-pos)))
		(delete-region start-pos end-pos)
		(funcall (intern command-name))))
	(goto-char end-pos))))
  )


;;;;;;;;;;;;;;;;;;;;;;;;;;
; no-space-mode
;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq no-space-mode nil)


(or (assq 'no-space-mode minor-mode-alist)
    (setq minor-mode-alist
	  (cons '(no-space-mode " No Space") minor-mode-alist)))


(defun no-space-mode (arg)
  "This mode vil switch the space bar off, when the mode is active.
 The mode loses the spaces Dragondictate makes between words.
Most useful when programming. Bind a speak command to the command
'space-bar', to make spaces manually. "
  (interactive "P")
  (setq no-space-mode
	(if (null arg) (not no-space-mode) 
	  (> (prefix-numeric-value arg) 0)))
  )


(global-set-key " " 'make-space-bar)


(defun make-space-bar ()
  "Insert space, but only when not in no-space-mode."
  (interactive)
  (if (not no-space-mode)
      (space-bar))
  )


(defun space-bar ()
  "Inserts space."
  (interactive)
  (insert " ")
  )


;;;;;;;;;;;;;;;;;;;;;
;; Undo
;;;;;;;;;;;;;;;;;;;;;

(defun dragondictate-undo ()
  "Undo made to 'counter' the DragonDictate 'No Space' mode."
  (interactive)
  (advertised-undo 1)
  (insert " ")
  )

;;;;;;;;;;;;;;;;;;;;;
;; Skip
;;;;;;;;;;;;;;;;;;;;;
(defun text-skip-forward ()
  "skips to the next logical place to write in text mode."
  (interactive)
  (let ((end-pos (get-end-pos)))
      (re-search-forward "[\]\[\'\")({}]" end-pos t))
  )


(defun text-skip-backward ()
  "skips to the previous logical place to write in text mode."
  (interactive)
  (let ((start-pos (get-start-pos)))
      (re-search-backward "[\]\[\'\")({}]" start-pos t))
  )




;;;;;;;;;;;;;;;;;;;;;;;;;;
;; menu list
;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun open-main-list ()
  "Opens main menu list in Lisp, Latex or C depending on EMACS mode."
  (interactive)
  (if (member major-mode c-modes)
      (c-open-main-list)
    (if (member major-mode lisp-modes)
	(lisp-open-main-list)
      (if (member major-mode tex-modes)
	  (latex-open-main-list))))
  )














