/*
 * This class is a modification of java.awt.image.MemoryImageSource. For access
 * control reasons, it could not simply inherit from it, so, since I have
 * copied code from the Sun libraries, I include their copyright notice below.
 */

/*
 * @(#)MemoryImageSource.java	1.15 95/12/14 Jim Graham
 *
 * Copyright (c) 1994 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

package india;

import java.awt.image.*;
import java.util.Hashtable;

/**
 * This class is a modification of java.awt.image.MemoryImageSource which
 * allows its source pixels to be modified. This ought to get rid of some
 * problems with running out of memory.
 * @author	Colin Watson
 * @version	$Id: ReusableMemoryImageSource.java,v 1.2 1999/02/23 23:48:17 cjw44 Exp $
 * @see		MemoryImageSource
 */
public class ReusableMemoryImageSource implements ImageProducer
{

	protected int width;
	protected int height;
	protected ColorModel model;
	protected Object pixels;
	protected int pixeloffset;
	protected int pixelscan;
	protected Hashtable properties;

	/**
	 * Constructs an ImageProducer object which uses an array of bytes
	 * to produce data for an Image object.
	 * @see java.awt.Component#createImage
	 */
	public ReusableMemoryImageSource(int w, int h, ColorModel cm,
		byte[] pix, int off, int scan)
	{
		initialize(w, h, cm, (Object)pix, off, scan, null);
	}

	/**
	 * Constructs an ImageProducer object which uses an array of bytes
	 * to produce data for an Image object.
	 * @see java.awt.Component#createImage
	 */
	public ReusableMemoryImageSource(int w, int h, ColorModel cm,
		byte[] pix, int off, int scan, Hashtable props)
	{
		initialize(w, h, cm, (Object)pix, off, scan, props);
	}

	/**
	 * Constructs an ImageProducer object which uses an array of integers
	 * to produce data for an Image object.
	 * @see java.awt.Component#createImage
	 */
	public ReusableMemoryImageSource(int w, int h, ColorModel cm,
		int[] pix, int off, int scan)
	{
		initialize(w, h, cm, (Object)pix, off, scan, null);
	}

	/**
	 * Constructs an ImageProducer object which uses an array of integers
	 * to produce data for an Image object.
	 * @see java.awt.Component#createImage
	 */
	public ReusableMemoryImageSource(int w, int h, ColorModel cm,
		int[] pix, int off, int scan, Hashtable props)
	{
		initialize(w, h, cm, (Object)pix, off, scan, props);
	}

	private void initialize(int w, int h, ColorModel cm,
		Object pix, int off, int scan, Hashtable props)
	{
		width = w;
		height = h;
		model = cm;
		pixels = pix;
		pixeloffset = off;
		pixelscan = scan;
		if(props == null)
			props = new Hashtable();
		properties = props;
	}

	/**
	 * Constructs an ImageProducer object which uses an array of integers
	 * in the default RGB ColorModel to produce data for an Image object.
	 * @see java.awt.Component#createImage
	 * @see ColorModel#getRGBdefault
	 */
	public ReusableMemoryImageSource(int w, int h, int[] pix,
		int off, int scan)
	{
		initialize(w, h, ColorModel.getRGBdefault(),
			(Object)pix, off, scan, null);
	}

	/**
	 * Constructs an ImageProducer object which uses an array of integers
	 * in the default RGB ColorModel to produce data for an Image object.
	 * @see java.awt.Component#createImage
	 * @see ColorModel#getRGBdefault
	 */
	public ReusableMemoryImageSource(int w, int h, int[] pix,
		int off, int scan, Hashtable props)
	{
		initialize(w, h, ColorModel.getRGBdefault(),
			(Object)pix, off, scan, props);
	}

	// We can only have one consumer since we immediately return the data...
	private ImageConsumer theConsumer;

	/**
	 * Adds an ImageConsumer to the list of consumers interested in
	 * data for this image.
	 * @see ImageConsumer
	 */
	public synchronized void addConsumer(ImageConsumer ic)
	{
		theConsumer = ic;
		try
		{
			produce();
		} catch (Exception e)
		{
			if(theConsumer != null)
				theConsumer.imageComplete(
					ImageConsumer.IMAGEERROR);
		}
		theConsumer = null;
	}

	/**
	 * Determine if an ImageConsumer is on the list of consumers currently
	 * interested in data for this image.
	 * @return true if the ImageConsumer is on the list; false otherwise
	 * @see ImageConsumer
	 */
	public synchronized boolean isConsumer(ImageConsumer ic)
	{
		return ic == theConsumer;
	}

	/**
	 * Remove an ImageConsumer from the list of consumers interested in
	 * data for this image.
	 * @see ImageConsumer
	 */
	public synchronized void removeConsumer(ImageConsumer ic)
	{
		if(theConsumer == ic)
			theConsumer = null;
	}

	/**
	 * Adds an ImageConsumer to the list of consumers interested in
	 * data for this image, and immediately start delivery of the
	 * image data through the ImageConsumer interface.
	 * @see ImageConsumer
	 */
	public void startProduction(ImageConsumer ic)
	{
		addConsumer(ic);
	}

	/**
	 * Requests that a given ImageConsumer have the image data delivered
	 * one more time in top-down, left-right order.
	 * @see ImageConsumer
	 */
	public void requestTopDownLeftRightResend(ImageConsumer ic)
	{
		// Not needed.  The data is always in TDLR format.
	}

	private void produce()
	{
		if (theConsumer != null)
		{
			theConsumer.setDimensions(width, height);
			theConsumer.setProperties(properties);
			theConsumer.setColorModel(model);
			theConsumer.setHints(ImageConsumer.TOPDOWNLEFTRIGHT |
					 ImageConsumer.COMPLETESCANLINES |
					 ImageConsumer.SINGLEPASS |
					 ImageConsumer.SINGLEFRAME);
			if (pixels instanceof byte[])
				theConsumer.setPixels(0, 0, width, height,
					model, (byte[])pixels, pixeloffset,
					pixelscan);
			else
				theConsumer.setPixels(0, 0, width, height,
					model, (int[])pixels, pixeloffset,
					pixelscan);
			theConsumer.imageComplete(
				ImageConsumer.STATICIMAGEDONE);
		}
	}

	/*
	 * The following methods are the reason for this class: I want the
	 * ability to alter the MemoryImageSource after it has been
	 * constructed.
	 */

	public synchronized void modify(int w, int h, ColorModel cm,
		byte[] pix, int off, int scan)
	{
		initialize(w, h, cm, (Object)pix, off, scan, properties);
	}

	public synchronized void modify(int w, int h, ColorModel cm,
		byte[] pix, int off, int scan, Hashtable props)
	{
		initialize(w, h, cm, (Object)pix, off, scan, props);
	}

	public synchronized void modify(int w, int h, ColorModel cm,
		int[] pix, int off, int scan)
	{
		initialize(w, h, cm, (Object)pix, off, scan, properties);
	}

	public synchronized void modify(int w, int h, ColorModel cm,
		int[] pix, int off, int scan, Hashtable props)
	{
		initialize(w, h, cm, (Object)pix, off, scan, props);
	}

	public synchronized void modify(int w, int h, int[] pix,
		int off, int scan)
	{
		initialize(w, h, model, (Object)pix, off, scan, properties);
	}

	public synchronized void modify(int w, int h, int[] pix,
		int off, int scan, Hashtable props)
	{
		initialize(w, h, model, (Object)pix, off, scan, props);
	}

}
