package india;

import java.awt.*;
import java.awt.image.*;

/**
 * A visual representation of an Edsac register.
 * @author	Colin Watson
 * @version	$Id: RegisterTank.java,v 2.13 1999/03/09 01:10:03 cjw44 Exp $
 * @see		FrontEnd
 */

public class RegisterTank extends FloatingComponent
{

	protected static long updateThreshold = 40;
	protected static boolean draw = true;

	protected long lastUpdate = 0;
	protected boolean updated = false;

	protected StatusBar status;
	protected String name;
	protected int length;
	protected boolean peeping;

	// AccumulatorTank doesn't need these
	private long value = 0, lastUpdatedValue = 0;

	protected int ledstartx, ledstarty;
	protected int ledwidth, ledheight;
	protected int width, height;

	protected int[] pixOff, pixOn;
	protected int[] pixNow;	// the current appearance of the RegisterTank
	protected Image setValueImage, paintImage;
	protected MemoryImageSource setValueSource, paintSource;

	/**
	 * Creates a new register tank display with the specified images. The
	 * images should both be the same size.
	 *
	 * @param	name	The name of the register.
	 * @param	status	The status bar on which to display information
	 *			when the mouse moves over the register tank.
	 * @param	off	An image of the tank with all LEDs off.
	 * @param	on	An image of the tank with all LEDs on.
	 * @param	x	The x-coordinate of the top left of the image.
	 * @param	y	The y-coordinate of the top left of the image.
	 * @param	width	The width of the image.
	 * @param	height	The height of the image.
	 * @param	ledx	The x-coordinate of the top left corner of the
	 *			area covered by LEDs.
	 * @param	ledy	The y-coordinate of the top left corner of the
	 *			area covered by LEDs.
	 * @param	ledwidth	The width of one LED.
	 * @param	ledheight	The height of one LED.
	 * @param	length	The length of the tank (number of LEDs).
	 */
	public RegisterTank(String name, StatusBar status, Image off, Image on,
			int x, int y, int width, int height,
			int ledx, int ledy, int ledwidth, int ledheight,
			int length)
		throws ImageLoadingException
	{
		this.name = name;
		this.status = status;
		ledstartx = ledx;
		ledstarty = ledy;
		this.ledwidth = ledwidth;
		this.ledheight = ledheight;
		this.length = length;

		Image[] imgs = {off, on};
		checkImages(imgs);

		this.width = width;
		this.height = height;

		pixOff = new int[width * height];
		pixOn = new int[width * height];
		pixNow = new int[width * height];

		try
		{
			PixelGrabber pg = new PixelGrabber(
				off, x, y, width, height, pixOff, 0, width);
			pg.grabPixels();
			pg = new PixelGrabber(
				on, x, y, width, height, pixOn, 0, width);
			pg.grabPixels();
		} catch(InterruptedException e)
		{
			throw new ImageLoadingException(
				"images passed to class " +
				getClass().getName() + " (interrupted)");
		}
		System.arraycopy(pixOff, 0, pixNow, 0, width * height);
		setValueSource = new MemoryImageSource(
			width, height, pixNow, 0, width);
		paintSource = new MemoryImageSource(
			width, height, pixNow, 0, width);
	}

	public static long getUpdateThreshold()
	{
		return updateThreshold;
	}

	public static void setUpdateThreshold(long updateThreshold)
	{
		RegisterTank.updateThreshold = updateThreshold;
	}

	public static boolean getDrawState()
	{
		return draw;
	}

	public static void setDrawState(boolean draw)
	{
		RegisterTank.draw = draw;
	}

	public synchronized boolean mouseMove(Event e, int x, int y)
	{
		if(draw && !peeping)
		{
			peeping = true;
			status.setRegister(name, value);
		}
		return true;
	}

	public synchronized boolean mouseDrag(Event e, int x, int y)
	{
		if(x < 0 || x >= width || y < 0 || y >= height)
		{
			if(!draw || peeping)
			{
				peeping = false;
				status.clearPeep();
			}
		}
		else if(draw && !peeping)
		{
			peeping = true;
			status.setRegister(name, value);
		}
		return true;
	}

	public synchronized boolean mouseEnter(Event e, int x, int y)
	{
		if(draw && !peeping)
		{
			peeping = true;
			status.setRegister(name, value);
		}
		return true;
	}

	public synchronized boolean mouseExit(Event e, int x, int y)
	{
		if(!draw || peeping)
		{
			peeping = false;
			status.clearPeep();
		}
		return true;
	}

	/**
	 * Returns the current value stored in this register tank.
	 */
	public long getValue()
	{
		return value;
	}

	/**
	 * Sets the value stored in this register tank.
	 * @param	value	The new value for this register.
	 */
	public synchronized void setValue(long value)
	{
		this.value = value;

		if(!draw)
		{
			updated = false;
			return;
		}

		long time = System.currentTimeMillis();
		if(time - lastUpdate < updateThreshold)
		{
			updated = false;
			return;
		}
		lastUpdate = time;
		updated = true;

		if(value == lastUpdatedValue) return;
		long v = value, lastv = lastUpdatedValue;
		lastUpdatedValue = value;

		// Draw the change
		int startpos = ledstarty * width + ledstartx +
			ledwidth * length;
		for(int led = 0; led < length; led++)
		{
			startpos -= ledwidth;
			int linepos = startpos;
			if((v & 1) != (lastv & 1))
			{
				int[] pix = (v & 1) == 0 ? pixOff : pixOn;
				for(int y = 0; y < ledheight; y++)
				{
					for(int x = 0; x < ledwidth; x++)
						pixNow[linepos] =
							pix[linepos++];
					linepos += width - ledwidth;
				}
			}
			v >>>= 1;
			lastv >>>= 1;
		}

		Graphics g = getGraphics();
		if(setValueImage == null)
			setValueImage = createImage(setValueSource);
		else
			setValueImage.flush();
		g.drawImage(setValueImage, 0, 0, null);

		if(peeping)
			status.setRegister(name, value);
	}

	public synchronized void forceUpdate()
	{
		if(!draw)
		{
			updated = false;
			return;
		}

		if(updated) return;
		updated = true;

		if(value == lastUpdatedValue) return;
		long v = value, lastv = lastUpdatedValue;
		lastUpdatedValue = value;

		int startpos = ledstarty * width + ledstartx +
			ledwidth * length;
		for(int led = 0; led < length; led++)
		{
			startpos -= ledwidth;
			int linepos = startpos;
			if((v & 1) != (lastv & 1))
			{
				int[] pix = (v & 1) == 0 ? pixOff : pixOn;
				for(int y = 0; y < ledheight; y++)
				{
					for(int x = 0; x < ledwidth; x++)
						pixNow[linepos] =
							pix[linepos++];
					linepos += width - ledwidth;
				}
			}
			v >>>= 1;
			lastv >>>= 1;
		}

		Graphics g = getGraphics();
		if(setValueImage == null)
			setValueImage = createImage(setValueSource);
		else
			setValueImage.flush();
		g.drawImage(setValueImage, 0, 0, null);

		if(peeping)
			status.setRegister(name, value);
	}

	public void repaint(long tm, int x, int y, int w, int h) {}
	public void update(Graphics g) {}

	/**
	 * Paints the register tank.
	 */
	public void paint(Graphics g)
	{
		if(paintImage == null)
			paintImage = createImage(paintSource);
		else
			paintImage.flush();
		g.drawImage(paintImage, 0, 0, null);
	}

}

/*
 * $Log: RegisterTank.java,v $
 * Revision 2.13  1999/03/09 01:10:03  cjw44
 * Tanks are now properly turned off when "Show short tanks" is unchecked.
 *
 * Revision 2.12  1999/03/01 16:03:38  cjw44
 * Sorted out a few bugs in the peeping routines.
 *
 * Revision 2.11  1999/02/28 23:46:35  cjw44
 * Made the update threshold a variable, for the benefit of speed control.
 * Expanded the area over which the mouse needs to be moved to activate
 * peeping.
 *
 * Revision 2.10  1999/02/27 11:53:28  cjw44
 * Implemented Joe's suggestion that images don't need to be recreated after a
 * flush(), and streamlined the LED-setting code to only bother about LEDs that
 * have changed since the last update.
 *
 * Revision 2.9  1999/02/23 14:57:24  cjw44
 * Changed constructor to use new checkImages(Image[]) method in
 * FloatingComponent.
 *
 * Revision 2.8  1999/02/22 01:24:02  cjw44
 * Implemented "peeping", using the status bar.
 *
 * Revision 2.7  1999/02/18 03:14:47  cjw44
 * Fixed bug in register updating: the check for whether a register was already
 * displaying the requested value was being made against the last requested
 * value rather than the last displayed value.
 *
 * Revision 2.6  1999/02/16 22:09:41  cjw44
 * Added periodic updating (currently set with a threshold of 40 milliseconds).
 *
 * Revision 2.5  1999/02/16 20:31:55  cjw44
 * Added image flushing before each new image is created.
 *
 * Revision 2.4  1999/02/15 13:23:59  cjw44
 * Changed drawing code to use ReusableMemoryImageSources.
 *
 * Revision 2.3  1999/02/12 22:54:06  cjw44
 * Removed explicit garbage-collection code, yielding an enormous speed
 * increase (a factor of around 3).
 *
 * Revision 2.2  1999/02/12 18:39:16  cjw44
 * Fiddled with image drawing code in an attempt to fix a bug which was causing
 * the event dispatching thread to be locked out (I had thought it was a memory
 * problem). The bug was eventually found to be an issue of thread priorities
 * and a design flaw in the JVM's scheduler (see EdsacThread.java).
 *
 * Revision 2.1  1999/02/11 10:46:01  cjw44
 * Changed drawing system to use int arrays rather than images according to
 * Joe's suggestions: only one drawImage() call per method is now required.
 * Changed access control on value to private, since AccumulatorTank doesn't
 * need it.
 * Added the facility to turn updating of RegisterTanks on and off at run time.
 * Added explicit garbage-collection every GC_THRESHOLD updates.
 *
 * Revision 1.8  1999/02/08 18:10:17  cjw44
 * Changed method calls on the Graphics object passed to paint() to use this
 * component as an ImageObserver.
 *
 * Revision 1.7  1999/02/05 16:52:39  cjw44
 * Corrected two places where a variable was being referred to by the wrong
 * name.
 *
 * Revision 1.6  1999/02/04 22:54:40  cjw44
 * Changed superclass to FloatingComponent.
 * Changed constructor to take images of the tank with all LEDs off and with
 * all LEDs on, and to extract images of a single LED off and on from that.
 * Implemented paint() method to draw the appropriate pattern of LEDs.
 *
 * Revision 1.5  1999/01/31 18:54:22  cjw44
 * Method to set an LED is now called setValue(boolean).
 *
 * Revision 1.4  1999/01/31 18:39:42  cjw44
 * Changed access control on length, value, and leds to protected to allow
 * AccumulatorTank to work.
 *
 * Revision 1.3  1999/01/31 18:36:37  cjw44
 * Changed name of set() method to setValue().
 * Removed paint() method; this is now handled by subcomponents.
 * Added getValue() method.
 * Fully implemented constructor and all methods.
 *
 * Revision 1.2  1999/01/25 15:22:11  cjw44
 * Fixed the bit that was *supposed* to insert an RCS log ...
 */
