package india;

import java.applet.Applet;
import java.awt.*;
import java.awt.image.*;

/**
 * The Edsac's phone dial, via which input can be accepted from the user.
 * @author	Colin Watson
 * @version	$Id: PhoneDial.java,v 1.5 1999/03/10 13:39:50 cjw44 Exp $
 */

public class PhoneDial extends FloatingComponent
{

	private int size;
	private int[] holesx, holesy;
	private int holeradius;

	private FrontEnd fe;
	private Image offscreenImage;
	private int[] pixDial, pixCover, pixNow, pixCoverBackup;
	private Graphics osG;
	private Image rotatedImage;
	private MemoryImageSource rotatedSource;

	private double angle = 0, oldAngle = 0;

	private volatile boolean turning = false;

	/**
	 * Creates a new phone dial with the specified image.
	 *
	 * @param	fe	The calling FrontEnd.
	 * @param	dial	An image of the phone dial, minus "cover".
	 * @param	cover	An image of the "cover", which is rotated on
	 *			top of the bare dial.
	 * @param	startx	The x-coordinate of the top left of the image.
	 * @param	starty	The y-coordinate of the top left of the image.
	 * @param	size	The width = height of the phone dial.
	 */
	public PhoneDial(FrontEnd fe, Image dial, Image cover,
			int dialx, int dialy, int coverx, int covery, int size,
			int[] holesx, int[] holesy, int holeradius)
		throws ImageLoadingException
	{
		this.fe = fe;
		this.size = size;
		if(holesx.length != holesy.length)
			throw new IllegalArgumentException(
			"PhoneDial: holesx and holesy have different lengths");
		int length = holesx.length;
		this.holesx = new int[length];
		this.holesy = new int[length];
		System.arraycopy(holesx, 0, this.holesx, 0, length);
		System.arraycopy(holesy, 0, this.holesy, 0, length);
		this.holeradius = holeradius;

		Image[] imgs = {dial, cover};
		checkImages(imgs);

		pixDial = new int[size * size];
		pixCover = new int[size * size];
		pixNow = new int[size * size];

		try
		{
			PixelGrabber pg = new PixelGrabber(
				dial, dialx, dialy, size, size,
				pixDial, 0, size);
			pg.grabPixels();
			pg = new PixelGrabber(
				cover, coverx, covery, size, size,
				pixCover, 0, size);
			pg.grabPixels();
		} catch(InterruptedException e)
		{
			throw new ImageLoadingException(
				"images passed to class " +
				getClass().getName() + " (interrupted)");
		}
		for(int i = 0; i < size * size; i++)
		{
			int rdial = (pixDial[i] & 0xFF0000) >> 16;
			int gdial = (pixDial[i] & 0x00FF00) >> 8;
			int bdial = pixDial[i] & 0x0000FF;
			int rcover = (pixCover[i] & 0xFF0000) >> 16;
			int gcover = (pixCover[i] & 0x00FF00) >> 8;
			int bcover = pixCover[i] & 0x0000FF;
			pixNow[i] = (255 << 24) |
				((rdial * rcover / 255) << 16) |
				((gdial * gcover / 255) << 8) |
				(bdial * bcover / 255);
		}
		rotatedSource = new MemoryImageSource(
			size, size, pixNow, 0, size);
	}

	private int getHole(int x, int y)
	{
		int hr2 = holeradius * holeradius;
		for(int i = 0; i < holesx.length; i++)
		{
			int dx = x - holesx[i], dy = y - holesy[i];
			if(dx * dx + dy * dy <= hr2)
				return i;
		}
		return -1;
	}

	public boolean mouseUp(Event event, int x, int y)
	{
		int hole;
		// We synchronize here for the period between the test and the
		// set of turning, just in case.
		synchronized(this)
		{
			if(fe.isRunning() || turning) return true;
			hole = getHole(x, y);
			if(hole == -1) return true;
			turning = true;
		}
		Graphics g = getGraphics();
		Thread turner = new Thread(new DialTurner(
			this, hole, holesx[hole], holesy[hole],
			holeradius + 1));
		turner.setPriority(Thread.MIN_PRIORITY);
		turner.start();
		return true;
	}

	public void doneTurning(int hole)
	{
		fe.inputNumber(hole);
		synchronized(this)
		{
			turning = false;
		}
	}

	public void transformByMatrix(double[][] matrix)
	{
		double relx, rely;	// Co-ordinates centred on (0, 0)
		double transx, transy;	// Transformed co-ordinates
		double shift = size / 2;
		int pos = 0;

		rely = -shift;
		for(int y = 0; y < size; y++)
		{
			relx = -shift;
			for(int x = 0; x < size; x++)
			{
				transx = relx * matrix[0][0] +
					rely * matrix[0][1] + shift;
				transy = relx * matrix[1][0] +
					rely * matrix[1][1] + shift;
				if(transx > 0 && transx < size &&
						transy > 0 && transy < size)
				{
					int dial = pixDial[pos];
					int cover = pixCover[(int)transx +
						(int)transy * size];
					int rdial = (dial & 0xFF0000) >> 16;
					int gdial = (dial & 0x00FF00) >> 8;
					int bdial = dial & 0x0000FF;
					int rcover = (cover & 0xFF0000) >> 16;
					int gcover = (cover & 0x00FF00) >> 8;
					int bcover = cover & 0x0000FF;
					pixNow[pos++] = (255 << 24) |
					    ((rdial * rcover / 255) << 16) |
					    ((gdial * gcover / 255) << 8) |
					    (bdial * bcover / 255);
				}
				else
					pos++;
				relx++;
			}
			rely++;
		}
	}

	public void rotateClockwise(double angle)
	{
		double[][] matrix = {
			{ Math.cos(angle), Math.sin(angle) },
			{ -Math.sin(angle), Math.cos(angle) } };
		transformByMatrix(matrix);
	}

	public void darken(int x, int y, int radius)
	{
		if(pixCoverBackup != null) return;
		pixCoverBackup = new int[radius * radius * 4];
		int r2 = radius * radius;
		for(int i = 0; i < radius * 2; ++i)
			for(int j = 0; j < radius * 2; ++j)
			{
				int index = (i + y - radius) * size +
					j + x - radius;
				int cover = pixCoverBackup[i * radius * 2
					+ j] = pixCover[index];
				int dx = j - radius, dy = i - radius;
				if(dx * dx + dy * dy > r2)
					continue;
				int rcover = (cover & 0xFF0000) >> 16;
				int gcover = (cover & 0x00FF00) >> 8;
				int bcover = cover & 0x0000FF;
				pixCover[index] = (255 << 24) |
					((rcover * 3 / 4) << 16) |
					((gcover * 3 / 4) << 8) |
					(bcover * 3 / 4);
			}
	}

	public void restore(int x, int y, int radius)
	{
		if(pixCoverBackup == null) return;
		for(int i = 0; i < radius * 2; ++i)
			for(int j = 0; j < radius * 2; ++j)
				pixCover[(i + y - radius) * size +
						j + x - radius] =
					pixCoverBackup[i * radius * 2 + j];
		pixCoverBackup = null;
	}

	public void redraw(double angle, Graphics g)
	{
		oldAngle = this.angle = angle;
		rotateClockwise(angle);
		if(rotatedImage != null)
			rotatedImage.flush();
		else
			rotatedImage = createImage(rotatedSource);
		if(g != null)
			g.drawImage(rotatedImage, 0, 0, null);
		paint(g);
	}

	public void paint(Graphics g)
	{
		if(angle != oldAngle)
		{
			oldAngle = angle;
			rotateClockwise(angle);
			if(rotatedImage != null)
				rotatedImage.flush();
		}

		if(rotatedImage == null)
			rotatedImage = createImage(rotatedSource);

		if(g != null)
			g.drawImage(rotatedImage, 0, 0, null);
	}

}

/*
 * $Log: PhoneDial.java,v $
 * Revision 1.5  1999/03/10 13:39:50  cjw44
 * Added darken() and restore() methods so that the current hole can be
 * highlighted.
 *
 * Revision 1.4  1999/03/10 02:33:01  cjw44
 * Switched sense of dial.
 *
 * Revision 1.3  1999/02/27 11:10:03  cjw44
 * On Joe's suggestion, we don't need to create a new Image any more after
 * calling flush().
 *
 * Revision 1.2  1999/02/24 01:14:27  cjw44
 * Fixed a bug where clicking on the dial but not in a hole locked up the dial.
 *
 * Revision 1.1  1999/02/23 23:50:27  cjw44
 * Initial revision
 *
 */
