package india;

import java.awt.*;

/**
 * A window holding output from the Edsac, emulating a teleprinter.
 * @author	Colin Watson
 * @version	$Id: OutputWindow.java,v 1.9 1999/02/28 23:45:41 cjw44 Exp $
 * @see		FrontEnd
 */

public class OutputWindow extends Frame
{

	private static final String
		EDIT_CLEAR = "Clear";

	private TextArea text;

	/**
	 * Creates a new output window, and tries to set it to a monospaced
	 * font.
	 */
	public OutputWindow()
	{
		super("EDSAC Output");
		text = new TextArea(20, 40);
		text.setEditable(false);
		add("Center", text);

		// Try to get a monospaced font
		String[] fonts = Toolkit.getDefaultToolkit().getFontList();
		Font font;
		for(int i = 0; i < fonts.length; i++)
			if(fonts[i].toLowerCase().startsWith("monospaced"))
			{
				Font curFont = getFont();
				font = new Font(fonts[i],
					curFont != null ?
						curFont.getSize() : 12,
					Font.PLAIN);
				if(font != null)
				{
					text.setFont(font);
					break;
				}
			}

		MenuBar bar = new MenuBar();
		Menu editMenu = new Menu("Edit");
		editMenu.add(EDIT_CLEAR);
		bar.add(editMenu);
		setMenuBar(bar);

		pack();
	}

	/**
	 * Appends a character to the output.
	 * @param	ch	The character to append.
	 */
	public void appendText(char ch)
	{
		if(ch != '\r')
			text.appendText(String.valueOf(ch));
	}

	/**
	 * Appends a string to the output.
	 * @param	s	The string to append.
	 */
	public void appendText(String s)
	{
		text.appendText(s);
	}

	/**
	 * Clears the output.
	 */
	public void clearText()
	{
		text.setText("");
	}

	/**
	 * Called on menu events.
	 */
	public boolean action(Event event, Object what)
	{
		if(!(what instanceof String))
			return super.action(event, what);
		/* We get a canonical form for the name of the menu item here;
		 * see MainFrame.action(Event, Object) for the full
		 * explanation.
		 */
		String name = ((String)what).intern();
		if(name == EDIT_CLEAR)
		{
			clearText();
			return true;
		}
		return super.action(event, what);
	}

}

/*
 * $Log: OutputWindow.java,v $
 * Revision 1.9  1999/02/28 23:45:41  cjw44
 * Added javadoc comments.
 * Added clearText() method so that loading a tape can clear the output window.
 *
 * Revision 1.8  1999/02/27 11:00:09  cjw44
 * Suppressed CRs in the output.
 *
 * Revision 1.7  1999/02/18 00:23:40  cjw44
 * Changed to monospaced font.
 *
 * Revision 1.6  1999/02/15 17:34:58  cjw44
 * Changed text area to non-editable.
 *
 * Revision 1.5  1999/02/13 06:38:41  cjw44
 * Changed size of TextArea to 20 rows by 40 columns for the aid of 800x600
 * screens.
 *
 * Revision 1.4  1999/02/12 18:42:18  cjw44
 * Added the facility to clear the output window, via Edit / Clear.
 *
 * Revision 1.3  1999/02/11 10:34:36  cjw44
 * Named the output window and fixed the size of its TextArea so that it
 * doesn't default to zero size at startup.
 *
 * Revision 1.2  1999/02/06 16:51:41  cjw44
 * Imported java.awt.*.
 * Fixed semantics of converting a char to a String.
 *
 * Revision 1.1  1999/02/01 22:14:56  cjw44
 * Initial revision
 *
 */
