package india;

/**
      * An actual EDSAC class
      * Works with a FrontEndInterface which is called to handle
      * Edsac input/output. 
      *
      * @author  Joseph Marshall
      * @version $Id: Edsac50.java,v 1.7 1999/02/15 19:06:43 cjw44 Exp $
      * @see     india.Edsac
      */


public class Edsac50 extends Edsac
{
	private long nextSequence=0; // the next SCR value
	private long lastCharacter=0; // the last character output
	private boolean letter_shift=true;
	private static final String PRINTER_CHARACTERS="PQWERTYUIOJ SZK  F\rD HNM\nLXGABCV";
	private static final String PRINTER_NUMBERS=  "0123456789  \"+(  $\r; ,.\n)/#-?:=";

	public static final int OPCODE_A=28;
	public static final int OPCODE_S=12;
	public static final int OPCODE_H=21;
	public static final int OPCODE_V=31;
	public static final int OPCODE_N=22;
	public static final int OPCODE_T=5;
	public static final int OPCODE_U=7;
	public static final int OPCODE_C=30;
	public static final int OPCODE_R=4;
	public static final int OPCODE_L=25;
	public static final int OPCODE_E=3;
	public static final int OPCODE_G=27;
	public static final int OPCODE_I=8;
	public static final int OPCODE_O=9;
	public static final int OPCODE_F=17;
	public static final int OPCODE_X=26;
	public static final int OPCODE_Y=6;
	public static final int OPCODE_Z=13;

	public Edsac50(FrontEndInterface fe)
	{
		super(fe);
	}


	long totalTime=0;

	public synchronized void clear()
	{
		accumulator=new Accumulator();
		multiplier=0;
		multiplicand=0;
		order=0;
		sequenceControl=0;// about to start
		nextSequence=0;
		for(int c=0;c<memory.length;c++)memory[c]=0;
	}

	public synchronized void initialise(int[]initialOrders)
	{
		for(int c=0;c<initialOrders.length&&c<41;c++)
		{
			memory[c]=initialOrders[c];
		}
		for(int c=0;c<initialOrders.length&&c<41;c+=2)
		{
			frontEnd.changeMemory(getMemoryLong(c),c);
		}
		sequenceControl=0; // about to start
		nextSequence=0;
	}

// Ok this is the actual work bit here

	private long getMemoryLong(int address)
	{
		// Makes a long from two adjacent memory locations
		long ret;
		// Use even address
		address=address&0xfffe;
		ret=memory[address];
		ret+=((long)memory[address+1])<<18;
//		System.out.println("RL:"+address+":"+ret+":"+memory[address]+":"+memory[address+1]);
		return ret;
	}
	private long getMemoryShort(int address)
	{
		// Makes a long from one short
		long ret;
		// in the next line I AND with 1ffff to remove the sandwich digit
		ret=((long)memory[address]&0x1ffff)<<18;
//		System.out.println("RS:"+address+":"+ret+":"+memory[address]);

		return ret;
	}
	private void setMemoryShort(int address,long value)
	{
		memory[address]&=0x20000; // remove all but the sandwich digit
		memory[address]|=(value>>18)&0x1ffff;
//		System.out.println("S:"+address+":"+value+":"+memory[address]);
	}
	private void setMemoryLong(int address,long value)
	{
		address=address&0xfffe; // Uses even address
		memory[address]=(int)(value&0x3ffff);		// this one gets sandwich digit
		memory[address+1]=(int)((value>>18)&0x1ffff);	// only 17 bits here
//		System.out.println("L:"+address+":"+value+":"+memory[address]+":"+memory[address+1]);
	}

	public synchronized void singleStep(Tape ta)throws EdsacException
	{
		// Increment the sequence control reg and load the order
		sequenceControl=nextSequence&0x3ff;
		frontEnd.changeSequenceControl(sequenceControl);
		nextSequence=sequenceControl+1;
		order=memory[(int)sequenceControl];
		frontEnd.changeOrder(order);

		// Split the instruction into the seperate fields
		// Instruction is (high bits first) :-
		// 5 bits OPCODE
		// 1 bit spare
		// 10 bit ADDRESS (except for shifting which uses all the instruction bits)
		// 1 bit LENGTH (0= single 1=double)

		int opcode=((int)order>>12)&0x1f; // the instruction type
		int address=((int)order>>1)&0x3ff; // the address in the instruction
		boolean length=false;
		if((order&0x1)!=0)length=true; // the length bit


		switch(opcode)
		{
			case OPCODE_A://Add
				if(length)multiplicand=getMemoryLong(address);
				 else multiplicand=getMemoryShort(address);
				accumulator.add(multiplicand);
				frontEnd.changeMultiplicand(multiplicand);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_S://Subtract
				if(length)multiplicand=getMemoryLong(address);
				 else multiplicand=getMemoryShort(address);
				accumulator.sub(multiplicand);

				frontEnd.changeMultiplicand(multiplicand);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_H://Load multiplier
				if(length)multiplier=getMemoryLong(address);
				 else multiplier=getMemoryShort(address);

				frontEnd.changeMultiplier(multiplier);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_V://Multiply add
				if(length)multiplicand=getMemoryLong(address);
				 else multiplicand=getMemoryShort(address);
				accumulator.mulAdd(multiplier,multiplicand);

				frontEnd.changeMultiplicand(multiplicand);
				frontEnd.changeAccumulator(accumulator);
			
				break;
			case OPCODE_N://Multiply sub
				if(length)multiplicand=getMemoryLong(address);
				 else multiplicand=getMemoryShort(address);
				accumulator.mulSub(multiplier,multiplicand);
				frontEnd.changeMultiplicand(multiplicand);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_T:// Store and zero accumulator
				if(length)setMemoryLong(address,accumulator.getHigh());
				  else setMemoryShort(address,accumulator.getHigh());
				accumulator=new Accumulator();
				frontEnd.changeAccumulator(accumulator);
				frontEnd.changeMemory(getMemoryLong(address),address);
				break;
			case OPCODE_U:// Store 
				if(length)setMemoryLong(address,accumulator.getHigh());
				  else setMemoryShort(address,accumulator.getHigh());
				frontEnd.changeMemory(getMemoryLong(address),address);
				break;
			case OPCODE_C:// accumulator+= multiplier (bitwise)AND multiplicand
				if(length)multiplicand=getMemoryLong(address);
				 else multiplicand=getMemoryShort(address);
				accumulator.and(multiplier,multiplicand);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_R:// Right shift
				// shifts depending on the lowest bit set in the entire order
				accumulator.shiftRight((int)order);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_L:// Left shift
				// shifts depending on the lowest bit set in the entire order
				accumulator.shiftLeft((int)order);
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_E://Jump if positive
				if(accumulator.isPositive())nextSequence=address;
				break;
			case OPCODE_G://Jump if negative
				if(!accumulator.isPositive())nextSequence=address;
				break;
			case OPCODE_I://Input value from tape
				long val=ta.nextValue();
				val=val<<18; // Put it in the top 17 bits
				if(length)
				{
					setMemoryLong(address,val);
				}else
				{
					setMemoryShort(address,val);
				}
				frontEnd.changeMemory(getMemoryLong(address),address);
				break;
			case OPCODE_O://Output value to printer
				if(length)
				{
					// take the high half of the long address
					address=address|1;
				}
				lastCharacter=getMemoryShort(address)>>30;
				if(lastCharacter==16)break;
				else if(lastCharacter==11)letter_shift=false;
				else if(lastCharacter==15)letter_shift=true;
				else if(letter_shift)frontEnd.outputText(PRINTER_CHARACTERS.charAt((int)lastCharacter));
				else frontEnd.outputText(PRINTER_NUMBERS.charAt((int)lastCharacter));
				break;
			case OPCODE_F://Load last value output to printer into memory
				if(length)
				{
					setMemoryShort(address|1,((long)lastCharacter)<<30);
					setMemoryShort(address&0xfffe,0);
				}else setMemoryShort(address,((long)lastCharacter)<<30);
				frontEnd.changeMemory(getMemoryLong(address),address);
				break;
			case OPCODE_X://Do nothing
			// a no-op
				break;
			case OPCODE_Y:// Round the accumulator off to 35 bits
				accumulator.round();
				frontEnd.changeAccumulator(accumulator);
				break;
			case OPCODE_Z:// Ring the bell and stop
				throw(new EdsacHaltedException("Stopped: Bell"));
				//break;
			default:
				throw(new EdsacException("Stopped: Instruction Unknown:"+order));
			
		}

		switch(opcode) // Instruction timings
		{
			case OPCODE_A://Add
			case OPCODE_S://Subtract
			case OPCODE_H://Load multiplier
			case OPCODE_T:// Store and zero accumulator
			case OPCODE_U:// Store 
			case OPCODE_C:// accumulator+= multiplier (bitwise)AND multiplicand
			case OPCODE_R:// Right shift
			case OPCODE_L:// Left shift
			case OPCODE_E://Jump if positive
			case OPCODE_G://Jump if negative
			case OPCODE_F://Load last value output to printer into memory
			case OPCODE_X://Do nothing
			case OPCODE_Y:// Round the accumulator off to 35 bits
			case OPCODE_Z:// Ring the bell and stop
				totalTime+=15;
				break;
			case OPCODE_V://Multiply add
			case OPCODE_N://Multiply sub
				totalTime+=60;
				break;
			case OPCODE_I://Input value from tape
				totalTime+=200;
				break;
			case OPCODE_O://Output value to printer
				totalTime+=1500;
				break;
		}
		frontEnd.endRead();
	}


	public synchronized long getMultiplier()
	{
		return multiplier;
	}
	public synchronized long getMultiplicand()
	{
		return multiplicand;
	}
	public synchronized long getOrder()
	{
		return order;
	}
	public synchronized long getSequenceControl()
	{
		return sequenceControl;
	}
	public synchronized Accumulator getAccumulator()
	{
		return new Accumulator(accumulator);
	}
	public synchronized void setAccumulator(Accumulator acc)
	{
		accumulator=acc;
	}
	public synchronized long getEdsacTime()
	{
		return totalTime;
	}
	public synchronized void zeroEdsacTime()
	{
		totalTime=0;
	}

	public synchronized int[]getMemoryTank(int tank) throws EdsacException
	{
		int[]tankData=new int[32];
		if(tank>31 || tank<0)
		  throw(new EdsacException("Bad tank number ["+tank+"] in getMemoryTank") );
		try
		{
			System.arraycopy(memory,32*tank,tankData,0,32);
		}
		catch(Exception e)
		{
			throw(new EdsacException("Unexpected exception ["+e+"] in getMemoryTank") );
		}
		return tankData;
	}


}

/*
 * $Log: Edsac50.java,v $
 * Revision 1.7  1999/02/15 19:06:43  cjw44
 * Reinstated changes from revision 1.5 which were lost in an accident with
 * RCS.
 *
 * Revision 1.6  1999/02/11 18:24:13  jm266
 * Added a check to make the memory loop round rather than
 * array index fault.
 *
 * Revision 1.5  1999/02/11 02:19:34  cjw44
 * Changed initialise() to update the FrontEnd with the initial orders.
 *
 * Revision 1.4  1999/02/10 00:01:18  jm266
 * Made the multiplier update correctly
 *
 * Revision 1.3  1999/02/08 23:01:26  jm266
 * fixed a bug in displaying
 *
 * Revision 1.2  1999/02/08 13:41:14  jm266
 * Added timings for instructions and put on the log and javadoc
 * class comment.
 *
 */
