/* Note the dollar ID dollar in the comment below will get substituted by rcs
 * for the current revision number, file size etc. when the file is checked out */

/**
      * A class representing the 71 bit EDSAC accumulator.
      * Only used by the Edsac class and derived classes.
      * @author  Joseph Marshall
      * @version $Id: Accumulator.java,v 1.10 1999/03/09 04:09:21 cjw44 Exp $
      * @see     india.Edsac
      */


package india;

public class Accumulator
{

	// Data
	
    /**
     * The low 36 bits of the 71 bit accumulator
     */
	long loWord=0;
    /**
     * The high 35 bits of the 71 bit accumulator
     */
	long hiWord=0;

	// Constructor

    /**
      * This creates an instance of the Accumulator object.
      */
  public Accumulator() 
	{
	}

	public Accumulator(Accumulator a)
	{
		hiWord=a.hiWord;
		loWord=a.loWord;
	}


	// Manipulation methods

    /**
      * This adds a 35 bit double value to the accumulator.
      * @param value the value to be added to the accumulator
      */
	public void add(long value)
	{
		hiWord+=value;
		hiWord=hiWord&0x7ffffffffL; // Mustn't let it get bigger than 35 bits
		// This should never need to change the low word
		// of the accumulator		
	}
    /**
      * This adds a 17 bit single value to the accumulator.
      * @param value the value to be added to the accumulator
      */
	public void add(int value)
	{
				// Shift left 18 so that the highest bit is at the top of the high word
		// of the accumulator
		hiWord+=(long)value<<18;
		hiWord=hiWord&0x7ffffffffL; // Mustn't let it get bigger than 35 bits
		// This should never need to change the low word
		// of the accumulator
	}
    /**
      * This subtracts a 35 bit double value from the accumulator.
      * Not really needed as could just negate, but might save
      * a bit of hassle.
      * @param value the value to be subtracted from the accumulator
      */
	public void sub(long value)
	{
		// Shift left 18 so that the highest bit is at the top of the high word
		// of the accumulator
		hiWord-=value;
		hiWord=hiWord&0x7ffffffffL; // Mustn't let it get bigger than 35 bits
		// This should never need to change the low word
		// of the accumulator

	}
	
    /**
      * This subtracts a 17 bit single value from the accumulator.
      * Not really needed as could just negate, but might save
      * a bit of hassle.
      * @param value the value to be subtracted from the accumulator
      */
	public void sub(int value)
	{
		// Shift left 18 so that the highest bit is at the top of the high word
		// of the accumulator
		hiWord-=(long)value<<18;
		hiWord=hiWord&0x7ffffffffL; // Mustn't let it get bigger than 35 bits
		// This should never need to change the low word
		// of the accumulator
	}


    /**
      * This adds the product of 2 35 bit values to the accumulator.
      * @param value1 the first value to be multiplied
      * @param value2 the second value to be multiplied
      */
	public void mulAdd(long value1,long value2)
	{
		// I split each value into two longs, holding 18 & 17 bits respectively
		// Then these are multiplied together and shifted appropriately to add them
		// into each half of the accumulator

	  // All shifts are sign extended so that negative numbers multiply
	  // correctly
		value1=(value1<<29)>>29;// Sign extend - makes multiply work
		value2=(value2<<29)>>29;// Sign extend (29=64-35 )
		
		long low1=value1&0x3ffff;	// The low 18 bits
		long high1=value1>>18;		// The high 17 bits
		long low2=value2&0x3ffff; 	// The low 18 bits
		long high2=value2>>18;		// The high 17 bits

		// Use formula (A*.5^16+B*.5^34)(C*.5^16+B*.5^34)
		//	= AC*.5^32+(AD+BC)*.5^50+BD*.5^68

		// Remember that accumulator=hiWord*.5^34+loWord*.5^70

		long AC=high1*high2;
		long AD=high1*low2;
		long BC=low1*high2;
		long BD=low1*low2;

		long loRet=loWord;
		long hiRet=hiWord;

		hiRet+=AC<<2;
		hiRet+=(BD>>(34));

		hiRet+=(AD+BC)>>(50-34);
		loRet+=((AD+BC)<<(70-50))&0xfffffffffL;
		loRet&=0xfffffffffL;
		if(loRet<loWord)
		{
			hiRet++;
		}
		
		loWord=loRet;
		loRet+=(BD<<2)&0xfffffffffL;
		loRet&=0xfffffffffL;
		if(loRet<loWord)
		{
			hiRet++;
		}
		loWord=loRet&0xfffffffffL;
		hiWord=hiRet&0x7ffffffffL;
	}
    /**
      * This subtracts the product of 2 35 bit values from the accumulator.
      * @param value1 the first value to be multiplied
      * @param value2 the second value to be multiplied
      */
	public void mulSub(long value1,long value2)
	{
	  // All shifts are sign extended so that negative numbers multiply
	  // correctly
				value1=(value1<<29)>>29;// Sign extend - makes multiply work
		value2=(value2<<29)>>29;// Sign extend (29=64-35 )


		// I split each value into two longs, holding 18 & 17 bits respectively
		// Then these are multiplied together and shifted appropriately to add them
		// into each half of the accumulator
		long low1=value1&0x3ffff;	// The low 18 bits
		long high1=value1>>18;		// The high 17 bits
		long low2=value2&0x3ffff; 	// The low 18 bits
		long high2=value2>>18;		// The high 17 bits



		// Use formula (A*.5^16+B*.5^34)(C*.5^16+B*.5^34)
		//	= AC*.5^32+(AD+BC)*.5^50+BD*.5^68

		// Remember that accumulator=hiWord*.5^34+loWord*.5^70

		long AC=high1*high2;
		long AD=high1*low2;
		long BC=low1*high2;
		long BD=low1*low2;

		long loRet=loWord;
		long hiRet=hiWord;

		hiRet-=AC<<2;

		hiRet-=(BD>>(34));
		hiRet-=(AD+BC)>>(50-34);

		loRet-=((AD+BC)<<(70-50))&0xfffffffffL;
		loRet&=0xfffffffffL;
		if(loRet>loWord)
		{
			hiRet--;
		}


		loWord=loRet;
		loRet-=(BD<<2)&0xfffffffffL;
		loRet&=0xfffffffffL;

		if(loRet>loWord)
		{
			hiRet--;
		}
		loWord=loRet&0xfffffffffL;
		hiWord=hiRet&0x7ffffffffL;
	}


    /**
      * This performs a logical AND operation on two values
      * and adds to the accumulator.
      * @param value1 the first value
      *	@param value2 the value to AND it with.
      */
	public void and(long value1,long value2)
	{
		this.add(value1&value2);
	}
    /**
      * This shifts the accumulator left.
      * Works in a strange way for historical reasons
      * and to ease EDSAC instruction decoding.
      * @param value the shift control value.
      *	The accumulator is shifted left depending on the lowest
      *	bit which is set in this value. For example 1 will shift
      *	by 1 bits, 8 will shift left 4 bits. 
      */
	public void shiftLeft(long value)
	{
		int shiftnum=0;
		if(value!=0)
		{
			shiftnum++;
			while((value&1)!=1)
			{
				value=value>>1;
				shiftnum++;
			}
		}
		for(int c=0;c<shiftnum;c++)
		{

			hiWord=hiWord<<1;
			hiWord|=((loWord&0x800000000L)>>35); // Add the 36th bit of loWord 
			loWord=loWord<<1;
		}

		loWord=loWord&0xfffffffffL;
		hiWord=hiWord&0x7ffffffffL;
	}

    /**
      * This shifts the accumulator right.
      * Works in a strange way for historical reasons
      * and to ease EDSAC instruction decoding.
      * @param value the shift control value.
      *	The accumulator is shifted right depending on the lowest
      *	bit which is set in this value. For example 1 will shift
      *	by 1 bits, 8 will shift left 4 bits. 
      */
	public void shiftRight(long value)
	{
		int shiftnum=0;
		if(value!=0)
		{
			shiftnum++;
			while((value&1)!=1)
			{
				value=value>>>1;
				shiftnum++;
			}
		}
		for(int c=0;c<shiftnum;c++)
		{
			loWord=loWord>>>1;
			loWord|=(hiWord&1L)<<35; // Add the 1st bit of hiWord;
			hiWord=(hiWord&0x400000000L)|(hiWord>>>1);
		}
		loWord=loWord&0xfffffffffL;
		hiWord=hiWord&0x7ffffffffL;
	}

    /**
      * Round the number in the accumulator to 34 bits (and a sign bit)
      * just adds 1 to the 36th bit
      */
	public void round()
	{
		loWord+=0x800000000L;
		loWord&=0xfffffffffL;
		if(loWord<0x800000000L)hiWord++;
	}


	// Query methods

    /**
      * Returns the sign of the value in the accumulator.
      * @return the sign of the accumulator value
      */
	public boolean isPositive()
	{
		if((hiWord&0x400000000L)==0)return true;
		return false;		
	}

    /**
      * Returns the low half (36 bits) of the accumulator.
      * @return the low 36 bits of the accumulator.
      */
	public long getLow()
	{
				return loWord;
	}
    /**
      * Returns the high half (35 bits) of the accumulator.
      * @return the high 35 bits of the accumulator.
      */
	public long getHigh()
	{
				return hiWord;
	}

	/**
	 * Compare to another Accumulator for equality
	 */
	public boolean equals(Accumulator a)
	{
		return a != null && loWord == a.loWord && hiWord == a.hiWord;
	}

// toString method for test purposes
	public String toString()
	{
		String s="";
		for(int c=34;c>=0;c--)
		  if((hiWord&(1L<<c))!=0)s=s+"1";
		    else s=s+"0";
		s=s+" ";
		for(int c=35;c>=0;c--)
		  if((loWord&(1L<<c))!=0)s=s+"1";
		    else s=s+"0";
		s=s+"\n01234567890123456789012345678901234 56789012345678901234567890123456789012345678";
		return s;
	}

    static final String digits="0123456789ABCDEF";

    public String toHexString()
    {

	long num=loWord;

	String hexStr=new String();

	for(int c=0;c<36;c+=4)
	    {
		hexStr=digits.charAt((int)(num&0xf))+hexStr;
		num>>=4;
	    }
	num=hiWord;
	for(int c=0;c<35;c+=4)
	    {
		hexStr=digits.charAt((int)(num&0xf))+hexStr;
		num>>=4;
	    }
	return hexStr;
    }

    public String toDecString()
    {
	if(loWord == 0 && hiWord == 0) return "0";
	// 23 characters is more than large enough for 71 bits
	StringBuffer buf = new StringBuffer(23);
	long low = loWord, high = hiWord;
	while(high > 0)
	{
	    long rem = (high * 6 + low) % 10;	// (2 ^ 36) % 10 == 6
	    buf.append(Character.forDigit((int)rem, 10));
	    low = ((high % 10) * 0x1000000000L + low - rem) / 10;
	    high /= 10;
	}
	if(low == 0)
	    return buf.reverse().toString();
	else
	    return String.valueOf(low) + buf.reverse().toString();
    }

    public String toFractionString()
    {
	if(loWord == 0 && hiWord == 0) return "0.0";
	// This is somewhat less complicated than toDecString(), since we
	// can work using the built-in types - i.e. double.
	boolean sign = (hiWord & (1L << 34)) != 0;
	double place = 0.5;
	double total = 0.0;
	for(long bit = 1L << 33; bit != 0; bit >>>= 1)
	{
	    if((hiWord & bit) != 0) total += place;
	    place /= 2;
	}
	for(long bit = 1L << 35; bit != 0; bit >>>= 1)
	{
	    if((loWord & bit) != 0) total += place;
	    place /= 2;
	}
	return String.valueOf(sign ? -total : total);
    }

    public static void main(String []args)
    {
	Accumulator z=new Accumulator();
	z.loWord=0x745453456L;
	z.hiWord=0x34fff4567L;
	System.out.println(z.toHexString());
    }

}

/*
 * $Log: Accumulator.java,v $
 * Revision 1.10  1999/03/09 04:09:21  cjw44
 * Added equals(Accumulator) method and a method (toFractionString()) to
 * convert to a string representation of the appropriate fixed-point fraction.
 *
 * Revision 1.9  1999/03/07 23:58:34  cjw44
 * Added conversion to decimal.
 *
 * Revision 1.8  1999/03/03 17:42:41  mnw21
 * Fixed log
 *
 * Revision 1.7  1999/02/23 15:44:20  jm266
 * Added the toHexString method.
 *
 * Revision 1.6  1999/02/03 23:58:49  jm266
 * Sorted out the stupid comments on the shift instructions
 *
 * Revision 1.5  1999/02/03  23:07:35  jm266
 * removed a few possible problems with >> instead of >>>
 *
 * Revision 1.4  1999/01/29  19:53:02  jm266
 * Now with REAL methods
 *
 * Revision 1.3  1999/01/23  01:01:57  jm266
 * Corrected capitalisation of class methods
 *
 * Revision 1.2  1999/01/22  17:56:23  mnw21
 * No change.
 *
 * Revision 1.1  1999/01/21 16:44:34  jm266
 * Initial revision
 *
 */
