package india;

import java.net.*;

/**
      * The interface between a FrontEnd object and the 
      * Edsac simulator object. Function names
      * are self explanatory. Each function is called
      * when the Edsac class wants to update the interface.
      * 
      * @author  Joseph Marshall
      * @version $Id: FrontEndInterface.java,v 1.10 1999/02/15 13:26:56 cjw44 Exp $
      * @see     india.FrontEnd
      * @see     india.Edsac
      */

public interface FrontEndInterface
{

	/**
	 * Notifies the front end that the value of the 71-bit accumulator has
	 * changed.
	 */
	public void changeAccumulator(Accumulator a); 

	/**
	 * Notifies the front end that the value of the 35-bit multiplier
	 * register has changed.
	 */
	public void changeMultiplier(long value); 

	/**
	 * Notifies the front end that the value of the 10-bit sequence control
	 * register has changed.
	 */
	public void changeSequenceControl(long value); 

	/**
	 * Notifies the front end that the value of the 35-bit multiplicand
	 * register has changed.
	 */
	public void changeMultiplicand(long value); 

	/**
	 * Notifies the front end that the value of the 17-bit order register
	 * has changed.
	 */
	public void changeOrder(long value);

	/**
	 * Outputs a character on the teleprinter. The Edsac class handles the
	 * conversion from Edsac character codes to characters.
	 */
	public void outputText(char ch);

	/**
	 * Notifies the front end that the contents of memory have changed.
	 * @param	memory	35-bit value of changed memory location.
	 * @param	address	10-bit address of changed memory location;
	 *	will be rounded down to an even number if it is not already.
	 */
	public void changeMemory(long memory, int address);

	/**
	 * Notifies the front end that the simulator has finished updating
	 * the Edsac's state, and that the display can be updated. Calling
	 * this method guarantees that the display will be updated; depending
	 * on this interface's implementation, it may already have been
	 * updated beforehand.
	 */
	public void endRead();

        /**
         * Load a tape in and run it.
         */
        public void loadTape(Tape t);

        /**
         * Load a tape in (and run it unless runMe is false).
         */
        public void loadTape(Tape t,boolean runMe);

       /**
         * Makes a new editor window with this text and shows the window
         * if the FrontEnd is showing.
         * @param       text    Text with which to initialize editor window.
         * @param       name    Name of editor window.
         */
        public Editor newEditor(String text, String name);

        /**
         * Removes a window - called when a window is closed.
         * removeEditor() does the hide() operation on the editor window.
         */

        public void removeEditor(Editor ed);

        /**
         * Gets a list of all the windows.
         */
        public Editor[] getEditors();

	/**
         * Determines whether we are running in an applet.
         * Useful for things like the code editor which may need to save
         * files.
         */
        public boolean isApplet();

	/**
	 * Gets the base URL from which example programs are to be downloaded.
	 */
	public URL getBase() throws MalformedURLException;

}

/*
 * $Log: FrontEndInterface.java,v $
 * Revision 1.10  1999/02/15 13:26:56  cjw44
 * Added support for retrieving the simulator's base URL.
 *
 * Revision 1.9  1999/02/05 18:34:24  jm266
 * made newEditor return the new Editor it has made
 *
 * Revision 1.8  1999/02/05 16:23:06  jm266
 * Added return of Editor to newEditor method
 *
 * Revision 1.7  1999/02/05 15:40:49  jm266
 * Added the methods in FrontEnd needed by the editor.
 *
 * Revision 1.6  1999/02/01 21:32:14  cjw44
 * Fixed the odd syntax error and renamed a few variables.
 *
 * Revision 1.5  1999/01/31 17:57:13  cjw44
 * Added javadoc comments to each method.
 *
 * Revision 1.4  1999/01/31 17:41:30  cjw44
 * Removed startRead(), on the grounds that we declared it unnecessary at our
 * meeting on Thursday.
 *
 * Revision 1.3  1999/01/29 20:05:15  jm266
 * sorted out the comments around the Log tag at the end
 *
 * Revision 1.2  1999/01/25  14:59:14  cjw44
 * Log added. :)
 * An explanation of the sequence of events would be useful - for instance, is
 * the front end entitled to update tank displays as soon as it retrieves a
 * changeXxx() call, or does it have to wait until endRead()?
 * And what does startRead() do? Lock the display?
 * (cjw44)
 */
