/* Note the dollar ID dollar in the comment below will get substituted by rcs
 * for the current revision number, file size etc. when the file is checked out */

/**
      * A class representing the Edsac execution and memory tanks.
      * Works with a FrontEndInterface which is called to handle
      * Edsac input/output. 
      *
      * JM: I'll derive Edsac50.java from this for our Edsac 1949-50 Version
      * @author  Joseph Marshall
      * @version $Id: Edsac.java,v 1.4 1999/03/03 17:29:58 mnw21 Exp $
      * @see     india.FrontEndInterface
      * @see     india.FrontEnd
      */

package india;

abstract public class Edsac
{
    /**
     * The first set of initial orders for the EDSAC.
     * These function as the bootloader and a crude assembler.
     *
     * @see #INITIAL_ORDERS_2
     */


	public final static int []INITIAL_ORDERS_1=	{
0x5000,// 0:T0F
0x15004,// 1:H2F
0x5000,// 2:T0F
0x300c,// 3:E6F
0x2,// 4:P1F
0xa,// 5:P5F
0x5000,// 6:T0F
0x8000,// 7:I0F
0x1c000,// 8:A0F
0x4020,// 9:R16F
0x5001,// 10:T0D
0x8004,// 11:I2F
0x1c004,// 12:A2F
0xc00a,// 13:S5F
0x302a,// 14:E21F
0x5006,// 15:T3F
0x1f002,// 16:V1F
0x19010,// 17:L8F
0x1c004,// 18:A2F
0x5002,// 19:T1F
0x3016,// 20:E11F
0x4008,// 21:R4F
0x1c002,// 22:A1F
0x19001,// 23:L0D
0x1c000,// 24:A0F
0x503e,// 25:T31F
0x1c032,// 26:A25F
0x1c008,// 27:A4F
0x7032,// 28:U25F
0xc03e,// 29:S31F
0x1b00c// 30:G6F
};

    /**
     * The first set of initial orders for the EDSAC.
     * These function as the bootloader and a crude assembler.
     *
     * @see #INITIAL_ORDERS_1
     */

	public final static int []INITIAL_ORDERS_2=
{
0x5000,// 0:T0F
0x3028,// 1:E20F
0x2,// 2:P1F
0x7004,// 3:U2F
0x1c04e,// 4:A39F
0x4008,// 5:R4F
0x1f000,// 6:V0F
0x19010,// 7:L8F
0x5000,// 8:T0F
0x8002,// 9:I1F
0x1c002,// 10:A1F
0xc04e,// 11:S39F
0x1b008,// 12:G4F
0x19001,// 13:L0D
0xc04e,// 14:S39F
0x3022,// 15:E17F
0xc00e,// 16:S7F
0x1c046,// 17:A35F
0x5028,// 18:T20F
0x1c000,// 19:A0F
0x15010,// 20:H8F
0x1c050,// 21:A40F
0x5056,// 22:T43F
0x1c02c,// 23:A22F
0x1c004,// 24:A2F
0x502c,// 25:T22F
0x3044,// 26:E34F
0x1c056,// 27:A43F
0x3010,// 28:E8F
0x1c054,// 29:A42F
0x1c050,// 30:A40F
0x3032,// 31:E25F
0x1c02c,// 32:A22F
0x5054,// 33:T42F
0x8051,// 34:I40D
0x1c051,// 35:A40D
0x4020,// 36:R16F
0x5051,// 37:T40D
0x3010,// 38:E8F
0xb,// 39:P5D
0x1// 40:P0D
	};


    /**
     * The EDSAC accumulator with associated mathematical functions.
     * 
     * @see india.Accumulator
     */

	protected Accumulator accumulator;	// The accumulator class


    /**
     * The EDSAC multiplier register (35 bits).
     * This is used to load the operand for the
     * multiply instruction only.
     */

	protected long multiplier;

    /**
     * The EDSAC multiplicand register (35 bits).
     * This is not used explicitly ever, just loaded into
     * for multiply and various other instructions.
     */

	protected long multiplicand;

    /**
     * The EDSAC order tank (17 bits).
     * This stores the current instruction.
     */
	
	protected long order;

    /**
     * The EDSAC sequence control register (10 bits).
     * This is the equivalent of the program counter
     * on a modern machine.
     */
	
	protected long sequenceControl;


    /**
     * The EDSAC memory (512 * 35 bit word split into 18,17 bits)
     */

	protected int[]memory;

    /**
     * The FrontEndInterface used by the simulator for output.
     * @see india.FrontEndInterface
     */

    protected FrontEndInterface frontEnd;


    /**
      * This creates an instance of the EDSAC object.
      * Memory and registers begin cleared to zero.
      */

          
	public Edsac(FrontEndInterface fe)
	{

		frontEnd=fe;
		memory=new int[1024];
		clear();
	}

    /**
     * Does the equivalent to the clear button on the EDSAC.
     * The clear button sets all memory and registers to zero.
     */	

	public abstract synchronized void clear();



	/**
	 * Does the equivalent to the singleStep button on the EDSAC.
	 * This will also be used for running the program normally.
	 * Executes one instruction and then returns.
	 * 
     * @param     tape  the Tape object storing the currently inserted tape and the position
     *			  on the tape.
	 * @exception EdsacException
	 *              If the Edsac rings the bell and stops,
	 *              or if an undefined instruction occurs.
	 */
	public abstract synchronized void singleStep(Tape ta)throws EdsacException;

	/**
	 * Does the equivalent of the start button on the EDSAC.
	 * Not named start to avoid confusion with thread / applet
	 * start methods.
	 * <p> The start button loaded the initial orders into memory
	 * from switches and set the Sequence Control Register (Program Counter)
	 * to zero.
	 * <p> This loads the second version of the initial orders. To load a different
	 * version use <a href="#initialise(FrontendInterface fe,int[]initialOrders)">initialise(FrontendInterface fe,int[]initialOrders)</a>
	 */
	public void initialise()
	{
		initialise(this.INITIAL_ORDERS_2);
	}

	/**
	 * Does the equivalent of the start button on the EDSAC.
	 * Not named start to avoid confusion with thread / applet
	 * start methods.
	 * <p> The start button loaded the initial orders into memory
	 * from switches and set the Sequence Control Register (Program Counter)
	 * to zero.
	 * <p> Use the constants INITIAL_ORDERS_1 and INITIAL_ORDERS_2 to load the first
	 * and second versions of the initial orders. It is possible to provide new initial orders
	 * for example to read a binary program from the tape.
     * @param     initialOrders An array with the instructions of the initial orders.
     *          	Only the first 41 items in the array will be used as no more
     *				switches were provided on the original.
	 */

	public abstract synchronized void initialise(int[]initialOrders);



    /**
     * Returns the value of the multiplier register (35 bits).
     * 
     *
     * @return    the multiplier value
     */

	public abstract synchronized long getMultiplier();

    /**
     * Returns the value of the multiplicand register (35 bits).
     * 
     *
     * @return    the multiplicand value
     */
	
	public abstract synchronized long getMultiplicand();

    /**
     * Returns the value of the order tank (35 bits).
     * This is the current instruction being executed.
     *
     * @return    the value in the order tank
     */
	
	public abstract synchronized long getOrder();

    /**
     * Returns the value of the sequence control register (35 bits).
     * This is the equivalent of the program counter.
     *
     * @return    the value of the sequence control register.
     */
	
	public abstract synchronized long getSequenceControl();

    /**
     * Returns the value of the accumulator (71 bits).
     * This is returned 
     *
     * @return    an Accumulator class representing the accumulator
     */
	
	public abstract synchronized Accumulator getAccumulator();

    /**
     * Sets the Accumulator.
     * The only time this is needed is
     * when the telephone dial is used as the dial puts the value into the accumulator
     * and then does the same as the RESET button.
     * @param accum    the accumulator to set as the current one.
     */

	public abstract synchronized void setAccumulator(Accumulator a);


    /**
     * Returns the current EDSAC time in milliseconds.
     * 
     * @return    an Accumulator class representing the accumulator
     */



	public abstract synchronized long getEdsacTime();

    /**
     * Sets the EDSAC time to zero.
     */

	public abstract synchronized void zeroEdsacTime();

    /**
     * Returns the contents of a memory tank.
     * A single memory tank contains 16 35 bit words
     * which are stored as a low word of 18 bits
     * and a high word of 17 bits.
     * There are 32 tanks numbered 0 to 31.
     * <p> Note that this should only be used
     * by the interface when a whole tank is updated. For example
     * when the tank being viewed is changed.
     *
     * @param     tank  the number of the tank to read
     * @return    the contents of the tank.
     * @exception EdsacException
     *              if the tank number is not in the range <code>0</code> 
     *              to <code>32</code>.
     */

	public abstract synchronized int[]getMemoryTank(int tank) throws EdsacException;
}

/*
 * $Log: Edsac.java,v $
 * Revision 1.4  1999/03/03 17:29:58  mnw21
 * Fixed a few comments
 *
 * Revision 1.3  1999/01/29 19:53:45  jm266
 * Filled in the initial orders
 *
 * Revision 1.2  1999/01/23  16:09:58  jm266
 * Spelt synchronized in USA English etc.
 *
 * Revision 1.1  1999/01/23  12:30:50  jm266
 * Initial revision
 *
 */
