package india;

import java.applet.Applet;
import java.awt.*;
import java.awt.image.*;

/**
 * A clock which displays the current Edsac time.
 * @author	Dom Penfold
 * @version	$Id: Clock.java,v 1.4 1999/03/05 08:32:42 cjw44 Exp $
 */

public class Clock extends FloatingComponent
{

	private int startx, starty;
	private int centre;
	private int diameter;
	private int armlength;

	private Image face, offscreenImage;
	private Graphics osG;

	// The two variables used to store time.
	// oldTime 	= time last frame, used for clearing off the old clock
	//		  arms
	// time		= the time this frame.
	private long oldTime=0,time=0;

	// These two arrays are used for internal calculations when drawing the
	// arms of the clock.

	// More efficient to create the variables here so that they aren't
	// created and destroyed every time the function is called.
	private int x[] = new int[3];
	private int y[] = new int[3];

	// IMPORTANT VARIABLE TO CHANGE.. timeDiv specifies the "real" time
	// division between each value of the time variable
	// i.e. timeDiv=1 means that an increment of one in the variable time
	// is 1 second.
	// timeDiv=1000 means that an increment of one thousand in the variable
	// time is 1 second. (i.e. millisecond timer)
	private float timeDiv = 10000;

	private StatusBar status;
	private boolean peeping = false;

	/**
	 * Creates a new clock with the specified image.
	 * @param	status	The status bar to which to write the current
	 *			time if the mouse is hovering over us.
	 * @param	face	An image of an empty clock face.
	 * @param	startx	The x-coordinate of the top left of the image.
	 * @param	starty	The y-coordinate of the top left of the image.
	 * @param	diameter	The diameter of the clock face.
	 */
	public Clock(StatusBar status, Image face,
			int startx, int starty, int diameter)
		throws ImageLoadingException
	{
		this.startx = startx;
		this.starty = starty;
		centre=diameter/2;
		this.diameter=diameter;
		armlength=(diameter-13)/2;
		this.status = status;

		this.face = createImage(new FilteredImageSource(
			face.getSource(), new CropImageFilter(
				startx, starty, diameter, diameter)));
		Image[] imgs = {this.face};
		checkImages(imgs);
	}

	public synchronized boolean mouseMove(Event e, int x, int y)
	{
		if(((centre - x) * (centre - x) +
			(centre - y) * (centre - y) <=
				diameter * diameter / 4))
		{
			if(!peeping)
			{
				peeping = true;
				status.setOther("EDSAC time", getTimeString());
			}
		}
		else
		{
			if(peeping)
			{
				peeping = false;
				status.clearPeep();
			}
		}
		return true;
	}

	public synchronized boolean mouseDrag(Event e, int x, int y)
	{
		if(((centre - x) * (centre - x) +
			(centre - y) * (centre - y) <=
				diameter * diameter / 4))
		{
			if(!peeping)
			{
				peeping = true;
				status.setOther("EDSAC time", getTimeString());
			}
		}
		else
		{
			if(peeping)
			{
				peeping = false;
				status.clearPeep();
			}
		}
		return true;
	}

	public synchronized boolean mouseEnter(Event e, int x, int y)
	{
		if(((centre - x) * (centre - x) +
			(centre - y) * (centre - y) <=
				diameter * diameter / 4))
		{
			if(!peeping)
			{
				peeping = true;
				status.setOther("EDSAC time", getTimeString());
			}
		}
		else
		{
			if(peeping)
			{
				peeping = false;
				status.clearPeep();
			}
		}
		return true;
	}

	public synchronized boolean mouseExit(Event e, int x, int y)
	{
		if(peeping)
		{
			peeping = false;
			status.clearPeep();
		}
		return true;
	}

	public long getTime()
	{
		return time;
	}

	public synchronized String getTimeString()
	{
		long secondsTime = time / 10000;
		int seconds = (int)(secondsTime % 60);
		String secStr = seconds < 10 ?
			"0" + seconds : String.valueOf(seconds);
		long minutesTime = secondsTime / 60;
		int minutes = (int)(minutesTime % 60);
		String minStr = minutes < 10 ?
			"0" + minutes : String.valueOf(minutes);
		long hours = minutesTime / 60;
		return hours + ":" + minStr + ":" + secStr;
	}

	public synchronized void setTime(long time)
	{
		if(time < 0)
			throw new IllegalArgumentException("negative time");
		this.time=time;
		paint(getGraphics());
		if(peeping)
			status.setOther("EDSAC time", getTimeString());
	}

	// Separate function to reduce codesize,
	// Graphics gfx is the context drawn to.
	public void drawArm(Graphics gfx, double length, double width,
		double time)
	{
		x[0]=(int)(centre+(length*0.8)*
			Math.sin((double)(time-width)*Math.PI/30));
		x[1]=(int)(centre+length*
			Math.sin((double)time*Math.PI/30));
		x[2]=(int)(centre+(length*0.8)*
			Math.sin((double)(time+width)*Math.PI/30));
		y[0]=(int)(centre-(length*0.8)*
			Math.cos((double)(time-width)*Math.PI/30));
		y[1]=(int)(centre-length*
			Math.cos((double)time*Math.PI/30));
		y[2]=(int)(centre-(length*0.8)*
			Math.cos((double)(time+width)*Math.PI/30));
		gfx.drawLine(centre,centre,x[0],y[0]);
		gfx.drawLine(x[0],y[0],x[1],y[1]);
		gfx.drawLine(x[1],y[1],x[2],y[2]);
		gfx.drawLine(x[2],y[2],centre,centre);
	}

	public void paint(Graphics g)
	{
		if(offscreenImage==null)
		{
			offscreenImage=createImage(diameter,diameter);
			osG=offscreenImage.getGraphics();
			osG.drawImage(face,0,0,null);
		}

		if(time!=oldTime)
		{
			osG.setColor(Color.white);
			drawArm(osG,armlength,0.4,(double)oldTime/timeDiv);
			drawArm(osG,armlength,1.0,(double)oldTime/(60*timeDiv));
			drawArm(osG,armlength/2,2.0,
				5*((double)oldTime/(3600*timeDiv)));
		}

		osG.setColor(Color.red);
		drawArm(osG,armlength,0.4,(double)time/timeDiv);
		osG.setColor(Color.blue);
		drawArm(osG,armlength,1.0,(double)time/(60*timeDiv));
		drawArm(osG,armlength/2,2.0,5*((double)time/(3600*timeDiv)));

		oldTime=time;

		if(g != null)
			g.drawImage(offscreenImage,0,0,null);
	}
	
}

/*
 * $Log: Clock.java,v $
 * Revision 1.4  1999/03/05 08:32:42  cjw44
 * Implemented peeping.
 *
 * Revision 1.3  1999/03/05 01:14:13  cjw44
 * Changed M_PI constant to the built-in Math.PI.
 *
 * Revision 1.2  1999/02/23 14:57:24  cjw44
 * Changed constructor to use new checkImages(Image[]) method in
 * FloatingComponent.
 *
 * Revision 1.1  1999/02/22 00:47:04  cjw44
 * Initial revision
 *
 */
