package india;

import java.awt.*;
import java.awt.image.*;

/**
 * A visual representation of the Edsac memory.
 * @author	Colin Watson
 * @version	$Id: CRT.java,v 2.11 1999/03/05 06:55:28 cjw44 Exp $
 * @see		FrontEnd
 */

public class CRT extends FloatingComponent
{

	private StatusBar status;

	private long[][] values = new long[32][16];
	private int currentTank = 0;
	private int currentStatusAddress = -1;

	private int ledstartx, ledstarty;
	private int ledwidth, ledheight;
	private int ledhspace, ledvspace;
	private int width, height;

	private int[] pixOff, pixOn;
	private int[] pixNow;	// the current appearance of the CRT
	private Image setMemoryImage, clearMemoryImage,
		setCurrentTankImage, paintImage;
	private ReusableMemoryImageSource
		setMemorySource = new ReusableMemoryImageSource(
			0, 0, null, 0, 0);
	private MemoryImageSource
		clearMemorySource, setCurrentTankSource, paintSource;

	/**
	 * Creates a new memory tank display with the specified images. The
	 * images should both be the same size.
	 *
	 * @param	status	The status bar on which to display information
	 *			when the mouse moves over the CRT.
	 * @param	off	An image of the tank with all LEDs off.
	 * @param	on	An image of the tank with all LEDs on.
         * @param       x       The x-coordinate of the top left of the image.
         * @param       y       The y-coordinate of the top left of the image.
         * @param       width   The width of the image.
         * @param       height  The height of the image.
	 * @param	ledstartx	The x-coordinate of the top left corner
					of the area covered by LEDs.
	 * @param	ledstarty	The y-coordinate of the top left corner
					of the area covered by LEDs.
	 * @param	ledwidth	The width of one LED.
	 * @param	ledheight	The height of one LED.
	 */
	public CRT(StatusBar status, Image off, Image on,
			int x, int y, int width, int height,
			int ledstartx, int ledstarty,
			int ledwidth, int ledheight,
			int ledhspace, int ledvspace)
		throws ImageLoadingException
	{
		this.status = status;
		this.ledstartx = ledstartx;
		this.ledstarty = ledstarty;
		this.ledwidth = ledwidth;
		this.ledheight = ledheight;
		this.ledhspace = ledhspace;
		this.ledvspace = ledvspace;

		Image[] imgs = {off, on};
		checkImages(imgs);

		this.width = width;
		this.height = height;

		pixOff = new int[width * height];
		pixOn = new int[width * height];
		pixNow = new int[width * height];

		try
		{
			PixelGrabber pg = new PixelGrabber(
				off, x, y, width, height, pixOff, 0, width);
			pg.grabPixels();
			pg = new PixelGrabber(
				on, x, y, width, height, pixOn, 0, width);
			pg.grabPixels();
		} catch(InterruptedException e)
		{
			throw new ImageLoadingException(
				"images passed to class " +
				getClass().getName() + " (interrupted)");
		}
		System.arraycopy(pixOff, 0, pixNow, 0, width * height);
		clearMemorySource = new MemoryImageSource(
			width, height, pixNow, 0, width);
		setCurrentTankSource = new MemoryImageSource(
			width, height, pixNow, 0, width);
		paintSource = new MemoryImageSource(
			width, height, pixNow, 0, width);
	}

	public boolean mouseMove(Event e, int x, int y)
	{
		if(x < ledstartx ||
				x >= ledstartx + (ledwidth + ledhspace) * 35 ||
				y < ledstarty ||
				y >= ledstarty + (ledheight + ledvspace) * 16)
			synchronized(this)
			{
				currentStatusAddress = -1;
				status.clearPeep();
				return true;
			}
		synchronized(this)
		{
			currentStatusAddress = (currentTank << 5) + 30 -
				(y - ledstarty) / (ledheight + ledvspace) * 2 +
				((x - ledstartx) >=
					17 * (ledwidth + ledhspace) ? 1 : 0);
			status.setMemory(currentStatusAddress,
				getMemory(currentStatusAddress -
					(currentStatusAddress & 1)));
		}
		return true;
	}

	public boolean mouseDrag(Event e, int x, int y)
	{
		if(x < ledstartx ||
				x >= ledstartx + (ledwidth + ledhspace) * 35 ||
				y < ledstarty ||
				y >= ledstarty + (ledheight + ledvspace) * 16)
			synchronized(this)
			{
				currentStatusAddress = -1;
				status.clearPeep();
				return true;
			}
		synchronized(this)
		{
			currentStatusAddress = (currentTank << 5) + 30 -
				(y - ledstarty) / (ledheight + ledvspace) * 2 +
				((x - ledstartx) >=
					17 * (ledwidth + ledhspace) ? 1 : 0);
			status.setMemory(currentStatusAddress,
				getMemory(currentStatusAddress -
					(currentStatusAddress & 1)));
		}
		return true;
	}

	public boolean mouseEnter(Event e, int x, int y)
	{
		if(x < ledstartx ||
				x >= ledstartx + (ledwidth + ledhspace) * 35 ||
				y < ledstarty ||
				y >= ledstarty + (ledheight + ledvspace) * 16)
			synchronized(this)
			{
				currentStatusAddress = -1;
				status.clearPeep();
				return true;
			}
		synchronized(this)
		{
			currentStatusAddress = (currentTank << 5) + 30 -
				(y - ledstarty) / (ledheight + ledvspace) * 2 +
				((x - ledstartx) >=
					17 * (ledwidth + ledhspace) ? 1 : 0);
			status.setMemory(currentStatusAddress,
				getMemory(currentStatusAddress -
					(currentStatusAddress & 1)));
		}
		return true;
	}

	public synchronized boolean mouseExit(Event e, int x, int y)
	{
		currentStatusAddress = -1;
		status.clearPeep();
		return true;
	}

	/**
	 * Returns the current value of a given address in memory.
	 * @param	address	The address in memory to be examined
	 *			(should always be even).
	 */
	public long getMemory(int address)
	{
		return values[address >>> 5][(address & 30) >>> 1];
	}

	/**
	 * Sets the value of a given line of memory.
	 * @param	value	The new value for this register.
	 * @param	address	The address in memory to be changed
	 *			(should always be even).
	 */
	public synchronized void setMemory(long value, int address)
	{
		int tank = address >>> 5;
		int line = (address & 30) >>> 1;
		if(values[tank][line] == value)
			return;

		long v = value, lastv = values[tank][line];
		values[tank][line] = value;
		if(tank != currentTank)
			return;	// No redrawing to be done

		// Draw the change
		int startpos = 
			(ledstarty + (ledheight + ledvspace) * (15 - line)) *
				width +
			ledstartx + (ledwidth + ledhspace) * 35;
		for(int led = 0; led < 35; led++)
		{
			startpos -= ledwidth + ledhspace;
			int linepos = startpos;
			if((v & 1) != (lastv & 1))
			{
				int[] pix = (v & 1) == 0 ? pixOff : pixOn;
				for(int y = 0; y < ledheight; y++)
				{
					for(int x = 0; x < ledwidth; x++)
						pixNow[linepos] =
							pix[linepos++];
					linepos += width - ledwidth;
				}
			}
			v >>>= 1;
			lastv >>>= 1;
		}

		Graphics g = getGraphics();
		setMemorySource.modify(width, ledheight, pixNow,
			(ledstarty + (ledheight + ledvspace) *
				(15 - line)) * width, width);
		if(setMemoryImage == null)
			setMemoryImage = createImage(setMemorySource);
		else
			setMemoryImage.flush();
		g.drawImage(setMemoryImage, 0,
			ledstarty + (ledheight + ledvspace) *
				(15 - line), null);

		if((address & 1022) == (currentStatusAddress & 1022))
			status.setMemory(currentStatusAddress, value);
	}

	/**
	 * Clears the memory to 0, corresponding to pressing the "Clear"
	 * button on the front end.
	 */
	public synchronized void clearMemory()
	{
		for(int tank = 0; tank < 32; tank++)
			for(int line = 0; line < 16; line++)
				values[tank][line] = 0;

		for(int line = 0; line < 16; line++)
		{
			int startpos =
				(ledstarty + (ledheight + ledvspace) *
					(15 - line)) * width +
				ledstartx + (ledwidth + ledhspace) * 35;
			for(int led = 0; led < 35; led++)
			{
				startpos -= ledwidth + ledhspace;
				int linepos = startpos;
				for(int y = 0; y < ledheight; y++)
				{
					for(int x = 0; x < ledwidth; x++)
						pixNow[linepos] =
							pixOff[linepos++];
					linepos += width - ledwidth;
				}
			}
		}

		Graphics g = getGraphics();
		if(clearMemoryImage == null)
			clearMemoryImage = createImage(clearMemorySource);
		else
			clearMemoryImage.flush();
		g.drawImage(clearMemoryImage, 0, 0, null);

		if(currentStatusAddress != -1)
			status.setMemory(currentStatusAddress, 0);
	}

	/**
	 * Returns the index of the currently selected tank (0-31).
	 */
	public synchronized int getCurrentTank()
	{
		return currentTank;
	}

	/**
	 * Sets the index of the currently selected tank.
	 * As implemented, this class simply records all the setMemory() calls
	 * it receives, thus keeping its own copy of the Edsac's memory. This
	 * is practical since the Edsac only has a 512-word memory; for larger
	 * memories, it might be necessary to only cache a small part of them
	 * in the display class.
	 *
	 * @param	tank	The index, in the range 0-31, of the tank to be
	 *			displayed by this CRT.
	 */
	public void setCurrentTank(int tank)
	{
	    if(tank < 0 || tank > 31)
		throw new IndexOutOfBoundsException(
			"Invalid tank index passed to CRT");
	    if(currentTank == tank) return;

	    synchronized(this)
	    {
		currentTank = tank;

		for(int line = 0; line < 16; line++)
		{
			int startpos = 
				(ledstarty + (ledheight + ledvspace) *
					(15 - line)) * width +
				ledstartx + (ledwidth + ledhspace) * 35;
			long v = values[tank][line];
			for(int led = 0; led < 35; led++)
			{
				startpos -= ledwidth + ledhspace;
				int linepos = startpos;
				int[] pix = (v & 1) == 0 ? pixOff : pixOn;
				for(int y = 0; y < ledheight; y++)
				{
					for(int x = 0; x < ledwidth; x++)
						pixNow[linepos] =
						    pix[linepos++];
					linepos += width - ledwidth;
				}
				v >>>= 1;
			}
		}

		Graphics g = getGraphics();
		if(setCurrentTankImage == null)
			setCurrentTankImage = createImage(setCurrentTankSource);
		else
			setCurrentTankImage.flush();
		g.drawImage(setCurrentTankImage, 0, 0, null);

		if(currentStatusAddress != -1)
		{
			currentStatusAddress = (tank << 5) |
				(currentStatusAddress & 31);
			status.setMemory(currentStatusAddress,
				getMemory(currentStatusAddress));
		}
	    }
	}

	public void repaint(long tm, int x, int y, int w, int h) {}
	public void update(Graphics g) {}

	/**
	 * Paints the memory tank.
	 */
	public void paint(Graphics g)
	{
		if(paintImage == null)
			paintImage = createImage(paintSource);
		else
			paintImage.flush();
		g.drawImage(paintImage, 0, 0, null);
	}

}

/*
 * $Log: CRT.java,v $
 * Revision 2.11  1999/03/05 06:55:28  cjw44
 * Added support for upgraded peeping.
 *
 * Revision 2.10  1999/02/27 11:53:28  cjw44
 * Implemented Joe's suggestion that images don't need to be recreated after a
 * flush(), and streamlined the LED-setting code to only bother about LEDs that
 * have changed since the last update.
 *
 * Revision 2.9  1999/02/23 14:57:24  cjw44
 * Changed constructor to use new checkImages(Image[]) method in
 * FloatingComponent.
 *
 * Revision 2.8  1999/02/22 01:02:19  cjw44
 * Implemented "peeping", using the status bar.
 *
 * [Revision 2.7 (cjw44) lost in RCS archive accident; log follows.]
 * Put some synchronization back in to prevent mistimed redraws when changing
 * tanks.
 *
 * Revision 2.6  1999/02/16 20:31:55  cjw44
 * Added image flushing before each new image is created.
 *
 * Revision 2.5  1999/02/16 16:43:25  cjw44
 * Inverted CRT.
 *
 * Revision 2.4  1999/02/15 13:23:59  cjw44
 * Changed drawing code to use ReusableMemoryImageSources.
 *
 * Revision 2.3  1999/02/12 22:54:06  cjw44
 * Removed explicit garbage-collection code, yielding an enormous speed
 * increase (a factor of around 3).
 *
 * Revision 2.2  1999/02/12 18:39:16  cjw44
 * Fiddled with image drawing code in an attempt to fix a bug which was causing
 * the event dispatching thread to be locked out (I had thought it was a memory
 * problem). The bug was eventually found to be an issue of thread priorities
 * and a design flaw in the JVM's scheduler (see EdsacThread.java).
 *
 * Revision 2.1  1999/02/11 10:55:14  cjw44
 * Changed drawing system to use int arrays rather than images according to
 * Joe's suggestions: only one drawImage() call per method is now required.
 * Added explicit garbage-collection every GC_THRESHOLD updates.
 * There are actually 32 tanks - I'd forgotten that the memory array in Edsac
 * is made up of ints, not longs. This also resulted in only every second line
 * of the CRT being used. Both problems are now fixed.
 * There was also a bug in the last version that caused a one-bit overlap in
 * the middle line of the CRT. This has also been fixed.
 *
 * Revision 1.5  1999/02/09 15:37:38  cjw44
 * There are 64 tanks, not 32.
 * Added a clearMemory() method.
 *
 * Revision 1.4  1999/02/08 18:12:10  cjw44
 * Changed method calls on the Graphics object passed to paint() to use this
 * component as an ImageObserver.
 *
 * Revision 1.3  1999/02/06 22:58:55  cjw44
 * Changed getLine() and setLine() to getMemory() and setMemory(), to be
 * consistent with FrontEnd.
 * Added methods to get and set the current tank, made leds and values 3- and
 * 2-dimensional respectively to support this, and modified paint()
 * accordingly.
 *
 * Revision 1.2  1999/02/06 13:05:46  cjw44
 * Allowed for horizontal and vertical spacing between LEDs.
 *
 * Revision 1.1  1999/02/05 16:54:23  cjw44
 * Initial revision
 *
 */
