package india;

import java.applet.Applet;
import java.awt.*;
import java.io.File;
import java.net.*;
import java.util.Vector;

/**
 * The Edsac simulator's main frame, in which things like the FrontEnd are
 * instantiated.
 * @author	Colin Watson
 * @version	$Id: MainFrame.java,v 3.2 1999/03/10 07:45:13 cjw44 Exp $
 * @see		FrontEnd
 */

public class MainFrame extends Frame
{

	public static final String[] imageFilenames = {
		"logo.jpg", "unselect.jpg", "select.jpg", "grey.jpg",
		"dial.jpg"};

	private static final String
		FILE_NEW = "New editor",
		FILE_OPEN = "Open...",
		FILE_EXIT = "Exit",
		EDSAC_START = "Start",
		EDSAC_RESET = "Reset",
		EDSAC_CLEAR = "Clear",
		EDSAC_STOP = "Stop",
		EDSAC_SINGLE_STEP = "Single step",
		OPTIONS_ORDERS_1 = "Initial orders 1",
		OPTIONS_ORDERS_2 = "Initial orders 2",
		OPTIONS_SHORT_TANKS = "Show short tanks",
		OPTIONS_SPEED_CONTROLS = "Speed controls...",
		OPTIONS_PRINTING = "Use real teleprinter",
		HELP_ABOUT = "About...";

	private static final String
		IMAGE_ERROR_TITLE = "Image loading failure",
		IMAGE_ERROR_MESSAGE =
			"The following images failed to load: ";

	private MenuItem
		edsacStart = new MenuItem(EDSAC_START),
		edsacReset = new MenuItem(EDSAC_RESET),
		edsacClear = new MenuItem(EDSAC_CLEAR),
		edsacStop = new MenuItem(EDSAC_STOP),
		edsacSingleStep = new MenuItem(EDSAC_SINGLE_STEP);

	private CheckboxMenuItem[] orders = {
		new CheckboxMenuItem(OPTIONS_ORDERS_1),
		new CheckboxMenuItem(OPTIONS_ORDERS_2)};
	private CheckboxMenuItem
		shortTanks = new CheckboxMenuItem(OPTIONS_SHORT_TANKS),
		printing = new CheckboxMenuItem(OPTIONS_PRINTING);

	private FrontEnd fe;
	private StatusBar status;
	private boolean isInApplet = true;
	private Applet applet;

        private RemoteFileMenu exampleMenu = null;

	/**
	 * Creates a new main simulator frame.
	 * @param	isInApplet	True if we're running in an applet,
	 *				otherwise false.
	 * @param	applet	The parent applet if one exists, otherwise
	 *			null.
	 */
	public MainFrame(boolean isInApplet, Applet applet)
		throws ImageLoadingException
	{
		super("EDSAC 1999");
		this.isInApplet = isInApplet;
		this.applet = applet;

		Dimension screenSize =
			Toolkit.getDefaultToolkit().getScreenSize();
		Dimension windowSize;
		Dialog pleaseWait = null;

		try
		{
			// Load images
			Toolkit tk = Toolkit.getDefaultToolkit();
			Image[] frontEndImages =
				new Image[imageFilenames.length];
			Vector problems = new Vector();
			if(isInApplet)
				for(int i = 0; i < imageFilenames.length; i++)
				{
					frontEndImages[i] = applet.getImage(
						applet.getDocumentBase(),
						imageFilenames[i]);
					if(frontEndImages[i] == null)
						problems.addElement(
							imageFilenames[i]);
				}
			else
				for(int i = 0; i < imageFilenames.length; i++)
				{
					frontEndImages[i] = tk.getImage(
						imageFilenames[i]);
					if(frontEndImages[i] == null)
						problems.addElement(
							imageFilenames[i]);
				}
			if(!problems.isEmpty())
			{
				StringBuffer message = new StringBuffer(
					"Couldn't find ");
				message.append(problems.firstElement());
				for(int i = 1; i < problems.size(); i++)
				{
					message.append(", ");
					message.append(problems.elementAt(i));
				}
				throw new ImageLoadingException(
					message.toString());
			}

			// Show splash screen
			if(isInApplet)
			{
				if(applet.countComponents() == 0)
				{
					Logo logo = new Logo(
						frontEndImages[0]);
					applet.add("Center", logo);
				}
				applet.paint(applet.getGraphics());
			}
			else
			{
				pleaseWait = new Dialog(this,
					"EDSAC 1999", false);
				pleaseWait.add("Center",
					new Logo(frontEndImages[0]));
				Panel p = new Panel();
				p.add(new Label(
					"Loading EDSAC. Please wait ..."));
				pleaseWait.add("South", p);
				pleaseWait.pack();
				windowSize = pleaseWait.size();
				pleaseWait.move(
				  (screenSize.width - windowSize.width) / 2,
				  (screenSize.height - windowSize.height) / 2);
				pleaseWait.show();
			}

			// Create FrontEnd
			add("South", status = new StatusBar());
			add("Center", fe = new FrontEnd(this, isInApplet,
				status, frontEndImages[0], frontEndImages[1],
				frontEndImages[2], frontEndImages[3],
				frontEndImages[4]));
		} catch(ImageLoadingException e)
		{
			if(pleaseWait != null)
				pleaseWait.dispose();
			ErrorDlg dialog = new ErrorDlg(this,
				IMAGE_ERROR_MESSAGE + e.getMessage(),
				IMAGE_ERROR_TITLE);
			dialog.pack();
			windowSize = dialog.size();
			dialog.move((screenSize.width - windowSize.width) / 2,
				(screenSize.height - windowSize.height) / 2);
			dialog.show();
			if(!isInApplet)
				System.exit(1);
			// Rethrow exception to stop main frame being shown
			throw e;
		}

		// Set up menus
		MenuBar bar = new MenuBar();
		Menu fileMenu = new Menu("File");
		fileMenu.add(FILE_NEW);
		if(!isInApplet)
		{
			fileMenu.add(FILE_OPEN);
			fileMenu.addSeparator();
			fileMenu.add(FILE_EXIT);
		}
		Menu edsacMenu = new Menu("EDSAC");
		edsacMenu.add(edsacStart);
		edsacMenu.add(edsacReset);
		edsacMenu.add(edsacClear);
		edsacMenu.add(edsacStop);
		edsacMenu.add(edsacSingleStep);
		Menu optionsMenu = new Menu("Options");
		optionsMenu.add(orders[0]);
		optionsMenu.add(orders[1]);
		orders[0].setState(false);
		orders[1].setState(true);
		optionsMenu.addSeparator();
		optionsMenu.add(shortTanks);
		shortTanks.setState(true);
		optionsMenu.add(OPTIONS_SPEED_CONTROLS);
		if(!isInApplet)
		{
			optionsMenu.addSeparator();
			optionsMenu.add(printing);
			printing.setState(false);
		}
		Menu helpMenu = new Menu("Help");
		helpMenu.add(HELP_ABOUT);
		bar.add(fileMenu);
		bar.add(edsacMenu);
		bar.add(optionsMenu);
		setMenuBar(bar);

		try
		{
			exampleMenu = new RemoteFileMenu(
				new URL(fe.getBase(), "example"));
			bar.add(exampleMenu.getMenu());
		} catch(Exception e) {}
		bar.setHelpMenu(helpMenu);

		pack();
		setResizable(false);
		if(pleaseWait != null)
			pleaseWait.dispose();
	}

	public Applet getApplet()
	{
		return applet;
	}

	public void show()
	{
		super.show();
		paint(getGraphics());
		/* This is a quick kludge to fix the way that the output window
		 * is displayed on top of the simulator window, at least under
		 * fvwm in X Windows on my machine. We pause for a brief period
		 * to give the window manager a chance to display the
		 * simulator window and catch on that it might not want to
		 * display other windows right on top of it.
		 */
		try
		{
			Thread.sleep(50);
		} catch(InterruptedException e) {}
		if(fe != null)
		{
			fe.showAllWindows(true);
			fe.setRunning(false);
		}
	}

	public void hide()
	{
		if(fe != null)
			fe.showAllWindows(false);
		super.hide();
	}

	public void update(Graphics g)
	{
		paint(g);
	}

	public void setRunning(boolean cond)
	{
		edsacStart.enable(!cond);
		edsacReset.enable(!cond);
		edsacClear.enable(!cond);
		edsacSingleStep.enable(!cond);
		edsacStop.enable(cond);
	}

	public boolean handleEvent(Event event)
	{
		if(event.id == Event.WINDOW_DESTROY)
		{
			fe.stopEdsac();
			hide();
			if(!isInApplet)
				System.exit(0);
			return true;
		}
		return super.handleEvent(event);
	}

	public boolean action(Event event, Object what)
	{
		/* Since we've used static final Strings above, they
		 * are in canonical form. Just in case the AWT has
		 * decided to change them for some bizarre reason, we
		 * call intern() on the name of the menu item to
		 * convert it to canonical form so that we can compare
		 * strings by reference identity.
		 */
		String name;
		if(what instanceof Boolean && event.target instanceof MenuItem)
			name = ((MenuItem)event.target).getLabel();
		else if(what instanceof String)
			name = (String)what;
		else
			return super.action(event, what);
		name = name.intern();
		
		// example program menu item loader here
		if(event.target instanceof MenuItem)
		{
			try
			{
				MenuItem example = (MenuItem)event.target;
				String s = exampleMenu.getFile(example);
				if(s != null)
				{
					fe.newEditor(s, example.getLabel());
					return true;
				}
			} catch(Exception e) {}
		}
	
		// if it gets here no example program item was clicked

		if(name == FILE_NEW)
		{
			fe.newEditor("", "EDSAC Tape Editor " +
				fe.getEditorSequenceNumber());
			return true;
		}
		else if(name == FILE_OPEN)
		{
			FileDialog loader = new FileDialog(this,
				"Open program", FileDialog.LOAD);
			
			loader.setFilenameFilter(new EdsacFilenameFilter());


			loader.show();
			if(loader.getFile() != null)
			{
				Editor ed = fe.newEditor("",
					loader.getFile());
				ed.open(new File(loader.getDirectory(),
					loader.getFile()).toString());
			}
			return true;
		}
		else if(name == FILE_EXIT)
		{
			fe.stopEdsac();
			hide();
			if(!isInApplet)
				System.exit(0);
			return true;
		}
		else if(name == EDSAC_START)
		{
			if(!fe.isRunning())
				fe.startEdsac();
			return true;
		}
		else if(name == EDSAC_RESET)
		{
			if(!fe.isRunning())
				fe.resetEdsac();
			return true;
		}
		else if(name == EDSAC_CLEAR)
		{
			if(!fe.isRunning())
				fe.clearEdsac();
			return true;
		}
		else if(name == EDSAC_STOP)
		{
			if(fe.isRunning())
				fe.stopEdsac();
			return true;
		}
		else if(name == EDSAC_SINGLE_STEP)
		{
			if(!fe.isRunning())
				fe.singleStepEdsac();
			return true;
		}
		else if(name == OPTIONS_ORDERS_1)
		{
			orders[0].setState(true);
			orders[1].setState(false);
			fe.setInitialOrders(Edsac.INITIAL_ORDERS_1);
			return true;
		}
		else if(name == OPTIONS_ORDERS_2)
		{
			orders[1].setState(true);
			orders[0].setState(false);
			fe.setInitialOrders(Edsac.INITIAL_ORDERS_2);
			return true;
		}
		else if(name == OPTIONS_SHORT_TANKS)
		{
			fe.showShortTanks(shortTanks.getState());
			return true;
		}
		else if(name == OPTIONS_SPEED_CONTROLS)
		{
			SpeedControls controls = new SpeedControls(this, fe);
			Dimension screenSize =
				Toolkit.getDefaultToolkit().getScreenSize();
			Dimension windowSize = controls.size();
			controls.move(
				(screenSize.width - windowSize.width) / 2,
				(screenSize.height - windowSize.height) / 2);
			controls.show();
			return true;
		}
		else if(name == OPTIONS_PRINTING)
		{
			fe.setStdoutPrinting(printing.getState());
			return true;
		}
		else if(name == HELP_ABOUT)
		{
			fe.showAboutBox();
			return true;
		}
		return super.action(event, what);
	}

	public boolean keyDown(Event event, int key)
	{
		switch(key)
		{
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			if(!fe.isRunning())
				fe.inputNumber(key - '0');
			return true;
		case 's': case 'S':
			if(!fe.isRunning())
				fe.startEdsac();
			return true;
		case '\n':	// Enter
			if(!fe.isRunning())
				fe.resetEdsac();
			return true;
		case 'c': case 'C':
		case '\b':	// Backspace
		case 127:	// Delete
			if(!fe.isRunning())
				fe.clearEdsac();
			return true;
		case 27:	// Escape
			if(fe.isRunning())
				fe.stopEdsac();
			return true;
		case ' ':
			if(!fe.isRunning())
				fe.singleStepEdsac();
			return true;
		case Event.LEFT: case Event.UP:
			fe.changeTank(false, 1);
			return true;
		case Event.RIGHT: case Event.DOWN:
			fe.changeTank(true, 1);
			return true;
		case Event.PGUP:
			fe.changeTank(false, 8);
			return true;
		case Event.PGDN:
			fe.changeTank(true, 8);
			return true;
		case Event.HOME:
			fe.changeTank(0);
			return true;
		case Event.END:
			fe.changeTank(31);
			return true;
		}
		return false;
	}

}

/*
 * $Log: MainFrame.java,v $
 * Revision 3.2  1999/03/10 07:45:13  cjw44
 * applet.setBackground(Color.white) is now done in EdsacSimulator.init().
 * Only add logo if the applet is empty, thus allowing for restarts.
 *
 * Revision 3.1  1999/03/10 04:00:10  cjw44
 * Added facility to print to a real teleprinter (the user must manually
 * redirect standard output to a printer port).
 *
 * Revision 2.10  1999/03/10 02:34:04  cjw44
 * Renamed dial.gif to dial.jpg.
 *
 * Revision 2.9  1999/03/10 02:03:47  cjw44
 * Passed the FrontEnd to the SpeedControls constructor so that it can work in
 * multiples of EDSAC speed.
 *
 * Revision 2.8  1999/03/09 01:11:58  cjw44
 * Fixed disabling/re-enabling of short tanks.
 *
 * Revision 2.7  1999/03/05 07:11:29  cjw44
 * Made more of an effort to stop running threads when exiting, as not doing
 * this has caused the threads to hang around afterwards in at least some
 * circumstances.
 *
 * Revision 2.6  1999/03/04 15:46:21  cjw44
 * Background is no longer cleared on a repaint.
 * Move splash screen around before showing it so that window managers get a
 * say and so that we don't get a jerking effect as it is moved.
 *
 * Revision 2.5  1999/03/04 01:20:42  cjw44
 * Moved detection of null images from FrontEnd into MainFrame.
 *
 * Revision 2.4  1999/03/02 01:17:13  cjw44
 * Expanded the area of code where ImageLoadingExceptions are caught.
 * Changed the titles of the splash screen and main window to "EDSAC 1999".
 *
 * Revision 2.3  1999/02/28 23:56:00  cjw44
 * Rearranged the way the splash screen is displayed, so that, in an applet,
 * the logo is displayed in the applet panel instead.
 * Added speed controls.
 * Do an explicit paint on show, since this has occasionally failed to happen
 * automatically.
 *
 * Revision 2.2  1999/02/24 11:45:13  jm266
 * Made it use the .eds filename filter.
 *
 * Revision 2.1  1999/02/23 23:54:47  cjw44
 * Implemented greyout of items on the Edsac menu and disabled the
 * corresponding shortcut keys as appropriate.
 * show() now tells FrontEnd to set the running state to false (this is the
 * earliest point at which we can do this, since otherwise we get exceptions as
 * the greyout code tries to write to the screen).
 *
 * Revision 1.20  1999/02/22 16:16:21  cjw44
 * ImageLoadingExceptions in the FrontEnd constructor are now rethrown out to
 * EdsacSimulator to stop it trying to show the MainFrame.
 *
 * Revision 1.19  1999/02/22 12:54:08  jm266
 * Added the example programs menu.
 *
 * Revision 1.18  1999/02/22 01:21:39  cjw44
 * Added the status bar and passed it to the FrontEnd.
 *
 * [Revision 1.17 (cjw44) lost in RCS archive accident; log follows.]
 * Added key events on the cursor keys, Page Up/Down, Home, and End which
 * change the current tank.
 *
 * Revision 1.16  1999/02/19 18:24:01  cjw44
 * Removed debugging code in handleEvent().
 *
 * Revision 1.15  1999/02/19 00:01:04  cjw44
 * Fixed incompatibilities with IE4. (I *wish* JDKs would all behave in
 * approximately the same way ...)
 *
 * Revision 1.14  1999/02/18 03:17:29  cjw44
 * The add() call which adds the FrontEnd now explicitly tells the BorderLayout
 * to place it in the centre. This should be the default, but is broken  in IE4.
 * Grrr.
 *
 * Revision 1.13  1999/02/16 22:09:13  cjw44
 * Backspace and delete now clear the Edsac as well.
 *
 * Revision 1.12  1999/02/15 13:25:37  cjw44
 * Added support for retrieving the applet from which this class was
 * instantiated (if applicable).
 *
 * Revision 1.11  1999/02/13 02:45:05  cjw44
 * Added support for keystroke control of the Edsac.
 *
 * Revision 1.10  1999/02/12 22:22:37  cjw44
 * show() and hide() now check that fe is non-null to prevent
 * NullPointerExceptions should the FrontEnd's constructor throw an exception.
 *
 * Revision 1.9  1999/02/12 18:58:26  cjw44
 * Fixed the way that the output window is displayed on top of the simulator
 * window (at least on my system).
 * Altered the constructor so that the program now works as an applet.
 *
 * Revision 1.8  1999/02/11 11:01:30  cjw44
 * Moved handling of EdsacExceptions to FrontEnd.
 * Added menu option to turn updating of short tanks on or off.
 * Added a separator to each of the File and Options menus.
 * Added splash screen at startup.
 * Made sure the output window is shown with the main frame.
 *
 * Revision 1.7  1999/02/09 15:54:51  cjw44
 * Pickiness in the extreme: changed Help / About to Help / About... .
 *
 * Revision 1.6  1999/02/09 15:42:07  cjw44
 * Added support for Options / Initial Orders 1 and 2.
 *
 * Revision 1.5  1999/02/09 09:39:48  cjw44
 * Added menus. Lots of menus ...
 * Also general bug-fixing.
 *
 * Revision 1.4  1999/02/08 14:38:47  cjw44
 * Changed handleEvent() to call the method it overrides rather than returning
 * false if the event is not handled here.
 *
 * Revision 1.3  1999/02/07 13:14:36  cjw44
 * Added handleEvent(Event) to exit when the user closes the window.
 *
 * Revision 1.2  1999/02/07 12:50:59  cjw44
 * Constructor now no longer shows the frame immediately.
 *
 * Revision 1.1  1999/02/06 23:00:54  cjw44
 * Initial revision
 *
 */
