package india;

import java.applet.Applet;
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import java.util.Enumeration;
import java.util.Vector;

/**
 * The main front end in the Edsac simulator.
 * @author	Colin Watson
 * @version	$Id: FrontEnd.java,v 4.2 1999/03/10 07:43:32 cjw44 Exp $
 * @see		FrontEndInterface
 * @see		india.Edsac
 */

public class FrontEnd extends Panel implements FrontEndInterface
{

	private static final int GC_THRESHOLD = 1000000;

	private int gcCounter = 0;

	private MainFrame mainFrame;
	private boolean isInApplet;
	private StatusBar status;
	private Image logo;

	private Edsac edsac;
	private CRT longTank;
	private CRTSelector crtSelector;
	private AccumulatorTank accumulator;
	private RegisterTank multiplier, multiplicand, sequenceControl, order;
	private FloatingButton
		startButton, resetButton, clearButton, stopButton,
		singleStepButton, aboutButton;
	private Clock clock;
	private PhoneDial dial;

	private boolean running = false;
	private long time, edsacTime;
	private double timeFactor = 0.0;

	private OutputWindow output;
	private boolean stdoutPrinting = false;
	private Vector editors = new Vector();
	private int editorSequenceNumber = 1;
	private AboutBox aboutBox;

	private FloatingComponent background;

	private Dimension size;

	private EdsacThread currentThread;
	private Tape currentTape;
	private int[] currentOrders = Edsac.INITIAL_ORDERS_2;

	public FrontEnd(MainFrame mainFrame, boolean isInApplet,
			StatusBar status, Image logo,
			Image unselected, Image selected, Image greyed,
			Image dialCover)
		throws ImageLoadingException
	{
		setLayout(null);
		this.mainFrame = mainFrame;
		this.isInApplet = isInApplet;
		this.status = status;
		this.logo = logo;
		output = new OutputWindow();

		// Make sure images are loaded fully.
		// Leaving this out causes weird bugs like zero-size frames
		// being created.
		MediaTracker tracker = new MediaTracker(this);
		tracker.addImage(unselected, 0);
		tracker.addImage(selected, 1);
		tracker.addImage(greyed, 2);
		try
		{
			tracker.waitForAll();
		} catch(InterruptedException e) {}
		if(tracker.isErrorAny())
		{
			// Work out which images have failed
			StringBuffer buf = new StringBuffer();
			int i;
			for(i = 0; i < 3; i++)
			{
				if(tracker.isErrorID(i))
					buf.append(
						MainFrame.imageFilenames[i]);
				break;
			}
			for(i++; i < 3; i++)
				if(tracker.isErrorID(i))
				{
					buf.append(", ");
					buf.append(
						MainFrame.imageFilenames[i]);
				}
			throw new ImageLoadingException(buf.toString());
		}

		size = new Dimension(
			unselected.getWidth(this), unselected.getHeight(this));

		if(isInApplet)
		  {
		    OldNetscapeCheck checker=new OldNetscapeCheck(null,0,0,
					  size.width,size.height,this);
		    this.add(checker);
		    checker.move(0,0);
		    checker.resize(size);
		  }

		add(background = new FloatingComponent(
			new Image[] {unselected}, 0, 0,
			size.width, size.height), 0);
		background.move(0, 0);
		background.resize(size);
		resize(size);

		// Add CRT, CRT selector, and register tanks

		add(longTank = new CRT(status, unselected, selected,
			10, 77, 180, 134, 3, 5, 4, 4, 1, 4), 0);
		longTank.reshape(10, 77, 180, 134);
		longTank.show();

		add(crtSelector = new CRTSelector(this, status, longTank,
			unselected, selected,
			50, 240, 100, 20, 30, 50, 243, 10, 14), 0);
		crtSelector.reshape(50, 240, 100, 20);
		crtSelector.show();
		CRTSelectorButton
			left = crtSelector.getLeft(),
			right = crtSelector.getRight();
		add(left, 0);
		left.reshape(20, 240, 30, 20);
		left.show();
		add(right, 0);
		right.reshape(150, 240, 30, 20);
		right.show();

		add(accumulator = new AccumulatorTank(status,
			unselected, selected,
			10, 370, 360, 10, 3, 3, 5, 5), 0);
		accumulator.reshape(10, 370, 360, 10);
		accumulator.show();
		add(multiplier = new RegisterTank("Multiplier", status,
			unselected, selected,
			10, 290, 180, 10, 3, 3, 5, 5, 35), 0);
		multiplier.reshape(10, 290, 180, 10);
		multiplier.show();
		add(multiplicand = new RegisterTank("Multiplicand", status,
			unselected, selected,
			10, 330, 180, 10, 3, 3, 5, 5, 35), 0);
		multiplicand.reshape(10, 330, 180, 10);
		multiplicand.show();
		add(sequenceControl = new RegisterTank("SCR", status,
			unselected, selected,
			230, 290, 55, 10, 3, 3, 5, 5, 10), 0);
		sequenceControl.reshape(230, 290, 55, 10);
		sequenceControl.show();
		add(order = new RegisterTank("Order", status,
			unselected, selected,
			230, 330, 90, 10, 3, 3, 5, 5, 17), 0);
		order.reshape(230, 330, 90, 10);
		order.show();

		// Add control panel buttons
		add(startButton = new StartButton(this, status,
			unselected, selected, greyed, 210, 40, 85, 25), 0);
		startButton.reshape(210, 40, 85, 25);
		startButton.show();
		add(resetButton = new ResetButton(this, status,
			unselected, selected, greyed, 210, 70, 85, 25), 0);
		resetButton.reshape(210, 70, 85, 25);
		resetButton.show();
		add(clearButton = new ClearButton(this, status,
			unselected, selected, greyed, 210, 100, 85, 25), 0);
		clearButton.reshape(210, 100, 85, 25);
		clearButton.show();
		add(stopButton = new StopButton(this, status,
			unselected, selected, greyed, 305, 40, 85, 25), 0);
		stopButton.reshape(305, 40, 85, 25);
		stopButton.show();
		add(singleStepButton = new SingleStepButton(this, status,
			unselected, selected, greyed, 305, 70, 85, 25), 0);
		singleStepButton.reshape(305, 70, 85, 25);
		singleStepButton.show();
		add(aboutButton = new AboutButton(this, status,
			unselected, selected, greyed, 305, 100, 85, 25), 0);
		aboutButton.reshape(305, 100, 85, 25);
		aboutButton.show();

		// Add phone dial and clock

		// Co-ordinates of centres of holes
		int[]
			holesx = {24, 72, 77, 72, 60, 43, 25, 13, 8, 12},
			holesy = {71, 60, 42, 25, 13, 8, 13, 25, 42, 59};

		add(dial = new PhoneDial(this, unselected, dialCover,
			210, 155, 0, 0, 85, holesx, holesy, 6), 0);
		dial.reshape(210, 155, 85, 85);
		dial.show();
		add(clock = new Clock(status, unselected, 305, 155, 85), 0);
		clock.reshape(305, 155, 85, 85);
		clock.show();

		edsac = new Edsac50(this);
		status.setState("Stopped");

		// We've just created a *lot* of objects. Better
		// garbage-collect now.
		System.gc();
	}

	// Methods which change some of the default AWT behaviour.

	public void showAllWindows(boolean cond)
	{
		show();
		paint(getGraphics());
		output.show(cond);
		synchronized(editors)
		{
			Enumeration enum = editors.elements();
			while(enum.hasMoreElements())
				((Editor)enum.nextElement()).show(cond);
		}
	}

	/**
	 * Returns the minimum size, in case layout managers try and resize
	 * us.
	 */
	public Dimension minimumSize()
	{
		return size;
	}

	/**
	 * Returns the preferred size, in case layout managers try and resize
	 * us.
	 */
	public Dimension preferredSize()
	{
		return size;
	}

	public void repaint(long tm, int x, int y, int w, int h) {}
	public void update(Graphics g) {}
	public void paint(Graphics g) {}

	// Methods concerned with controlling the Edsac.

	/**
	 * Sets the initial orders to be used: normally
	 * Edsac.INITIAL_ORDERS_1 or Edsac.INITIAL_ORDERS_2.
	 */
	public void setInitialOrders(int[] orders)
	{
		currentOrders = orders;
	}

	/**
	 * Loads the initial orders into the Edsac and sets the Sequence
	 * Control Register to zero.
	 */
	public void startEdsac()
	{
		setRunning(true);

		status.setState("Initializing");
		edsac.zeroEdsacTime();
		clock.setTime(0L);
		edsac.initialise(currentOrders);
		sequenceControl.forceUpdate();
		if(currentTape != null)
			currentTape.reload();
		resetEdsac();
	}

	/**
	 * Creates an EdsacThread with the current tape and starts it running.
	 */
	public void resetEdsac()
	{
		if(currentTape != null)
		{
			setRunning(true);

			output.toFront();
			mainFrame.toFront();

			if(currentThread != null && currentThread.isAlive())
				currentThread.stopRequest();
			currentThread = new EdsacThread(
				this, currentTape);
			status.setState("Running");
			time = System.currentTimeMillis();
			edsacTime = edsac.getEdsacTime();
			currentThread.start();
		}
	}

	/**
	 * Stops the current EdsacThread, if one exists.
	 */
	public void stopEdsac()
	{
		if(currentThread != null)
		{
			currentThread.stopRequest();
			currentThread = null;
			accumulator.forceUpdate();
			multiplier.forceUpdate();
			multiplicand.forceUpdate();
			sequenceControl.forceUpdate();
			order.forceUpdate();
			status.setState("Stopped");
		}
		setRunning(false);
	}

	/**
	 * Single-steps the Edsac on the current tape, forcing a register tank
	 * update.
	 */
	public void singleStepEdsac()
	{
		singleStepEdsac(currentTape);
		accumulator.forceUpdate();
		multiplier.forceUpdate();
		multiplicand.forceUpdate();
		sequenceControl.forceUpdate();
		order.forceUpdate();
	}

	/**
	 * Single-steps the Edsac on the supplied tape, allowing register
	 * tanks to choose not to update if it has been a short time since the
	 * last update.
	 */
	public void singleStepEdsac(Tape tape)
	{
		/*
		 * currentThread.killThread() is used here rather than
		 * currentThread.stopRequest(). This is because, if
		 * currentThread is non-null, we are being called from
		 * EdsacThread.run(). Calling stopRequest() would then call
		 * join(), which would wait for run() to exit - but run() will
		 * not exit until this method exits ...
		 * This now ought to be safe - even if we briefly have more
		 * than one EdsacThread in existence, the first will be in the
		 * process of cleaning up and not of single-stepping.
		 */
		if(tape != null)
		{
			try
			{
				edsac.singleStep(tape);
				while(gcCounter++ >= GC_THRESHOLD)
				{
					gcCounter = 0;
					System.gc();
				}
			} catch(EdsacHaltedException e)
			{
				// This is normal; just tell the calling thread
				// (if one exists) to stop.
				if(currentThread != null)
					currentThread.killThread();
				accumulator.forceUpdate();
				multiplier.forceUpdate();
				multiplicand.forceUpdate();
				sequenceControl.forceUpdate();
				order.forceUpdate();
				status.setState("Stopped");
				setRunning(false);
			} catch(EdsacException e)
			{
				// This is not normal; complain.
				if(currentThread != null)
					currentThread.killThread();
				accumulator.forceUpdate();
				multiplier.forceUpdate();
				multiplicand.forceUpdate();
				sequenceControl.forceUpdate();
				order.forceUpdate();
				status.setState("Stopped");
				(new ErrorDlg(mainFrame, e.getMessage(),
					"EDSAC halted unexpectedly")).show();
				setRunning(false);
			} finally
			{
				long time = edsac.getEdsacTime();
				if(time - clock.getTime() >= 10000)
					clock.setTime(time - time % 10000);
			}
			synchronized(this)
			{
			    if(timeFactor != 0.0 && running)
			    {
				long edsacTimeDiff =
					(long)((edsac.getEdsacTime() -
						edsacTime) * timeFactor);
				long timeDiff =
					System.currentTimeMillis() - time;
				if(timeDiff < edsacTimeDiff)
					try
					{
						Thread.sleep(edsacTimeDiff -
							timeDiff);
					} catch(InterruptedException e) {}
			    }
			}
		}
	}

	/**
	 * Clears the Edsac's memory and registers.
	 */
	public void clearEdsac()
	{
		edsac.clear();

		longTank.clearMemory();
		accumulator.setValue(edsac.getAccumulator());
		multiplier.setValue(edsac.getMultiplier());
		multiplicand.setValue(edsac.getMultiplicand());
		sequenceControl.setValue(edsac.getSequenceControl());
		order.setValue(edsac.getOrder());

		boolean drawState = RegisterTank.getDrawState();
		RegisterTank.setDrawState(true);

		accumulator.forceUpdate();
		multiplier.forceUpdate();
		multiplicand.forceUpdate();
		sequenceControl.forceUpdate();
		order.forceUpdate();

		RegisterTank.setDrawState(drawState);
	}

	public synchronized void showAboutBox()
	{
		if(aboutBox != null)
			if(aboutBox.isShowing())
			{
				aboutBox.toFront();
				return;
			}
			else
			{
				aboutBox.dispose();
				aboutBox = null;
			}
		aboutBox = new AboutBox(logo);
		Dimension screenSize =
			Toolkit.getDefaultToolkit().getScreenSize();
		Dimension frameSize = aboutBox.size();
		aboutBox.move((screenSize.width - frameSize.width) / 2,
			(screenSize.height - frameSize.height) / 2);
		aboutBox.show();
	}

	public void inputNumber(int n)
	{
		if(n < 0 || n > 9)
			throw new IllegalArgumentException(
				"inputNumber(int) should be passed a number " +
				"between 0 and 9");
		Accumulator acc = edsac.getAccumulator();
		acc.add((n == 0 ? 10 : n) * 2);
		edsac.setAccumulator(acc);
		accumulator.setValue(acc);
		accumulator.forceUpdate();
		resetEdsac();
	}

	public double getTimeFactor()
	{
		return timeFactor;
	}

	public synchronized void setTimeFactor(double factor)
	{
		timeFactor = factor;
		time = System.currentTimeMillis();
		edsacTime = edsac.getEdsacTime();
	}

	// Methods which centralize the enabling and disabling of components.

	public synchronized void setRunning(boolean cond)
	{
		running = cond;
		startButton.greyOut(cond);
		resetButton.greyOut(cond);
		clearButton.greyOut(cond);
		singleStepButton.greyOut(cond);
		stopButton.greyOut(!cond);
		mainFrame.setRunning(cond);
	}

	public synchronized boolean isRunning()
	{
		return running;
	}

	public synchronized void showShortTanks(boolean showTanks)
	{
		if(showTanks)
		{
			accumulator.setValue(edsac.getAccumulator());
			multiplier.setValue(edsac.getMultiplier());
			multiplicand.setValue(edsac.getMultiplicand());
			sequenceControl.setValue(edsac.getSequenceControl());
			order.setValue(edsac.getOrder());
		}
		else
		{
			accumulator.setValue(new Accumulator());
			multiplier.setValue(0);
			multiplicand.setValue(0);
			sequenceControl.setValue(0);
			order.setValue(0);
			status.clearPeep();
		}

		RegisterTank.setDrawState(true);

		accumulator.forceUpdate();
		multiplier.forceUpdate();
		multiplicand.forceUpdate();
		sequenceControl.forceUpdate();
		order.forceUpdate();

		RegisterTank.setDrawState(showTanks);
	}

	public void setStdoutPrinting(boolean print)
	{
		stdoutPrinting = print;
	}

	// Some callbacks and utility methods required by FrontEndInterface.

	/**
	 * Determines whether we are running in an applet.
	 * Useful for things like the code editor which may need to save
	 * files.
	 */
	public boolean isApplet()
	{
		return isInApplet;
	}

	/**
	 * Gets the base URL from which example programs are to be downloaded.
	 */
	public URL getBase() throws MalformedURLException
	{
		if(isInApplet)
			return mainFrame.getApplet().getDocumentBase();
		else
			return new URL("file:///" +
				System.getProperty("user.dir") + '/');
	}

	public void changeAccumulator(Accumulator a)
	{
		accumulator.setValue(a);
	}

	public void changeMultiplier(long value)
	{
		multiplier.setValue(value);
	}

	public void changeSequenceControl(long value)
	{
		sequenceControl.setValue(value);
	}

	public void changeMultiplicand(long value)
	{
		multiplicand.setValue(value);
	}

	public void changeOrder(long value)
	{
		order.setValue(value);
	}

	public void outputText(char ch)
	{
		output.appendText(ch);
		if(stdoutPrinting)
			System.out.print(ch);
	}

	public void changeMemory(long memory, int address)
	{
		longTank.setMemory(memory, address);
	}

	public void endRead() {}

	/**
	 * Changes the long tank currently being displayed on the CRT.
	 * @param	tank	The index of the tank to be displayed.
	 */
	public void changeTank(int tank)
	{
		crtSelector.changeTank(tank);
	}

	/**
	 * Changes the long tank currently being displayed on the CRT.
	 * @param	up	True if the change is an increment, false if
	 *			it is a decrement.
	 * @param	change	The amount by which to change the current tank
	 *			index.
	 */
	public void changeTank(boolean incr, int change)
	{
		crtSelector.changeTank(incr, change);
	}

	// Methods which manage editor windows.

	/**
	 * Makes a new editor window with this text and shows the window
	 * if the FrontEnd is showing.
	 * @param	text	Text with which to initialize editor window.
	 * @param	name	Name of editor window.
	 */
	public Editor newEditor(String text, String name)
	{
		synchronized(editors)
		{
		    Editor ed;
		    if(isApplet())ed=new EditorApplet(this,name,text);
		      else ed = new Editor99(this, name, text);
			editors.addElement(ed);
			if(isVisible()) ed.show();
			return ed;
		}
	}

	/**
	 * Removes a window - called when a window is closed.
	 * removeEditor() does the hide() operation on the editor window.
	 */
	public void removeEditor(Editor ed)
	{
		synchronized(editors)
		{
			editors.removeElement(ed);
			ed.hide();
		}
	}

	/**
	 * Gets a list of all the windows.
	 */
	public Editor[] getEditors()
	{
		synchronized(editors)
		{
			Editor[] editorArray = new Editor[editors.size()];
			Enumeration enum = editors.elements();
			for(int i = 0; i < editorArray.length; i++)
				editorArray[i] = (Editor)enum.nextElement();
			return editorArray;
		}
	}

	/**
	 * Gets a sequence number that can be assigned to the title bar of an
	 * editor on its creation.
	 */
	public int getEditorSequenceNumber()
	{
		return editorSequenceNumber++;
	}

	// Methods which manage tapes.

	/**
	 * Load a tape in and run it.
	 */
	public void loadTape(Tape t)
	{
		loadTape(t, true);
	}

	/**
	 * Load a tape in (and run it unless runMe is false).
	 */
	public void loadTape(Tape t,boolean runMe)
	{
		currentTape = t;
		requestFocus();
		stopEdsac();
		clearEdsac();
		output.clearText();
		if(runMe)
			startEdsac();
	}

}

/*
 * $Log: FrontEnd.java,v $
 * Revision 4.2  1999/03/10 07:43:32  cjw44
 * setTimeFactor(double) can now be called while the simulator is running.
 *
 * Revision 4.1  1999/03/10 03:58:48  cjw44
 * Added facility to print to a real teleprinter (the user must manually
 * redirect standard output to a printer port).
 *
 * Revision 3.19  1999/03/10 03:29:29  cjw44
 * Fixed synchronization to EDSAC speed.
 *
 * Revision 3.18  1999/03/10 02:33:17  cjw44
 * Rearranged the positions of the numbers on the dial in accordance with Dom's
 * new images.
 *
 * Revision 3.17  1999/03/10 02:01:49  cjw44
 * Switched speed control mechanism to multiples of EDSAC speed.
 *
 * Revision 3.16  1999/03/09 01:10:58  cjw44
 * Added showShortTanks(boolean) method, for the use of MainFrame, which
 * properly disables and re-enables short tanks on demand.
 *
 * Revision 3.15  1999/03/09 00:00:30  cjw44
 * Fixed bug with accumulator display not updating on dial input.
 *
 * Revision 3.14  1999/03/05 08:32:52  cjw44
 * Supported peeping in the clock.
 *
 * Revision 3.13  1999/03/05 00:38:48  cjw44
 * A couple of renamings.
 *
 * Revision 3.12  1999/03/04 15:56:30  cjw44
 * Tried to fix problem of FrontEnd not always being painted.
 *
 * Revision 3.11  1999/03/04 01:20:42  cjw44
 * Moved detection of null images from FrontEnd into MainFrame.
 *
 * Revision 3.10  1999/03/04 00:58:00  cjw44
 * Only one about box can now be showing at once.
 *
 * Revision 3.9  1999/03/01 16:08:06  cjw44
 * Added support for peeping in various components.
 *
 * Revision 3.8  1999/02/28 23:52:33  cjw44
 * The project logo is now passed to the constructor as well, since the about
 * box needs it.
 * Improved reporting of missing images.
 * Changed getBase() to use the document base rather than the codebase in an
 * applet.
 * Loading a tape now clears the output window.
 *
 * Revision 3.7  1999/02/27 13:56:59  cjw44
 * Decreased the garbage-collection frequency by another 100 times. Explicit
 * garbage-collection is hardly needed now, to be honest, but I'll leave it in
 * at very low frequency just in case the out of memory errors we used to get
 * pop up again.
 *
 * Revision 3.6  1999/02/26 23:20:24  cjw44
 * Decreased the garbage collection frequency at Joe's suggestion, for a factor
 * of 6 (!) speedup.
 *
 * Revision 3.5  1999/02/24 01:46:10  jm266
 * Added the check for an old netscape if it is being run as
 * an applet.
 *
 * Revision 3.4  1999/02/23  23:53:26  cjw44
 * Added the phone dial.
 * Moved code for greying out buttons and the like while the program is running
 * or stopped into a single method, setRunning(boolean).
 * Tried to make sure greyout was as completely implemented as possible (some
 * of this is in MainFrame revision 2.1).
 *
 * Revision 3.3  1999/02/23 01:15:48  cjw44
 * Modified the constructor slightly to take account of changes in CRTSelector.
 *
 * Revision 3.2  1999/02/22 16:17:10  cjw44
 * Added checks to ensure input images are not null.
 *
 * Revision 3.1  1999/02/22 01:12:35  cjw44
 * Added Dom's Edsac clock code and support for it.
 *
 * [Revision 2.18 (cjw44) lost in RCS archive accident; log follows.]
 * Updated constructor to use the status bar.
 * Fixed Z-order issues by inserting explicit show()s in the constructor.
 * Passed changes in the Edsac's state on to the status bar.
 *
 * [Revision 2.17 (cjw44) lost in RCS archive accident; log follows.]
 * Changed the constructor to reflect the fact that the CRTSelector component
 * now covers the whole area of the selector rather than just one of the two
 * buttons.
 * Calling startEdsac() or resetEdsac() now greys out all the buttons except
 * Stop and About; calling stopEdsac() or throwing an exception to
 * singleStepEdsac(Tape) cancels the greyout.
 *
 * [Revision 2.16 (cjw44) lost in RCS archive accident; log follows.]
 * Added methods to allow the MainFrame to have key events which call through
 * to the CRTSelector to change the current tank.
 *
 * Revision 2.15  1999/02/18 16:51:45  jm266
 * Made it make EditorApplet rather than Editor99 in an applet.
 * This is needed because netscape verifier complains about code
 * that creates classes it doesn't know about even if the
 * code is never called (ie. on loading the class). The cut,copy,paste
 * code made this happen
 *
 * Revision 2.14  1999/02/18 03:19:18  cjw44
 * Used the superclass' no-arg constructor and a call to setLayout() rather
 * than super(LayoutManager), since some JDKs are broken in this respect.
 * Moved the call to Edsac50's constructor to after the graphics are built, in
 * case it decides to talk to the front end graphics during construction. This
 * was put in to support an experiment which was later scrapped, but it doesn't
 * do any harm.
 * Register tanks are now properly updated when the Edsac throws an exception
 * (halting or otherwise).
 * clearEdsac() now clears the registers as well as the memory.
 *
 * Revision 2.13  1999/02/16 22:10:48  cjw44
 * Inserted forced updates where necessary to cope with periodic updating.
 *
 * Revision 2.12  1999/02/16 20:33:07  cjw44
 * Fixed the order tank's constructor to create a 17-bit-long register rather
 * than a 10-bit-long one.
 *
 * Revision 2.11  1999/02/15 19:08:29  cjw44
 * Changed singleStepEdsac(Tape) to stop EdsacThreads with the killThread()
 * method, to be consistent with the changes made in revision 1.9 of
 * EdsacThread.
 *
 * Revision 2.10  1999/02/15 13:26:56  cjw44
 * Added support for retrieving the simulator's base URL.
 *
 * Revision 2.9  1999/02/13 02:46:03  cjw44
 * Put in some explicit garbage-collection (though fairly infrequently to
 * avoid massive slowdown) to try and fix memory problems. This will probably
 * have to be attacked in a different way, though.
 * Added some debugging code to try to figure out why error dialogs are
 * behaving strangely.
 *
 * Revision 2.8  1999/02/12 18:56:26  cjw44
 * Various alterations to try and stop the event dispatching thread from being
 * locked out; see EdsacThread.java for a description of the main problem. Also
 * fixed a bug in singleStepEdsac(Tape) which may have contributed to this: see
 * the comments in that method for more details.
 *
 * Revision 2.7  1999/02/11 11:10:14  cjw44
 * Updated constructor extensively to handle the changes in FloatingComponent
 * (and its subclasses) and Dom's changes to the graphics.
 * The LEDs on the CRT are now drawn as 4x4 with a 1-pixel border below and to
 * the right instead of as 5x5, for reasons of efficiency.
 * Added an explicit garbage-collection to the end of the constructor.
 * Removed createRegisterTank, as the work is now done in the RegisterTank
 * constructor.
 * Removed the paint() method, as it's currently unnecessary.
 * Replaced show() and hide() by a single method, showAllWindows(boolean),
 * which doesn't change the behaviour of Component.show() or Component.hide().
 * Removed a lot of explicit paint calls (as they're now mostly done by the
 * components themselves only as needed), and replaced all calls to repaint()
 * by paint(getGraphics()) to avoid flicker.
 * startEdsac() now reloads the current tape, and also calls resetEdsac() as
 * per the controls on the original machine.
 * Removed resetEdsac(Tape) and moved its code to resetEdsac().
 * Added handling of EdsacExceptions to singleStepEdsac(Tape).
 * endRead() now does nothing.
 * loadTape() now stops and clears the Edsac even if runMe is false.
 *
 * Revision 2.6  1999/02/09 15:46:46  cjw44
 * Added support for varying initial orders.
 * Added CRT selector.
 * Fiddled around with synchronization a bit to try and avoid lockups.
 * Added more calls to repaint() to make sure things are redrawn properly on
 * changes.
 * Made clearEdsac() tell the CRT (longTank) about it so that the display can
 * be cleared.
 * If loadTape() is told to run a tape, it will now do a Stop, a Clear, a
 * Start, and a Reset to make absolutely sure that the Edsac gets back to a
 * well-defined initial state.
 *
 * Revision 2.5  1999/02/09 09:36:33  cjw44
 * Rearranged components so that passOnEvent() isn't needed; it turned out
 * that most of the problems were caused by passing null rather than this as
 * the ImageObserver argument to g.drawImage() and the like. Fixed.
 * Changed code to make sure images are loaded fully to a MediaTracker, which
 * seems to be the intended mechanism for this sort of thing.
 * Made a lot of other little changes, tweaks, and bug-fixes; by and large,
 * the FrontEnd ought to mostly work now, though it still hasn't been tested
 * on a running Edsac.
 *
 * Revision 2.4  1999/02/08 14:40:09  cjw44
 * Added code to make sure images are loaded fully.
 * Made the background the first component in the container, though it still
 * doesn't work in terms of event handling.
 * Added the passOnEvent() kludge to try and fix some very odd behaviour in
 * the AWT: this essentially tries to make absolutely sure that the
 * background never traps events that are meant for other components.
 * Warning: this is not necessarily immediately generalizable to other
 * containers.
 * Miscellaneous bug fixes.
 * Added debugging code.
 *
 * Revision 2.3  1999/02/07 13:02:40  cjw44
 * Made sure all components reshape themselves properly.
 *
 * Revision 2.2  1999/02/06 23:01:35  cjw44
 * Implemented the about box.
 *
 * Revision 2.1  1999/02/06 17:23:14  cjw44
 * Changed superclass back to Panel and used a FloatingComponent instead for
 * the background.
 * Fleshed out the constructor considerably to add most of the components we
 * want.
 * Went through several compilation attempts and generally tried to root out as
 * many syntax errors as I could. It's quite close now to being able to
 * compile, hence the major version upgrade; a few things still need to be
 * implemented, though.
 *
 * Revision 1.8  1999/02/05 12:26:34  cjw44
 * Got rid of drawImage(), and generally made sure things support the new
 * FloatingComponent architecture.
 * Added support for all the control buttons.
 * Miscellaneous tidying up.
 *
 * Revision 1.7  1999/02/02 09:41:34  cjw44
 * Made drawImage() call repaint() as well.
 *
 * Revision 1.6  1999/02/02 09:38:29  cjw44
 * Made FrontEnd extend Canvas rather than Panel.
 * Added facilities for drawing images directly onto the canvas. Constructor
 * now takes an Image object to reflect this.
 * Overrode paint() to paint a double-buffered image of the current state of
 * the canvas.
 *
 * Revision 1.5  1999/02/01 21:56:27  cjw44
 * Changed name of method called in outputText().
 *
 * Revision 1.4  1999/02/01 21:35:11  cjw44
 * Implemented editor management.
 * Implemented remainder of FrontEndInterface.
 * Added an output window.
 * Modified show() and hide() to also show and hide output and editor windows.
 *
 * Revision 1.3  1999/01/25 17:05:18  jm266
 * added the functions for the Editor window
 *
 * Revision 1.2  1999/01/25  15:02:51  cjw44
 * change{Register}() methods added as per FrontEndInterface.
 * Access control on edsac field changed to private (may need accessor method?)
 * Fields referring to representations of registers added (may also need
 * accessors).
 *
 * Revision 1.1  1999/01/22 18:03:51  cjw44
 * Initial revision
 */
