package india;

import java.awt.*;

/**
 * A button that "floats" over other components and is drawn with Images.
 * @author	Colin Watson
 * @version	$Id: FloatingButton.java,v 1.11 1999/03/05 06:55:28 cjw44 Exp $
 * @see		FloatingComponent
 * @see		Canvas
 */

public abstract class FloatingButton extends FloatingComponent
{

	private StatusBar status;
	private boolean down = false, enabled = true, peeping = false;

	public FloatingButton(StatusBar status,
			Image unselected, Image selected, Image greyed,
			int x, int y, int width, int height)
		throws ImageLoadingException
	{
		super(new Image[] {unselected, selected, greyed},
			x, y, width, height);
		this.status = status;
	}

	public synchronized boolean mouseDown(Event e, int x, int y)
	{
		if(enabled)
		{
			setCurrentImage(1);
			down = true;
		}
		return true;
	}

	public synchronized boolean mouseUp(Event e, int x, int y)
	{
		if(down)	// down => enabled
		{
			setCurrentImage(0);
			if(inside(x, y))
				buttonClicked();
			down = false;
		}
		return true;
	}

	public synchronized boolean mouseDrag(Event e, int x, int y)
	{
		if(x < 0 || x >= size.width || y < 0 || y >= size.height)
		{
			if(down)	// down => enabled
				if(getCurrentImage() != 0) setCurrentImage(0);
			if(peeping)
			{
				peeping = false;
				status.clearPeep();
			}
		}
		else
		{
			if(down)	// down => enabled
				if(getCurrentImage() != 1) setCurrentImage(1);
			if(!peeping)
			{
				peeping = true;
				status.setOther(getName(), getStatusMessage());
			}
		}
		return true;
	}

	public synchronized boolean mouseMove(Event e, int x, int y)
	{
		if(!peeping)
		{
			peeping = true;
			status.setOther(getName(), getStatusMessage());
		}
		return true;
	}

	public synchronized boolean mouseEnter(Event e, int x, int y)
	{
		if(!peeping)
		{
			peeping = true;
			status.setOther(getName(), getStatusMessage());
		}
		return true;
	}

	public synchronized boolean mouseExit(Event e, int x, int y)
	{
		if(peeping)
		{
			peeping = false;
			status.clearPeep();
		}
		return true;
	}

	public synchronized void greyOut(boolean grey)
	{
		enabled = !grey;
		down = false;
		setCurrentImage(grey ? 2 : 0);
	}

	public abstract void buttonClicked();

	public abstract String getName();
	public abstract String getStatusMessage();

}

/*
 * $Log: FloatingButton.java,v $
 * Revision 1.11  1999/03/05 06:55:28  cjw44
 * Added support for upgraded peeping.
 *
 * Revision 1.10  1999/03/04 00:58:31  cjw44
 * Fixed dragging semantics.
 *
 * Revision 1.9  1999/03/01 16:01:34  cjw44
 * Added facilities to display status bar messages about each button.
 *
 * Revision 1.8  1999/02/22 01:10:10  cjw44
 * Thought very carefully about the way the actions of greying out components
 * and of cancelling greyout should interact with mouse events, and implemented
 * this properly.
 *
 * Revision 1.7  1999/02/19 00:06:37  cjw44
 * Greying out a button will now disable it, and ungreying it will re-enable
 * it.
 *
 * Revision 1.6  1999/02/12 22:28:55  cjw44
 * Modified mouse behaviour when the mouse is dragged out of the area covered
 * by the button.
 *
 * Revision 1.5  1999/02/11 10:42:05  cjw44
 * Added x, y, width, and height parameters to the constructor to match
 * FloatingComponent.
 * Swapped order of method calls in mouseUp() so that the button is always
 * lifted immediately, even if the operation performed by buttonClicked() takes
 * a long time.
 *
 * Revision 1.4  1999/02/09 04:23:07  cjw44
 * Constructor now throws ImageLoadingException to match FloatingComponent.
 *
 * Revision 1.3  1999/02/08 20:53:36  cjw44
 * Removed debugging code.
 *
 * Revision 1.2  1999/02/08 14:34:34  cjw44
 * Removed mouseClicked(), since the AWT doesn't actually do mouse click
 * events, just down and up.
 * Changed the semantics of mouseDown() and mouseUp(): both are now
 * synchronized (as well as greyOut()), and mouseUp() now calls
 * buttonClicked().
 * Added debugging code.
 *
 * Revision 1.1  1999/02/04 00:59:17  cjw44
 * Initial revision
 *
 */
