package india;

import java.awt.*;

/**
 * A status bar, to be displayed below the FrontEnd, which displays the state
 * of the Edsac and the contents of its memory or registers as the mouse moves
 * over them.
 * @author	Colin Watson
 * @version	$Id: StatusBar.java,v 1.5 1999/03/09 04:11:24 cjw44 Exp $
 * @see		CRT
 */

public class StatusBar extends Panel
{

	protected static final char[] opcodes = {
		'\0', '\0', '\0', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', '\0',
		'\0', 'S', 'Z', '\0', '\0', '\0', 'F', '\0', '\0', '\0',
		'H', 'N', '\0', '\0', 'L', 'X', 'G', 'A', '\0', 'C', 'V'};

	private Label state, source, fraction, decimal, instruction;
	private boolean orders1 = false;

	/**
	 * Creates a new status bar.
	 */
	public StatusBar()
	{
		setBackground(Color.white);
		setLayout(new BorderLayout());
		Panel p = new Panel();
		p.setLayout(new GridLayout(1, 2));
		Panel p0 = new Panel();
		p0.setLayout(new FlowLayout(FlowLayout.CENTER));
		p0.add(new Label("EDSAC state:", Label.RIGHT));
		p0.add(state = new Label("", Label.LEFT));
		p.add(p0);
		p.add(source = new Label("", Label.CENTER));
		add("North", p);
		p = new Panel();
		p.setLayout(new GridLayout(2, 1));
		p.add(fraction = new Label("", Label.CENTER));
		p0 = new Panel();
		p0.setLayout(new GridLayout(1, 2));
		p0.add(decimal = new Label("", Label.CENTER));
		p0.add(instruction = new Label("", Label.CENTER));
		p.add(p0);
		add("Center", p);
	}

	/**
	 * Sets the description of the Edsac's state (stopped, initializing, or
	 * running).
	 * @param	edsacState	A description of the current state.
	 */
	public void setState(String edsacState)
	{
		state.setText(edsacState);
	}

	/**
	 * Shows a representation of the memory address currently under the
	 * mouse pointer.
	 * @param	address	The memory address over which the mouse is
	 *			currently positioned.
	 * @param	value	The word at the current memory address.
	 */
	public void setMemory(int address, long value)
	{
		int halfword;
		if((address & 1) == 0)
			halfword = (int)((value >> 18) & 0x1FFFF);
		else
			halfword = (int)(value & 0x1FFFF);
		setSource("Memory address " + address);
		setFraction(longToFraction(value) + " / " +
			intToFraction(halfword));
		setDecimal(value + " / " + halfword);
		setInstruction(intToInstruction(orders1, halfword));
	}

	/**
	 * Shows a representation of the register currently under the mouse
	 * pointer.
	 * @param	name	The name of the register over which the mouse
	 *			is currently positioned.
	 * @param	value	The word in the current register.
	 */
	public void setRegister(String name, long value)
	{
		setSource(name);
		setFraction(longToFraction(value));
		setDecimal(String.valueOf(value));
		setInstruction(intToInstruction(
			orders1, (int)(value & 0x1FFFF)));
	}

	/**
	 * Shows a representation of the accumulator.
	 * @param	value	The current accumulator.
	 */
	public void setAccumulator(Accumulator value)
	{
		setSource("Accumulator");
		setFraction(value.toFractionString());
		setDecimal(value.toDecString());
		setInstruction(intToInstruction(
			orders1, (int)(value.getLow() & 0x1FFFF)));
	}

	/**
	 * Set some other status comment.
	 * @param	src	The source of the comment.
	 * @param	comment	The text of the comment.
	 */
	public void setOther(String src, String comment)
	{
		setSource(src);
		setFraction(comment);
	}

	/**
	 * Set the label which identifies the source of whatever is contained
	 * in the rest of the status bar.
	 * @param	src	The source of the current contents of the
	 *			status bar.
	 */
	public void setSource(String src)
	{
		source.setText(src);
	}

	/**
	 * Shows a fixed-point fraction in the status bar.
	 * @param	frac	A string representation of the
	 *			fixed-point fraction over which the
	 *			mouse is currently positioned.
	 */
	public void setFraction(String frac)
	{
		fraction.setText(frac);
	}

	/**
	 * Set the main "peeping" label.
	 * @param	text	The text to be displayed in the main "peeping"
	 *			label.
	 */
	public void setDecimal(String text)
	{
		decimal.setText(text);
	}

	/**
	 * Shows a disassembled EDSAC instruction in the status bar.
	 * @param	inst	A string representation of the instruction over
	 *			which the mouse is currently positioned.
	 */
	public void setInstruction(String inst)
	{
		instruction.setText(inst);
	}

	/**
	 * Clears all the "peeping" labels.
	 */
	public void clearPeep()
	{
		source.setText("");
		fraction.setText("");
		decimal.setText("");
		instruction.setText("");
	}

	/**
	 * Sets which initial orders are currently in use, for the purposes of
	 * disassembly.
	 * @param	orders1	True if initial orders 1 are being used,
	 *			otherwise false.
	 */
	public void setInitialOrders(boolean orders1)
	{
		this.orders1 = orders1;
	}

	/**
	 * Gets the value of an EDSAC half-word in EDSAC fixed-point fraction
	 * form.
	 * @param	halfword	The half-word to be converted.
	 */
	protected static String intToFraction(int halfword)
	{
		boolean sign = (halfword & (1 << 16)) != 0;
		double place = 0.5;
		double total = 0.0;
		for(int bit = 1 << 15; bit != 0; bit >>>= 1)
		{
			if((halfword & bit) != 0) total += place;
			place /= 2;
		}
		return String.valueOf(sign ? -total : total);
	}

	/**
	 * Gets the value of an EDSAC word in EDSAC fixed-point fraction form.
	 * @param	word	The word to be converted.
	 */
	protected static String longToFraction(long word)
	{
		boolean sign = (word & (1L << 34)) != 0;
		double place = 0.5;
		double total = 0.0;
		for(long bit = 1L << 33; bit != 0; bit >>>= 1)
		{
			if((word & bit) != 0) total += place;
			place /= 2;
		}
		return String.valueOf(sign ? -total : total);
	}

	/**
	 * Disassembles an EDSAC half-word into an instruction.
	 * @param	orders1	True if initial orders 1 are being used,
	 *			otherwise false.
	 * @param	halfword	The half-word to be disassembled.
	 */
	protected static String intToInstruction(boolean orders1, int halfword)
	{
		int opcode = (halfword >> 12) & 0x1F;
		int address = (halfword >> 1) & 0x3FF;
		boolean length = (halfword & 0x1) != 0;
		if(opcodes[opcode] == '\0')
			return "<Undefined>";
		StringBuffer inst = new StringBuffer(
			String.valueOf(opcodes[opcode]));
		inst.append(' ');
		if(address != 0)
		{
			inst.append(address);
			inst.append(' ');
		}
		inst.append(orders1 ?
			(length ? 'L' : 'S') :
			(length ? 'D' : 'F'));
		return inst.toString();
	}

}

/*
 * $Log: StatusBar.java,v $
 * Revision 1.5  1999/03/09 04:11:24  cjw44
 * Undefined instructions now display as <Undefined> instead of <Empty>.
 * Filled in the rest of setAccumulator(Accumulator).
 *
 * Revision 1.4  1999/03/07 23:59:07  cjw44
 * Changed setRegister(String, String) to setAccumulator(Accumulator); the
 * toDecString() method in Accumulator is now used instead.
 *
 * Revision 1.3  1999/03/05 06:53:57  cjw44
 * Upgraded peeping to show fractions and to disassemble single instructions.
 *
 * Revision 1.2  1999/03/01 16:05:59  cjw44
 * Added the ability to put an arbitrary string into the peeping label.
 *
 * Revision 1.1  1999/02/22 00:48:10  cjw44
 * Initial revision
 *
 */
