package india;

import java.awt.*;

/**
 * A dialog box with options controlling the speed of the Edsac.
 * @author	Colin Watson
 * @version	$Id: SpeedControls.java,v 1.3 1999/03/10 07:42:20 cjw44 Exp $
 * @see		RegisterTank
 */

public class SpeedControls extends Dialog
{

	private static final String OK = "OK", CANCEL = "Cancel";

	private CheckboxGroup speedChoice = new CheckboxGroup();
	private Checkbox[] speedChoices = new Checkbox[2];
	private TextField speedField;
	private Button okButton, cancelButton;

	private boolean focused = false;

	private FrontEnd fe;

	/**
	 * Creates a new speed controls dialog box.
	 * @param	parent	The parent frame.
	 */
	public SpeedControls(Frame parent, FrontEnd fe)
	{
		super(parent, "Speed", true);
		this.fe = fe;
		double factor = fe.getTimeFactor();
		Panel p = new InsetPanel(5, 5, 5, 5);
		GridBagLayout bag = new GridBagLayout();
		GridBagConstraints c = new GridBagConstraints();
		p.setLayout(bag);
		p.add(speedChoices[0] = new Checkbox(
			"Run at:", speedChoice, factor != 0.0));
		c.anchor = GridBagConstraints.WEST;
		bag.setConstraints(speedChoices[0], c);
		p.add(speedField = new TextField(
			factor == 0.0 ? "1.0" :
				String.valueOf(0.1 / factor), 6));
		speedField.enable(factor != 0.0);
		bag.setConstraints(speedField, c);
		Label label = new Label("times EDSAC speed");
		p.add(label);
		c.gridwidth = GridBagConstraints.REMAINDER;
		bag.setConstraints(label, c);
		p.add(speedChoices[1] = new Checkbox(
			"Run at maximum speed", speedChoice, factor == 0.0));
		c.gridheight = GridBagConstraints.REMAINDER;
		bag.setConstraints(speedChoices[1], c);
		add("Center", p);
		p = new Panel();
		p.add(okButton = new Button(OK));
		p.add(cancelButton = new Button(CANCEL));
		add("South", p);
		pack();
		okButton.requestFocus();
	}

	/**
	 * Bring the focus to the OK button the first time we get the focus.
	 */
	public boolean gotFocus(Event event, Object what)
	{
		if(!focused)
		{
			okButton.requestFocus();
			focused = true;
		}
		return true;
	}

	/**
	 * Handle keypresses within the text field controlling the simulator's
	 * speed, so that we can stop invalid characters being entered.
	 */
	public boolean handleEvent(Event event)
	{
		switch(event.id)
		{
		case Event.KEY_PRESS:
			if(event.target == speedField &&
					event.key < '0' && event.key > '9' &&
					event.key != '.' && event.key != '\b')
				return true;
			else if(event.key == '\n')
			{
				event.target = okButton;
				return action(event, "\n");
			}
			else if(event.key == 27)
			{
				event.target = cancelButton;
				return action(event, "\033");
			}
			// fall through
		case Event.KEY_RELEASE:
		case Event.KEY_ACTION:
		case Event.KEY_ACTION_RELEASE:
			boolean b = super.handleEvent(event);
			try
			{
				double d = Double.valueOf(
					speedField.getText()).doubleValue();
				okButton.enable(d != 0.0);
			} catch(NumberFormatException e)
			{
				okButton.disable();
			}
			return b;
		}
		return super.handleEvent(event);
	}

	/**
	 * Handles action events within the dialog box, such as buttons being
	 * pushed and checkboxes being clicked.
	 * @param	event	The event that caused the action.
	 * @param	what	The action.
	 */
	public boolean action(Event event, Object what)
	{
		if(event.target == okButton ||
			event.target == speedField)
		{
			if(speedChoice.getCurrent() == speedChoices[1])
			{
				fe.setTimeFactor(0.0);
				dispose();
			}
			else if(okButton.isEnabled())
				try
				{
					String speed = speedField.getText();
					double multiple = Double.valueOf(
						speed).doubleValue();
					if(multiple != 0)
					{
					// Convert to time rather than speed,
					// and divide by ten because EDSAC time
					// is measured in tenths of
					// milliseconds.
						fe.setTimeFactor(
							0.1 / multiple);
						dispose();
					}
				} catch(NumberFormatException e) {}
			return true;
		}
		else if(event.target == cancelButton)
		{
			dispose();
			return true;
		}
		else if(event.target instanceof Checkbox)
		{
			if(speedChoice.getCurrent() == speedChoices[0])
			{
				speedField.enable();
				try
				{
					double d = Double.valueOf(
					  speedField.getText()).doubleValue();
					okButton.enable(d != 0.0);
				} catch(NumberFormatException e)
				{
					okButton.disable();
				}
			}
			else
			{
				speedField.disable();
				okButton.enable();
			}
			return true;
		}
		return false;
	}

}

/*
 * $Log: SpeedControls.java,v $
 * Revision 1.3  1999/03/10 07:42:20  cjw44
 * Implemented Enter = OK and Esc = Cancel, as well as making a more determined
 * effort to get the initial focus onto the OK button.
 *
 * Revision 1.2  1999/03/10 02:00:56  cjw44
 * Switched speed control mechanism to multiples of EDSAC speed.
 *
 * Revision 1.1  1999/02/28 23:51:58  cjw44
 * Initial revision
 *
 */
