package india;

import java.awt.*;
import java.awt.image.*;

/**
 * A component, implemented using a Canvas, that "floats" over other
 * components, and has its various states drawn with Images.
 * @author	Colin Watson
 * @version	$Id: FloatingComponent.java,v 1.9 1999/03/10 13:38:37 cjw44 Exp $
 * @see		Canvas
 */

public class FloatingComponent extends Canvas
{

	protected Dimension size;
	private Image[] images;
	private int currentImage = 0;

	/**
	 * Creates a floating component with no images.
	 */
	public FloatingComponent()
	{
		images = null;
	}

	/**
	 * Creates a floating component with the supplied array of images
	 * between which the programmer can switch.
	 * @param	imgs	The images to be drawn on the component.
	 * @param	x	The x-coordinate of the top left of the image.
	 * @param	y	The y-coordinate of the top left of the image.
	 * @param	width	The width of the image.
	 * @param	height	The height of the image.
	 */
	public FloatingComponent(Image[] imgs, int x, int y,
		int width, int height) throws ImageLoadingException
	{
		if(imgs == null || imgs.length == 0)
		{
			images = null;
			return;
		}

		checkImages(imgs);

		if(width == -1) width = imgs[0].getWidth(null);
		if(height == -1) height = imgs[0].getHeight(null);
		this.size = new Dimension(width, height);
		images = new Image[imgs.length];

		for(int i = 0; i < imgs.length; i++)
			if(x == 0 && y == 0 &&
					width == imgs[i].getWidth(null) &&
					height == imgs[i].getHeight(null))
				images[i] = imgs[i];
			else
				try
				{
					int[] pix = new int[width * height];
					PixelGrabber pg = new PixelGrabber(
						imgs[i], x, y, width, height,
						pix, 0, width);
					pg.grabPixels();
					images[i] = createImage(
						new MemoryImageSource(
							width, height, pix,
							0, width));
				} catch(InterruptedException e)
				{
					throw new ImageLoadingException(
						"images passed to class " +
						getClass().getName());
				}
	}

	public void checkImages(Image[] imgs) throws ImageLoadingException
	{
		MediaTracker tracker = new MediaTracker(this);
		for(int i = 0; i < imgs.length; i++)
			tracker.addImage(imgs[i], i);
		try
		{
			tracker.waitForAll();
		} catch(InterruptedException e)
		{
			throw new ImageLoadingException(
				"images passed to class " +
				getClass().getName() + " (interrupted)");
		}
		if(tracker.isErrorAny())
			throw new ImageLoadingException(
				"images passed to class " +
				getClass().getName());
	}

	public Dimension minimumSize()
	{
		return size;
	}

	public Dimension preferredSize()
	{
		return size;
	}

	public int getCurrentImage()
	{
		return currentImage;
	}

	public void setCurrentImage(int index)
	{
		if(images == null) return;
		if(index >= 0 && index < images.length)
		{
			if(index != currentImage)
			{
				currentImage = index;
				paint(getGraphics());
			}
		}
		else
			throw new IndexOutOfBoundsException(
				"Tried to set current image of " +
				getClass().getName() + " to " + index +
				"; index must satisfy 0 <= index < " +
				images.length);
	}

	public void update(Graphics g) {}

	public void paint(Graphics g)
	{
		if(images != null)
		// This makes having floating components that do their own
		// image drawing (e.g. when only parts of images need to be
		// changed) slightly more sensible.
			g.drawImage(images[currentImage], 0, 0, null);
		else
			super.paint(g);
	}

}

/*
 * $Log: FloatingComponent.java,v $
 * Revision 1.9  1999/03/10 13:38:37  cjw44
 * Added blank update(Graphics g) method.
 *
 * Revision 1.8  1999/03/01 15:12:48  cjw44
 * size needs to be protected so that subclasses can access it.
 *
 * Revision 1.7  1999/02/28 23:42:39  cjw44
 * Calling the constructor with a width or height of -1 now uses the width or
 * height respectively of the first element of the supplied image array.
 * Implemented minimumSize() and preferredSize(), for the benefit of layout
 * managers.
 *
 * Revision 1.6  1999/02/23 14:56:38  cjw44
 * Added checkImages(Image[]) method to save having to type it into every
 * constructor.
 *
 * Revision 1.5  1999/02/11 10:41:01  cjw44
 * Added x, y, width, and height parameters to the constructor; these are used
 * by the PixelGrabbers and MemoryImageSources that now create the stored form
 * of the images.
 *
 * Revision 1.4  1999/02/09 04:23:07  cjw44
 * Added MediaTracker code to constructor, which now throws an
 * ImageLoadingException.
 * Removed debugging code.
 *
 * Revision 1.3  1999/02/08 14:37:19  cjw44
 * g.drawImage() call in paint() changed to take this as an ImageObserver.
 * Added debugging code.
 *
 * Revision 1.2  1999/02/04 21:58:10  cjw44
 * Added minimal support for subclasses that want to draw images themselves
 * (e.g. the tank classes with groups of LEDs).
 * Added the (possibly unnecessary) ability to change the array of images after
 * initialization.
 *
 * Revision 1.1  1999/02/04 00:57:40  cjw44
 * Initial revision
 *
 */
