package india;

import java.awt.*;
import java.awt.image.*;

/**
 * A component which allows one of the Edsac's memory tanks to be selected.
 * @author	Colin Watson
 * @version	$Id: CRTSelector.java,v 1.8 1999/03/01 16:07:28 cjw44 Exp $
 * @see		FloatingComponent
 * @see		CRT
 */

public class CRTSelector extends FloatingComponent
{

	private CRT crt;
	private CRTSelectorButton left, right;

	private int width, height;
	private int numx, numy, numwidth, numheight;
	private int[] numbers;
	private int[] pixOff, pixNow;
	private Image numberImage, paintImage;
	private ReusableMemoryImageSource
		numberSource = new ReusableMemoryImageSource(0, 0, null, 0, 0),
		paintSource = new ReusableMemoryImageSource(0, 0, null, 0, 0);

	public CRTSelector(FrontEnd fe, StatusBar status, CRT crt,
			Image unselected, Image selected,
			int x, int y, int width, int height, int buttonwidth,
			int numx, int numy, int numwidth, int numheight)
		throws ImageLoadingException
	{
		this.crt = crt;
		left = new CRTSelectorButton(this, status,
			unselected, selected,
			x - buttonwidth, y, buttonwidth, height, false);
		right = new CRTSelectorButton(this, status,
			unselected, selected,
			x + width, y, buttonwidth, height, true);

		this.width = width;
		this.height = height;
		this.numx = numx - x;
		this.numy = numy - y;
		this.numwidth = numwidth;
		this.numheight = numheight;

		numbers = new int[numwidth * numheight * 10];
		pixOff = new int[width * height];
		pixNow = new int[width * height];

		try
		{
			PixelGrabber pg = new PixelGrabber(selected,
				numx, numy, numwidth * 10, numheight,
				numbers, 0, numwidth * 10);
			pg.grabPixels();
			pg = new PixelGrabber(unselected,
				x, y, width, height, pixOff, 0, width);
			pg.grabPixels();
		} catch(InterruptedException e)
		{
			throw new ImageLoadingException(
				"images passed to class " +
				getClass().getName() + " (interrupted)");
		}
		System.arraycopy(pixOff, 0, pixNow, 0, width * height);

		int pos = this.numy * width + this.numx + (numwidth * 9) / 2;
		int numpos = 0;
		for(int row = 0; row < numheight; row++)
		{
			for(int column = 0; column < numwidth; column++)
				pixNow[pos++] = numbers[numpos++];
			pos += width - numwidth;
			numpos += numwidth * 9;
		}
	}

	public CRTSelectorButton getLeft()
	{
		return left;
	}

	public CRTSelectorButton getRight()
	{
		return right;
	}

	/**
	 * Changes the long tank currently being displayed on the CRT.
	 * @param	tank	The index of the tank to be displayed.
	 */
	public void changeTank(int tank)
	{
		if(crt.getCurrentTank() == tank) return;
		crt.setCurrentTank(tank);
		// If we get to here, 0 <= tank < 32

		if(tank < 10)
		{
			int pos = numy * width + numx + numwidth * 4;
			int numpos = numwidth * tank;
			for(int y = 0; y < numheight; y++)
			{
				for(int x = 0; x < numwidth / 2; x++)
					pixNow[pos] = pixOff[pos++];
				for(int x = 0; x < numwidth; x++)
					pixNow[pos++] = numbers[numpos++];
				for(int x = 0; x < numwidth / 2; x++)
					pixNow[pos] = pixOff[pos++];
				pos += width - numwidth * 2;
				numpos += numwidth * 9;
			}
		}
		else
		{
			int pos = numy * width + numx + numwidth * 4;
			int numpos = numwidth * (tank / 10);
			for(int y = 0; y < numheight; y++)
			{
				for(int x = 0; x < numwidth; x++)
					pixNow[pos++] = numbers[numpos++];
				pos += width - numwidth;
				numpos += numwidth * 9;
			}
			pos = numy * width + numx + numwidth * 5;
			numpos = numwidth * (tank % 10);
			for(int y = 0; y < numheight; y++)
			{
				for(int x = 0; x < numwidth; x++)
					pixNow[pos++] = numbers[numpos++];
				pos += width - numwidth;
				numpos += numwidth * 9;
			}
		}

		numberSource.modify(width, height, pixNow, 0, width);
		if(numberImage != null) numberImage.flush();
		numberImage = createImage(numberSource);
		getGraphics().drawImage(numberImage, 0, 0, null);

		crt.paint(crt.getGraphics());
	}

	/**
	 * Changes the long tank currently being displayed on the CRT.
	 * @param	incr	True if the change is an increment, false if
	 *			it is a decrement.
	 * @param	change	The amount by which to change the current tank
	 *			index.
	 */
	public void changeTank(boolean incr, int change)
	{
		if(incr)
			synchronized(crt)
			{
				changeTank((crt.getCurrentTank() +
					change) % 32);
			}
		else
			synchronized(crt)
			{
				// Slightly messy to handle negative increments
				// because of the behaviour of % with negative
				// numbers
				changeTank((crt.getCurrentTank() + 32 -
					Math.abs(change) % 32) % 32);
			}
	}

	/**
	 * Paints the CRT selector.
	 */
	public void paint(Graphics g)
	{
		paintSource.modify(width, height, pixNow, 0, width);
		if(paintImage != null) paintImage.flush();
		paintImage = createImage(paintSource);
		g.drawImage(paintImage, 0, 0, null);
	}

}

/*
 * $Log: CRTSelector.java,v $
 * Revision 1.8  1999/03/01 16:07:28  cjw44
 * Added support for a status bar description of each of the selector buttons.
 *
 * Revision 1.7  1999/02/23 01:15:24  cjw44
 * Added numbering on the CRT selector.
 *
 * Revision 1.6  1999/02/22 01:06:18  cjw44
 * Fixed Z-order issues by inserting explicit show()s in the constructor.
 *
 * [Revision 1.5 (cjw44) lost in RCS archive accident; log follows.]
 * Made the CRTSelector component include the centre section as well, and
 * shifted the button functionality into CRTSelectorButton.
 * Added methods to change the tank index, in preparation for showing tank
 * numbers on the selector.
 *
 * [Revision 1.4 (cjw44) lost in RCS archive accident; log follows.]
 * Tank switching should be done through the CRTSelector to support a display
 * of the current tank index; added methods to support this.
 *
 * Revision 1.3  1999/02/11 11:00:30  cjw44
 * Added x, y, width, and height parameters to the constructor to match
 * FloatingComponent.
 *
 * Revision 1.2  1999/02/10 17:24:02  cjw44
 * Changed "crt.repaint();" to "crt.paint(crt.getGraphics());" in
 * buttonClicked().
 *
 * Revision 1.1  1999/02/09 15:38:38  cjw44
 * Initial revision
 *
 * Revision 1.4  1999/02/09 04:23:07  cjw44
 * Constructor now throws ImageLoadingException to match FloatingComponent.
 *
 * Revision 1.3  1999/02/08 20:53:36  cjw44
 * Removed debugging code.
 *
 * Revision 1.2  1999/02/08 14:34:34  cjw44
 * Removed mouseClicked(), since the AWT doesn't actually do mouse click
 * events, just down and up.
 * Changed the semantics of mouseDown() and mouseUp(): both are now
 * synchronized (as well as greyOut()), and mouseUp() now calls
 * buttonClicked().
 * Added debugging code.
 *
 * Revision 1.1  1999/02/04 00:59:17  cjw44
 * Initial revision
 *
 */
