package india;

import java.awt.*;
import java.awt.image.*;

/**
 * A visual representation of the Edsac accumulator.
 * @author	Colin Watson
 * @version	$Id: AccumulatorTank.java,v 2.15 1999/03/09 04:10:34 cjw44 Exp $
 * @see		RegisterTank
 * @see		FrontEnd
 */

public class AccumulatorTank extends RegisterTank
{

	private static final String
		getLongMessage =
			("getValue() called on AccumulatorTank; " +
			"use getLow() and getHigh() instead").intern(),
		setLongMessage =
			("setValue(long) called on AccumulatorTank; " +
			"use setValue(Accumulator) instead").intern();

	private Accumulator acc = new Accumulator(), lastUpdatedAcc = acc;
	private long
		low = 0, high = 0, lastUpdatedLow = 0, lastUpdatedHigh = 0;

	public AccumulatorTank(StatusBar status, Image off, Image on,
			int x, int y, int width, int height,
			int ledx, int ledy, int ledwidth, int ledheight)
		throws ImageLoadingException
	{
		super("Accumulator", status, off, on, x, y, width, height,
			ledx, ledy, ledwidth, ledheight, 71);
	}

	public synchronized boolean mouseMove(Event e, int x, int y)
	{
		if(draw && !peeping)
		{
			peeping = true;
			status.setAccumulator(acc);
		}
		return true;
	}

	public synchronized boolean mouseDrag(Event e, int x, int y)
	{
		if(x < 0 || x >= width || y < 0 || y >= height)
		{
			if(!draw || peeping)
			{
				peeping = false;
				status.clearPeep();
			}
		}
		else if(draw && !peeping)
		{
			peeping = true;
			status.setAccumulator(acc);
		}
		return true;
	}

	public synchronized boolean mouseEnter(Event e, int x, int y)
	{
		if(draw && !peeping)
		{
			peeping = true;
			status.setAccumulator(acc);
		}
		return true;
	}

	public synchronized boolean mouseExit(Event e, int x, int y)
	{
		if(!draw || peeping)
		{
			peeping = false;
			status.clearPeep();
		}
		return true;
	}

	/**
	 * getValue() should never be called on an AccumulatorTank.
	 * @throws	IllegalArgumentException	When called.
	 */
	public long getValue()
	{
		throw new IllegalArgumentException(getLongMessage);
	}

	/**
	 * setValue(long) should never be called on an AccumulatorTank.
	 * @throws	IllegalArgumentException	When called.
	 */
	public void setValue(long value)
	{
		throw new IllegalArgumentException(setLongMessage);
	}

	public long getLow()
	{
		return low;
	}

	public long getHigh()
	{
		return high;
	}

	/**
	 * Sets the value of the accumulator.
	 */
	public synchronized void setValue(Accumulator a)
	{
		acc = a;
		low = a.getLow();
		high = a.getHigh();

		if(!draw)
		{
			updated = false;
			return;
		}

		long time = System.currentTimeMillis();
		if(time - lastUpdate < updateThreshold)
		{
			updated = false;
			return;
		}
		lastUpdate = time;
		updated = true;

		if(a.equals(lastUpdatedAcc)) return;
		lastUpdatedAcc = a;

		// Run through low bits of accumulator
		int startpos = ledstarty * width + ledstartx +
			ledwidth * length;
		if(low != lastUpdatedLow)
		{
			long v = low, lastv = lastUpdatedLow;
			lastUpdatedLow = low;
			for(int led = 70; led >= 35; led--)
			{
				startpos -= ledwidth;
				int linepos = startpos;
				if((v & 1) != (lastv & 1))
				{
					int[] pix = (v & 1) == 0 ?
						pixOff : pixOn;
					for(int y = 0; y < ledheight; y++)
					{
						for(int x = 0; x < ledwidth;
								x++)
							pixNow[linepos] =
							    pix[linepos++];
						linepos += width - ledwidth;
					}
				}
				v >>>= 1;
				lastv >>>= 1;
			}
		}
		// Reinitialize startpos, in case the previous loop wasn't
		// executed
		startpos = ledstarty * width + ledstartx + ledwidth * 35;
		// Run through high bits of accumulator
		if(high != lastUpdatedHigh)
		{
			long v = high, lastv = lastUpdatedHigh;
			lastUpdatedHigh = high;
			for(int led = 34; led >= 0; led--)
			{
				startpos -= ledwidth;
				int linepos = startpos;
				if((v & 1) != (lastv & 1))
				{
					int[] pix = (v & 1) == 0 ?
						pixOff : pixOn;
					for(int y = 0; y < ledheight; y++)
					{
						for(int x = 0; x < ledwidth;
								x++)
							pixNow[linepos] =
							    pix[linepos++];
						linepos += width - ledwidth;
					}
				}
				v >>>= 1;
				lastv >>>= 1;
			}
		}

		Graphics g = getGraphics();
		if(setValueImage == null)
			setValueImage = createImage(setValueSource);
		else
			setValueImage.flush();
		g.drawImage(setValueImage, 0, 0, null);

		if(peeping)
			status.setAccumulator(acc);
	}

	public synchronized void forceUpdate()
	{
		if(!draw)
		{
			updated = false;
			return;
		}

		if(updated) return;
		updated = true;

		if(acc == lastUpdatedAcc) return;
		lastUpdatedAcc = acc;

		// Run through low bits of accumulator
		int startpos = ledstarty * width + ledstartx +
			ledwidth * length;
		if(low != lastUpdatedLow)
		{
			long v = low, lastv = lastUpdatedLow;
			lastUpdatedLow = low;
			for(int led = 70; led >= 35; led--)
			{
				startpos -= ledwidth;
				int linepos = startpos;
				if((v & 1) != (lastv & 1))
				{
					int[] pix = (v & 1) == 0 ?
						pixOff : pixOn;
					for(int y = 0; y < ledheight; y++)
					{
						for(int x = 0; x < ledwidth;
								x++)
							pixNow[linepos] =
							    pix[linepos++];
						linepos += width - ledwidth;
					}
				}
				v >>>= 1;
				lastv >>>= 1;
			}
		}
		// Reinitialize startpos, in case the previous loop wasn't
		// executed
		startpos = ledstarty * width + ledstartx + ledwidth * 35;
		// Run through high bits of accumulator
		if(high != lastUpdatedHigh)
		{
			long v = high, lastv = lastUpdatedHigh;
			lastUpdatedHigh = high;
			for(int led = 34; led >= 0; led--)
			{
				startpos -= ledwidth;
				int linepos = startpos;
				if((v & 1) != (lastv & 1))
				{
					int[] pix = (v & 1) == 0 ?
						pixOff : pixOn;
					for(int y = 0; y < ledheight; y++)
					{
						for(int x = 0; x < ledwidth;
								x++)
							pixNow[linepos] =
							    pix[linepos++];
						linepos += width - ledwidth;
					}
				}
				v >>>= 1;
				lastv >>>= 1;
			}
		}

		Graphics g = getGraphics();
		if(setValueImage == null)
			setValueImage = createImage(setValueSource);
		else
			setValueImage.flush();
		g.drawImage(setValueImage, 0, 0, null);

		if(peeping)
			status.setAccumulator(acc);
	}

}

/*
 * $Log: AccumulatorTank.java,v $
 * Revision 2.15  1999/03/09 04:10:34  cjw44
 * Display now only updates when the accumulator is not the same as it was at
 * the last update.
 *
 * Revision 2.14  1999/03/09 01:10:03  cjw44
 * Tanks are now properly turned off when "Show short tanks" is unchecked.
 *
 * Revision 2.13  1999/03/08 00:00:18  cjw44
 * Switched to using setAccumulator(Accumulator) rather than
 * setRegister(String, String).
 *
 * Revision 2.12  1999/03/05 06:55:28  cjw44
 * Added support for upgraded peeping.
 *
 * Revision 2.11  1999/03/01 16:05:40  cjw44
 * Sorted out a few bugs in the peeping routines.
 *
 * Revision 2.10  1999/03/01 14:35:56  cjw44
 * Added peeping.
 *
 * Revision 2.9  1999/02/28 23:49:59  cjw44
 * Made the update threshold a variable, for the benefit of speed control.
 * Expanded the area over which the mouse needs to be moved to activate
 * peeping.
 *
 * Revision 2.8  1999/02/27 11:53:28  cjw44
 * Implemented Joe's suggestion that images don't need to be recreated after a
 * flush(), and streamlined the LED-setting code to only bother about LEDs that
 * have changed since the last update.
 *
 * Revision 2.7  1999/02/22 00:50:39  cjw44
 * Partially implemented "peeping", using the status bar.
 *
 * Revision 2.6  1999/02/18 03:14:47  cjw44
 * Fixed bug in register updating: the check for whether a register was already
 * displaying the requested value was being made against the last requested
 * value rather than the last displayed value.
 *
 * Revision 2.5  1999/02/16 22:09:41  cjw44
 * Added periodic updating (currently set with a threshold of 40 milliseconds).
 *
 * Revision 2.4  1999/02/16 20:32:13  cjw44
 * Added image flushing before each new image is created.
 * Fixed bug in setValue() where the high half-word was being drawn over the
 * low half-word if the low half-word hadn't changed.
 *
 * Revision 2.3  1999/02/15 13:23:59  cjw44
 * Changed drawing code to use ReusableMemoryImageSources.
 *
 * Revision 2.2  1999/02/12 22:54:06  cjw44
 * Removed explicit garbage-collection code, yielding an enormous speed
 * increase (a factor of around 3).
 *
 * Revision 2.1  1999/02/11 10:52:37  cjw44
 * Added getLow() and getHigh() methods.
 * Changed setLongMessage to reflect the new method names from revision 1.4,
 * and added a corresponding getLongMessage.
 * Changed drawing system to int arrays and added update control and explicit
 * garbage-collection, as with RegisterTank.
 *
 * Revision 1.4  1999/02/06 16:56:27  cjw44
 * Renamed set() to setValue(), to be consistent with RegisterTank.
 *
 * Revision 1.3  1999/02/05 16:53:19  cjw44
 * Updated to fit in with changes to RegisterTank (mainly that leds is now an
 * array of boolean elements rather than of LED elements).
 *
 * Revision 1.2  1999/01/31 18:52:43  cjw44
 * Removed abstract flag from class.
 * Implemented set(Accumulator).
 * Overrode set(long) to throw an IllegalArgumentException.
 *
 * Revision 1.1  1999/01/25 15:25:53  cjw44
 * Initial revision
 */
