# Copyright (c) 2012 Matthew Kirk
# See LICENSE file for details

from os.path import join
import RPi.GPIO as GPIO
from plugins._sensor import Sensor

class TemperatureSensor(Sensor):

	can_quit = True

	def setup(self):
		"""Find all the connected sensors, setup the GPIO IO, and turn on LED."""
		self.w1_directory = "/sys/bus/w1/devices/w1_bus_master1"
		self.slave_list = "w1_master_slaves"
		self.slave_data = "w1_slave"


		# GPIO pin numbers.
		# Must be using same numbering system as used for GPIO.setmode
		self.button_pin = 17
		self.led_logging_pin = 25
		self.led_ready_pin = 18

		self.slave_ids = self._get_slave_ids()

		self.no_of_measurements = len(self.slave_ids)

		GPIO.setmode(GPIO.BCM)
		GPIO.setup(self.led_logging_pin,GPIO.OUT)
		GPIO.setup(self.led_ready_pin,GPIO.OUT)
		GPIO.setup(self.button_pin,GPIO.IN)

		GPIO.output(self.led_logging_pin,GPIO.LOW)
		# Turn on LED to indicate setup complete.
		GPIO.output(self.led_ready_pin,GPIO.HIGH)

	def wait_for_start(self):
		while not GPIO.input(self.button_pin):
			# Wait for the button to be pressed.
			pass

		while GPIO.input(self.button_pin):
			# Wait for the button to unpress.
			pass

		# Turn on LED to indicate start of logging.
		GPIO.output(self.led_logging_pin,GPIO.HIGH)
		# Turn off ready LED.
		GPIO.output(self.led_ready_pin,GPIO.LOW)

	def get_data(self):
		temperatures = []
		for slave_id in self.slave_ids:
			temperature_file = join(self.w1_directory, slave_id, self.slave_data)
			with open(temperature_file) as t:
				# Temperature is the last block in file, middle characters.
				temperature = t.read().split(" ")[-1][2:7]
				temperature = float(temperature)/1000
				temperatures.append(temperature)
		return temperatures

	def stop_measuring(self):
		return GPIO.input(self.button_pin)

	def finished(self):
		"""Turn off LED once logging is done."""
		GPIO.output(self.led_logging_pin,GPIO.LOW)


	def _get_slave_ids(self):
		"""Read 1-Wire IDs from file, and return list.

		The slave_list file contains 64-bit ID of each 1-Wire device,
		with one per line.
		"""
		slave_list_file = join(self.w1_directory, self.slave_list)
		slave_names = []
		with open(slave_list_file) as slaves:
			for slave in slaves:
				slave_names.append(slave.strip())
		return slave_names
