% -------------------------------------------------------------------------
%               RANDOM DISCRETE COLOUR SAMPLING
% -------------------------------------------------------------------------

% This code only supports 2D canvases. Email the authors for N-D code.

% PARAMETERS:
%   - C (colours): example: black,white,grey: 
%                                          [0 0 0;255 255 255;150 150 150]
%   - D (dimensions): dimensions of the colour matrix
%   - S (search patterns): example: 
%       vertical: S{1,forward} = 1:D(2), S{1,backward} = -S{1,forward}
%                all zeros in x axis
%       horizontal: S{2,forward} = 1:D(1), S{2,backward} = -S{2,forward}
%                all zeros in y axis
%       diagonal, upper left corner, increasing y axis downwards:
%         forward (towards the upper corner): {S{2,backward},S{1,backward}}
%         backward: {S{2,forward},S{1,forward}}
%   - parameters: {1,2,3,...}
%       - 1: weights: [w1,w1,...,w_number_of_search_patterns]
%       - 2: n: high n, less elongated clusters
%       - 3: sample control: [max_c1,max_c2,...,max_last_colour]; if 0,
%               do not use sample control
%       - 4: 1: penalise similar colours; 0: only penalise same colours

function [ result ] = RandomDiscreteColourSampling( C,D,S,parameters )

% set initial parameters
num = size(C,1);
numP = size(S,2);
result = zeros(D);
[emptyCR emptyCC] = find(result==0);
P = zeros(numP,1);
TP = 0;
weights = parameters{1};
colCounter = zeros(1,num);
L = parameters{3};
Wc = diag(ones(1,num));
wc = ones(1,num);
if parameters{4} % build weigths for colours
    collab = RGB2lab(C);
    tmp = fspecial('gaussian',[1 50],5);
    gauss = tmp(26:end);
    gauss = gauss/gauss(1);
    for r=1:num
        for c=1:num
            d = pdist([collab(r,:);collab(c,:)]);
            if d < 0.25
                penalty = gauss(round(d*50)+1);
            else
                penalty = 0;
            end
            Wc(r,c) = penalty;
        end
    end
end

% run algorithm
while ~isempty(emptyCR)
    % set initial parameters
    candidates = [];
    minE = bitmax;
    energies = zeros(1,num);
    E = 0;
    
    % pick a random point
    loc = max(ceil(rand()*size(emptyCR,1)),1);
    newP = [emptyCR(loc) emptyCC(loc)];

    % for each colour
    for i=1:num
        E = 0;
        % for each pattern count colours and compute E for that pattern:
        for j=1:numP
            c = countNeighbours(result,newP,D,S{j},i,Wc);
            E = E + ((P(j)+1)/(TP+1))*c^parameters{2}*weights(j);
        end

        % update minE, add to candidates
        if E <= minE
            if E < minE
                candidates = i;
                minE = E;
            else
                candidates(end+1) = i;
            end
        end
        
        energies(i) = E;
    end
    
    % do sample control and pick colour
    if L(1) ~= 0
        energies = wc.*(1-(energies./max(max(energies),1)));
        candidates = find(energies == max(energies));
        newC = candidates(max(ceil(rand()*numel(candidates)),1));
        colCounter(newC) = colCounter(newC) + 1;
        wc(newC) = 1 - colCounter(newC)/L(newC);
    else
        newC = candidates(max(ceil(rand()*numel(candidates)),1));
    end
    
    %  update pattern counters
    for k=1:numP
        tmp = countNeighbours(result,newP,D,S{k},newC,Wc);
        if tmp > 0
            P(k) = P(k) + 1;
        end
    end
    TP = sum(P);
    
    % insert colour into result
    result(newP(1),newP(2)) = newC;
    [emptyCR emptyCC] = find(result==0);
end

end

% counter function
function counter = countNeighbours(points,point,dim,P,col,Wc)

counter = 0;
for i=1:2 % forward/backward counter
    Pi = P{i};
    for k=1:size(Pi,2)
        c = point(2)+Pi(1,k);
        r = point(1)+Pi(2,k);
        if r > 0 && r <= dim(1) && c > 0 && c <= dim(2) && ...
                points(r,c) ~= 0 && Wc(col,points(r,c)) > 0
            counter = counter + Wc(col,points(r,c));
        else
            break;
        end
    end
end

end