#!/bin/sh
###############################################################################
#                                                                             #
#   Copyright 2005 University of Cambridge Computer Laboratory.               #
#                                                                             #
#   This file is part of Nprobe.                                              #
#                                                                             #
#   Nprobe is free software; you can redistribute it and/or modify            #
#   it under the terms of the GNU General Public License as published by      #
#   the Free Software Foundation; either version 2 of the License, or         #
#   (at your option) any later version.                                       #
#                                                                             #
#   Nprobe is distributed in the hope that it will be useful,                 #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of            #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
#   GNU General Public License for more details.                              #
#                                                                             #
#   You should have received a copy of the GNU General Public License         #
#   along with Nprobe; if not, write to the Free Software                     #
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA #
#                                                                             #
###############################################################################


# Compute architecture we are running on

# The user can force the result by setting the environment variable
# SYS_ARCH_* as appropriate.

USAGE='echo 1>&2 "Usage: `basename $0` [ -r | -k | -bin | -kbin | -binv | -kbinv ]
	-r      reports valid architectures
	-k      reports the kernel architecture
	-bin    report the directory in which binaries live
	-kbin   report the directory in which kernel specific binaries live
	-binv   same as -bin, but with OS version number appended
	-kbinv  same as -kbin, but with OS version number appended"'

# Use 'eval "$USAGE"' to print the usage string (don't execute basename unless needed).


valid="alpha arm hp9000s300 hp9000s700 hp9000s800 ia64 ix86 mips sun3 sun3x sun4 sun4c sun4d sun4e sun4m sun4u vax"

uname=/bin/uname

# If the variable SYS_ARCH_SET is set to 1, we are being invited to set values
# in the environment for the benefit of future calls. Other arguments should be
# ignored in this case, since we will have been called as a "." command. Also
# must be careful not to "exit".

# Otherwise decode the arguments, and return values directly if they are set
# in the environment.

if	[ "$SYS_ARCH_SET" != "1" ]
then
	arch="$SYS_ARCH_USER"
	karch="$SYS_ARCH_KERNEL"
	bin="$SYS_ARCH_BIN"
	kbin="$SYS_ARCH_KBIN"
	version="$SYS_ARCH_VERSION"
	case $# in
	0)	level=user
		case "$arch" in ?*) echo "$arch"; exit 0;; esac
		;;
	1)	case "$1" in
		-r)	echo $valid
			exit 0
			;;
		-k)	level=kernel
			case "$karch" in ?*) echo "$karch"; exit 0;; esac
			;;
		-bin)	level=bin
			case "$bin" in ?*) echo "$bin"; exit 0;; esac
			;;
		-kbin)	level=kbin
			case "$kbin" in ?*) echo "$kbin"; exit 0;; esac
			;;
		-binv)	level=binv
			case "${bin}_$version" in ?*_?*) echo "${bin}_$version"; exit 0;; esac
			;;
		-kbinv)	level=kbinv
			case "${kbin}_$version" in ?*_?*) echo "${kbin}_$version"; exit 0;; esac
			;;
		*)	eval "$USAGE";
			exit 2
			;;
		esac ;;
	*)	eval "$USAGE";
		exit 2 ;;
	esac
fi

# Now get the answers using the uname program.

set `$uname -s -r -m`

system=$1
version=$2
machine=$3

case "$version" in
[Vv][0-9]*)	version=`expr "$version" : ".\(.*\)"`;;
esac

case "$system" in

HP-UX)		case "$machine" in
		9000/3*)	arch=hp9000s300; bin=hp300_hpux;;
		9000/7*)	arch=hp9000s700; bin=hp700_hpux;;
		9000/8*)	arch=hp9000s800; bin=hp800_hpux;;
		*)		echo 1>&2 "Unknown $system machine $machine"; exit 3;;
		esac
		karch=$arch
		kbin=$bin;;

OSF1)		case "$machine" in
		alpha)		arch=alpha; bin=alpha_osf1;;
		*)		echo 1>&2 "Unknown $system machine $machine"; exit 3;;
		esac
		karch=$arch
		kbin=$bin;;

riscix)		case "$machine" in
		arm)		arch=arm; bin=arm_riscix;;
		*)		echo 1>&2 "Unknown $system machine $machine"; exit 3;;
		esac
		karch=$arch
		kbin=$bin;;

SunOS)		case "$version" in
		[1-4].*)	os=sos;;  # Versions below 5.0 are "sos".
		*)		os=sosV;; # 5.0 and above are "sosV" (System V).
		esac
		case "$machine" in
		sun3)		arch=sun3; bin=sun3_$os; karch=sun3;  kbin=sun3_$os;;
		sun3x)		arch=sun3; bin=sun3_$os; karch=sun3x; kbin=sun3x_$os;;
		sun4)		arch=sun4; bin=sun4_$os; karch=sun4;  kbin=sun4_$os;;
		sun4c)		arch=sun4; bin=sun4_$os; karch=sun4c; kbin=sun4c_$os;;
		sun4d)		arch=sun4; bin=sun4_$os; karch=sun4d; kbin=sun4d_$os;;
		sun4e)		arch=sun4; bin=sun4_$os; karch=sun4e; kbin=sun4e_$os;;
		sun4m)		arch=sun4; bin=sun4_$os; karch=sun4m; kbin=sun4m_$os;;
		sun4u)		arch=sun4; bin=sun4_$os; karch=sun4u; kbin=sun4u_$os;;
		tp_s2)		arch=sun4; bin=sun4_$os; karch=sun4m; kbin=sun4m_$os;;
		*)		echo 1>&2 "Unknown $system machine $machine"; exit 3;;
		esac;;

ULTRIX*)	case "$machine" in
		MIPS|RISC)	arch=mips; bin=mips_ultrix;;
		VAX)		arch=vax; bin=vax_ultrix;;
		*)		echo 1>&2 "Unknown $system machine $machine"; exit 3;;
		esac
		karch=$arch
		kbin=$bin;;

Linux)		case "$machine" in
		alpha)		arch=$machine; bin=${machine}_linux;;
		i[3456]86)	arch=ix86; machine=$arch; bin=${machine}_linux;;
		ia64)		arch=ia64; machine=$arch; bin=${machine}_linux;;
		sparc)		arch=$machine; bin=${machine}_linux;;
		*)		echo 1>&2 "Unknown $system machine $machine"; exit 3;;
		esac
		karch=$arch
		kbin=$bin;;

IRIX|IRIX64)	arch=`uname -p`;
		case "$arch+$machine" in
		mips+*)		bin=${arch}_irix;;
                *)              echo 1>&2 "Unknown $system machine $machine / $arch"; exit 3;;
                esac
                karch=$arch
                kbin=$bin;;

*)		echo 1>&2 "Unknown operating system: $system"; exit 3;;
esac

# Finally, either set the environment (and unset the variable which asked for
# that to be done), or return the answer.

if	[ "$SYS_ARCH_SET" != "1" ]
then	case $level in
	user)	echo $arch;;
	kernel)	echo $karch;;
	bin)	echo $bin;;
	kbin)	echo $kbin;;
	binv)	echo ${bin}_$version;;
	kbinv)	echo ${kbin}_$version;;
	esac
else	SYS_ARCH_USER=$arch
	SYS_ARCH_KERNEL=$karch
	SYS_ARCH_BIN=$bin
	SYS_ARCH_KBIN=$kbin
	SYS_ARCH_VERSION=$version
	export SYS_ARCH_USER SYS_ARCH_KERNEL SYS_ARCH_BIN SYS_ARCH_KBIN SYS_ARCH_VERSION
	unset SYS_ARCH_SET USAGE
fi
