/*
 * Copyright (c) 1993, 1994, 1995, 1996, 1997, 1998
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the Computer Systems
 *	Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: pcap.c,v 1.29 98/07/12 13:15:39 leres Exp $ (LBL)";
#endif

#include <sys/types.h>

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "pcap-int.h"

#define GET_RUSAGE
      
#ifdef GET_RUSAGE

#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>


long long
utvsub(struct timeval *t1, struct timeval *t0)
{
  struct timeval tdiff;
  tdiff.tv_sec = t1->tv_sec - t0->tv_sec;
  tdiff.tv_usec = t1->tv_usec - t0->tv_usec;
  if (tdiff.tv_usec < 0)
    tdiff.tv_sec--, tdiff.tv_usec += 1000000;
  
  return ((long long)tdiff.tv_sec*1000000) + tdiff.tv_usec;
}
char *
time_string(struct timeval *tvp)
{
  static char tbuf[1024];
  char *tbp = tbuf;
  time_t tm = tvp->tv_sec;
  struct tm *time = gmtime(&tm);
  tbp += strftime(tbuf, 1024, "%a %d %b %H:%M:%S", time);
  sprintf(tbp, ".%.3lu", tvp->tv_usec/1000);

  return tbuf;
}
  

#endif /* ifdef GET_RUSAGE */

int
pcap_dispatch(pcap_t *p, int cnt, pcap_handler callback, u_char *user)
{

	if (p->sf.rfile != NULL)
		return (pcap_offline_read(p, cnt, callback, user));
	return (pcap_read(p, cnt, callback, user));
}

int
pcap_loop(pcap_t *p, int cnt, pcap_handler callback, u_char *user)
{
  register int n;

  for (;;) 
    {
      if (p->sf.rfile != NULL)
	n = pcap_offline_read(p, cnt, callback, user);
      else 
	{
	  /*
	   * XXX keep reading until we get something (or an error occurs)
	   */
	  do 
	    {
	      n = pcap_read(p, cnt, callback, user);
	    } while (n == 0);
	}
      if (n <= 0)
	return (n);
      if (cnt > 0) 
	{
	  cnt -= n;
	  if (cnt <= 0)
	    return (0);
	}
      
#ifdef GET_RUSAGE
      {
	long long udiff;
	static int started = 0;
	struct timeval now;
	static struct timeval start, last;
	struct rusage rnow;
	static struct rusage rlast;
	static FILE *ucpuf, *scpuf;
	
	if (gettimeofday(&now,(struct timezone *)0) != 0)
	  {
	    fprintf (stderr, "ERROR pcap_loop: gettimeofday() %s\n", 
		     strerror(errno));
	    exit (1);
	  }
	
	if (getrusage(RUSAGE_SELF, &rnow) != 0)
	  {
	    fprintf (stderr, "ERROR pcap_loop: getrusage() %s\n", 
		     strerror(errno));
	    exit (1);
	  }

	udiff = utvsub(&now, &last);
	//fprintf (stderr, "%ull\n", udiff);
	
	
	if (!started) 
	  {
	    fprintf (stderr, "Rusage start %s\n", time_string(&now));
	    start = now; /* struct */
	    last = now; /* struct */
	    rlast = rnow;
	    if ((ucpuf = fopen("/tmp/td-%ucpu", "w")) == NULL)
	      {
		fprintf (stderr, "ERROR pcap_loop: fopen() %s\n", 
			 strerror(errno));
		exit (1);
	      }
	    fprintf (ucpuf, "#Plot-title=tcpdump user %% CPU\n");
	    fprintf (ucpuf, "#Plot-xlab=%s\n");
	    fprintf (ucpuf, "#Start %s", time_string(&now));
	    if ((scpuf = fopen("/tmp/td-%scpu", "w")) == NULL)
	      {
		fprintf (stderr, "ERROR pcap_loop: fopen() %s\n", 
			 strerror(errno));
		exit (1);
	      }
	    fprintf (scpuf, "#Plot-title=tcpdump sys %% CPU\n");
	    fprintf (scpuf, "#Plot-xlab=%s\n");
	    fprintf (scpuf, "#Start %s", time_string(&now));
	    started++;
	  }
	
	else if (udiff > 2000000ULL)
	  {
	    double usr = (utvsub(&rnow.ru_utime, &rlast.ru_utime)*100.0)/udiff;
	    double sys = (utvsub(&rnow.ru_stime, &rlast.ru_stime)*100.0)/udiff;
	    float rel = utvsub(&now, &start)/1000000.00;
	    fprintf(stderr, "%0.2f Usr %0.8f%% Sys %0.8f%%\n", rel, usr, sys);
	    fprintf(ucpuf, "%0.2f %0.2f\n", rel, usr);
	    fprintf(scpuf, "%0.2f %0.2f\n", rel, sys);
	    fflush(ucpuf);
	    fflush(scpuf);
	    last.tv_sec += 2;
	    rlast = rnow;
	  }
	
      }
     
#endif /* ifdef GET_RUSAGE */
    }
}

struct singleton {
  struct pcap_pkthdr *hdr;
  const u_char *pkt;
};


static void
pcap_oneshot(u_char *userData, const struct pcap_pkthdr *h, const u_char *pkt)
{
	struct singleton *sp = (struct singleton *)userData;
	*sp->hdr = *h;
	sp->pkt = pkt;
}

const u_char *
pcap_next(pcap_t *p, struct pcap_pkthdr *h)
{
	struct singleton s;

	s.hdr = h;
	if (pcap_dispatch(p, 1, pcap_oneshot, (u_char*)&s) <= 0)
		return (0);
	return (s.pkt);
}

int
pcap_datalink(pcap_t *p)
{
	return (p->linktype);
}

int
pcap_snapshot(pcap_t *p)
{
	return (p->snapshot);
}

int
pcap_is_swapped(pcap_t *p)
{
	return (p->sf.swapped);
}

int
pcap_major_version(pcap_t *p)
{
	return (p->sf.version_major);
}

int
pcap_minor_version(pcap_t *p)
{
	return (p->sf.version_minor);
}

FILE *
pcap_file(pcap_t *p)
{
	return (p->sf.rfile);
}

int
pcap_fileno(pcap_t *p)
{
	return (p->fd);
}

void
pcap_perror(pcap_t *p, char *prefix)
{
	fprintf(stderr, "%s: %s\n", prefix, p->errbuf);
}

char *
pcap_geterr(pcap_t *p)
{
	return (p->errbuf);
}

/*
 * Not all systems have strerror().
 */
char *
pcap_strerror(int errnum)
{
#ifdef HAVE_STRERROR
	return (strerror(errnum));
#else
	extern int sys_nerr;
	extern const char *const sys_errlist[];
	static char ebuf[20];

	if ((unsigned int)errnum < sys_nerr)
		return ((char *)sys_errlist[errnum]);
	(void)sprintf(ebuf, "Unknown error: %d", errnum);
	return(ebuf);
#endif
}

void
pcap_close(pcap_t *p)
{
	/*XXX*/
	if (p->fd >= 0)
		close(p->fd);
	if (p->sf.rfile != NULL) {
		(void)fclose(p->sf.rfile);
		if (p->sf.base != NULL)
			free(p->sf.base);
	} else if (p->buffer != NULL)
		free(p->buffer);
#ifdef linux
	if (p->md.device != NULL)
		free(p->md.device);
#endif
	
	free(p);
}
