/* (C) Cambridge University Computer Laboratory, 2000
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <linux/in.h>
#include <errno.h>
#include <stdlib.h>

#include "ecn.h"


/* Set the TOS byte on future packets sent on "fd" to "tos", which may
 * be the special value -1 to reset the TOS to the system's default
 * value.  Returns 0 for success, otherwise -1 and sets errno to one of:
 *  EBADF:	"fd" is not a file descriptor
 *  ENOTSOCK:	"fd" is a file descriptor for something other than a socket
 *  ENOPROTOOPT: this kernel does not support setting the TOS byte
 *  EFAULT:	can't happen - error in the code below
 *  EINVAL:	attempted to set a reserved bit in TOS byte (eg bottom bit)
 *  EPERM:	attempted to set TOS precedence higher than FLASHOVERRIDE
 *  		without network admin privileges.
 */
int
socket_settos (int fd, unsigned char tos)
{
    return setsockopt (fd, IPPROTO_IP, IP_TOS, &tos, sizeof (tos));
}


/* Find out the current TOS setting on the socket "fd".  Returns the
 * TOS byte that will be used for future packets sent on "fd", or -1
 * if there was an error, in which case errno will be set to one of:
 *  EBADF:	"fd" is not a file descriptor
 *  ENOTSOCK:	"fd" is a file descriptor but not on a socket
 *  ENOPROTOOPT: this kernel doesn't support reading the outgoing TOS byte
 *  EFAULT:	can't happen - error in thje code below
 */
int
socket_gettos (int fd)
{
    int ret;
    int buflen;
    unsigned char tos;

    buflen = sizeof(tos);
    ret = getsockopt (fd, IPPROTO_IP, IP_TOS, &tos, &buflen);
    if (ret < 0)
	return -1; /* error */

    if (buflen != 1)
    {
	errno = EFAULT;
	return -1;
    }

    return tos;
}

/* Turn on the reception of TOS bytes for socket "fd".  Return 0 for success,
 * or -1 on failure, setting errno to:
 *  EBADF:	"fd" is not a file descriptor
 *  ENOTSOCK:	"fd" is a file descriptor but not on a socket
 *  ENOPROTOOPT: this kernel doesn't support reading the outgoing TOS byte
 *  EFAULT:	can't happen - error in thje code below
 */
int
socket_setrecvtos (int fd, int on)
{
    return setsockopt (fd, IPPROTO_IP, IP_RECVTOS, &on, sizeof (on));
}
