﻿// $Id: DcbSuperResolutionUpsampler.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VideoLib.Parameters;
using Stereo.GpGpuLib;

namespace VideoLib.Stereo.GpGpu.StereoNodes
{
	/// <summary>
	/// Upsamples images using spatial-depth super resolution based on the DCB grid.
	/// </summary>
	class DcbSuperResolutionUpsampler : UpsamplerStereoNode, IDisposable
	{
		#region Parameters

		[Parameter(Minimum = 0, Maximum = 10, Default = 2)]
		public int Iterations { get; set; }

		[Parameter(Minimum = 1, Maximum = 100, Default = 8)]
		public float ClipCosts { get; set; }

		[Parameter(Minimum = 5, Maximum = 20, Default = 10, FriendlyName = "Spatial sigma")]
		public int SigmaS { get; set; }

		[Parameter(Minimum = 5, Maximum = 20, Default = 10, FriendlyName = "Colour sigma")]
		public int SigmaC { get; set; }

		#endregion

		private DepthMap result = new DepthMap();
		private CostSpace tempCosts = new CostSpace();
		private InputImage tempGrid = new InputImage();

		/// <summary>
		/// Initializes a new instance of the <see cref="JointBilateralUpsampler"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public DcbSuperResolutionUpsampler(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Up-samples the depth map.
		/// </summary>
		/// <param name="depthMap">The low resolution input depth map.</param>
		/// <param name="originalImage">The original image.</param>
		/// <param name="depthMapOut">The up-sampled output depth map.</param>
		public override void UpsampleDepthMap(DepthMap depthMap, int inputDisparities, InputImage leftImage, InputImage rightImage, out DepthMap depthMapOut, int outputDisparities)
		{
			if ((leftImage.Width == depthMap.Width) && (leftImage.Height == depthMap.Height))
				depthMapOut = depthMap;
			else
			{
				ManagedUpsamplers.DcbUpsample(depthMap, inputDisparities, leftImage, rightImage, tempCosts, tempGrid, result, outputDisparities, ClipCosts, SigmaS, SigmaC, (uint)Iterations);
				depthMapOut = result;
			}
		}

		public void Dispose()
		{
			result.Dispose();
			tempCosts.Dispose();
			tempGrid.Dispose();
			result = null;
			tempCosts = null;
			tempGrid = null;
		}
	}

	/// <summary>
	/// Factory that produces a <see cref="DcbSuperResolutionUpsampler"/>.
	/// </summary>
	class DcbSuperResolutionUpsamplerFactory : StereoNodeFactory<UpsamplerStereoNode>
	{
		/// <summary>
		/// Creates a <see cref="DcbSuperResolutionUpsampler"/> to run on the specified execution stream.
		/// </summary>
		/// <param name="stream">The execution stream.</param>
		/// <returns>
		/// An instance of <see cref="UpsamplerStereoNode"/>.
		/// </returns>
		public override UpsamplerStereoNode Create(ExecutionStream stream)
		{
			return new DcbSuperResolutionUpsampler(stream);
		}

		/// <summary>
		/// Checks if the <see cref="DcbSuperResolutionUpsampler"/> is compatible with the given datatypes.
		/// </summary>
		/// <param name="imageType">Image datatype.</param>
		/// <param name="gridType">Cost space datatype.</param>
		/// <param name="mapType">Depth map datatype.</param>
		public override void CheckValid(InputImageType imageType, CostSpaceType gridType, DepthMapType mapType)
		{
			isValid = (imageType == InputImageType.Xrgb_uint && mapType == DepthMapType.Single);
		}

		/// <summary>
		/// Gets the name of this node.
		/// </summary>
		public override string Name { get { return "DCB super-resolution"; } }
	}
}
