﻿// $Id: StereoNode.cs 65 2010-03-18 17:06:22Z cr333 $
using System.Collections.Generic;
using Stereo.GpGpuLib;
using VideoLib.Parameters;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// Represents 'stereo nodes' which are part of the depth map generation process, the details of which
	/// are hidden apart from the type of process that they perform.
	/// </summary>
	public interface IStereoNode
	{
		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		TechniqueType Type { get; }

		/// <summary>
		/// Gets a list of nodes that should have their parameters modified in-sync with the current node.
		/// </summary>
		List<object> FollowerNodes { get; }
	}

	/// <summary>
	/// The base class for a node performing an operation in the stereo image production process.
	/// </summary>
	abstract class StereoNode : IStereoNode
	{
		protected readonly ExecutionStream stream;

		/// <summary>
		/// Initializes a new instance of the <see cref="StereoNode"/> class.
		/// </summary>
		public StereoNode() : this(ExecutionStream.NullStream) { }

		/// <summary>
		/// Initializes a new instance of the <see cref="StereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public StereoNode(ExecutionStream theStream) 
		{ 
			stream = theStream;
			ParameterReflector.ApplyDefaults(this);
		}

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		/// <value></value>
		public abstract TechniqueType Type { get; }

		/// <summary>
		/// Gets or sets a list of nodes that should have their parameters modified in-sync with the current node.
		/// </summary>
		public List<object> FollowerNodes { get; internal set; }
	}

	#region Specializations

	abstract class DownsamplerStereoNode : StereoNode
	{
		#region Parameters

		/// <summary>Gets or sets the factor to downsample the images by</summary>
		[Parameter(Minimum = 1, Maximum = 16, Default = 2, FriendlyName = "Scale factor")]
		public virtual int DownsamplingFactor { get; set; }

		#endregion

		/// <summary>
		/// Initializes a new instance of the <see cref="DownsamplerStereoNode"/> class.
		/// </summary>
		public DownsamplerStereoNode() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="DownsamplerStereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public DownsamplerStereoNode(ExecutionStream theStream) : base(theStream) {  }

		/// <summary>
		/// Downsamples the pair of input images
		/// </summary>
		/// <param name="leftImage">The left image.</param>
		/// <param name="rightImage">The right image.</param>
		/// <param name="dsFactor">Is set to the down-sampling factor actually used</param>
		/// <param name="sampledLeftImage">The downsampled left image output</param>
		/// <param name="sampledRightImage">The downsampled right image output</param>
		public abstract void DownsampleImages(InputImage leftImage, InputImage rightImage, out int dsFactor, 
			out InputImage sampledLeftImage, out InputImage sampledRightImage);

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		public override TechniqueType Type { get { return TechniqueType.Downsampler; } }
	}

	/// <summary>
	/// Base class for all pre-processing <see cref="StereoNode"/>s.
	/// </summary>
	abstract class PreProcessorStereoNode : StereoNode
	{
		/// <summary>
		/// Initializes a new instance of the <see cref="PreProcessorStereoNode"/> class.
		/// </summary>
		public PreProcessorStereoNode() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="PreProcessorStereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public PreProcessorStereoNode(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Pre-processes the given stereo image pair.
		/// </summary>
		/// <param name="leftImage">The left image.</param>
		/// <param name="rightImage">The right image.</param>
		public abstract void PreProcessImages(InputImage leftImage, InputImage rightImage);

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		public override TechniqueType Type { get { return TechniqueType.PreProcessor; } }
	}

	/// <summary>
	/// Base class for all cost computation <see cref="StereoNode"/>s.
	/// </summary>
	abstract class CostComputerStereoNode : StereoNode
	{
		#region Parameters

		/// <summary>Gets or sets the number of disparity values to use</summary>
		[Parameter(Minimum = 2, Maximum = 128, Default = 64, FriendlyName = "Disparity levels")]
		public int NDisparityValues { get; set; }

		#endregion

		/// <summary>
		/// Initializes a new instance of the <see cref="CostComputerStereoNode"/> class.
		/// </summary>
		public CostComputerStereoNode() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="CostComputerStereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public CostComputerStereoNode(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Computes the costs in the left image.
		/// </summary>
		/// <param name="leftImage">The left input image.</param>
		/// <param name="rightImage">The right input image.</param>
		/// <param name="costGrid">The cost grid to produce.</param>
		/// <param name="downsampleFactor">The factor to reduce the number of disparity levels by</param>
		public void ComputeCosts(InputImage leftImage, InputImage rightImage, CostSpace costGrid, int downsampleFactor, out int nDisparities)
		{
			nDisparities = NDisparityValues / downsampleFactor;
			ComputeCostsInner(leftImage, rightImage, costGrid, nDisparities);
		}

		protected abstract void ComputeCostsInner(InputImage leftImage, InputImage rightImage, CostSpace costGrid, int nDisparities);

		/// <summary>
		/// Gets the type of the cost space.
		/// </summary>
		public abstract CostSpaceType GetCostSpaceType();

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		public override TechniqueType Type { get { return TechniqueType.CostComputer; } }
	}

	/// <summary>
	/// Base class for all cost aggregation <see cref="StereoNode"/>s.
	/// </summary>
	abstract class AggregatorStereoNode : StereoNode
	{
		/// <summary>
		/// Initializes a new instance of the <see cref="AggregatorStereoNode"/> class.
		/// </summary>
		public AggregatorStereoNode() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="AggregatorStereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public AggregatorStereoNode(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Aggregates (locally optimizes) the costs.
		/// </summary>
		/// <param name="leftImage">The left input image.</param>
		/// <param name="rightImage">The right input image.</param>
		/// <param name="costGrid">The cost grid.</param>
		public abstract void AggregateCosts(InputImage leftImage, InputImage rightImage, CostSpace costGrid);

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		public override TechniqueType Type { get { return TechniqueType.Aggregator; } }
	}

	/// <summary>
	/// Base class for all maximisation <see cref="StereoNode"/>s.
	/// </summary>
	abstract class MaximizerStereoNode : StereoNode
	{
		/// <summary>
		/// Initializes a new instance of the <see cref="MaximizerStereoNode"/> class.
		/// </summary>
		public MaximizerStereoNode() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="MaximizerStereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public MaximizerStereoNode(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Finds the best solution.
		/// </summary>
		/// <param name="costGrid">The cost grid.</param>
		/// <param name="depthMap">The depth map.</param>
		public abstract void FindSolution(CostSpace costGrid, DepthMap depthMap);

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		public override TechniqueType Type { get { return TechniqueType.Maximizer; } }
	}

	abstract class UpsamplerStereoNode : StereoNode
	{
		/// <summary>
		/// Initializes a new instance of the <see cref="UpsamplerStereoNode"/> class.
		/// </summary>
		public UpsamplerStereoNode() : base() { }

		/// <summary>
		/// Initializes a new instance of the <see cref="UpsamplerStereoNode"/> class.
		/// </summary>
		/// <param name="theStream">The execution stream to run this node on.</param>
		public UpsamplerStereoNode(ExecutionStream theStream) : base(theStream) { }

		/// <summary>
		/// Up-samples the depth map
		/// </summary>
		/// <param name="depthMap">The low resolution input depth map</param>
		/// <param name="inputDisparities">The number of input disparities.</param>
		/// <param name="leftImage">The original left input image.</param>
		/// <param name="rightImage">The original right input image.</param>
		/// <param name="depthMapOut">The up-sampled output depth map</param>
		/// <param name="outputDisparities">The number of output disparities.</param>
		public abstract void UpsampleDepthMap(DepthMap depthMap, int inputDisparities, InputImage leftImage, InputImage rightImage, out DepthMap depthMapOut, int outputDisparities);

		/// <summary>
		/// Gets the type of technique/sub-process implemented by the stereo node.
		/// </summary>
		public override TechniqueType Type { get { return TechniqueType.Upsampler; } }
	}

	#endregion
}
