﻿// $Id: DisparityEstimationNode.cs 65 2010-03-18 17:06:22Z cr333 $
using System;
using System.Collections.Generic;
using System.Threading;
using VideoLib.Frames;
using VideoLib.Parameters;

namespace VideoLib.Stereo.GpGpu
{
	/// <summary>
	/// A convenient data structure to pass disparity map processing data / references to data betweeen threads
	/// </summary>
	struct DisparityCallData
	{
		public byte[] inImageL;
		public byte[] inImageR;
		public int width;
		public int height;
		public byte[] outImage;
	}

	/// <summary>
	/// Provides a VideoLib.Node that implements disparity map generation, using an underlying instance of
	/// <see cref="DisparityEstimator"/>. Exposes parameters marked with the attributes given in VideoLib.Parameters.
	/// </summary>
	public class DisparityEstimationNode : Node
	{
		private DisparityEstimatorBase disparityEstimator = null;

		#region CUDA access thread + thread control

		private Thread cudaAccessThread;

		private DisparityCallData processData;
		private bool readyToProcess;
		private bool cudaAccessExit;

		private List<TechniqueDescription> todoQueue = new List<TechniqueDescription>();

		#endregion

		/// <summary>
		/// Represents a method handler (used by <see cref="Initialized"/>)
		/// </summary>
		/// <param name="sender">The instance of this class that triggered the event</param>
		public delegate void InitializedHandler(DisparityEstimationNode sender);

		/// <summary>
		/// Occurs once only when the node is initialized by the VideoLib framework
		/// </summary>
		public event InitializedHandler Initialized;

		/// <summary>
		/// Creates an instance of FlexibleDisparityEstimationNode
		/// </summary>
		public DisparityEstimationNode()
		{
			readyToProcess = false;
			cudaAccessExit = false;
			processData = new DisparityCallData();

			ParameterReflector.ApplyDefaults(this);

			AddInputPins("inL", "inR");
			AddOutputPin("out");
		}

		// The main loop for the exclusive library access thread
		private void CudaAccessMain()
		{
			lock(this)
			{
				try
				{
					while (!cudaAccessExit)
					{
						while (readyToProcess && !cudaAccessExit)
							Monitor.Wait(this);

						// If there are any technique settings to apply, set them (in order of arrival)
						if (todoQueue.Count > 0)
						{
							for (int i = 0; i < todoQueue.Count; ++i)
								disparityEstimator.SetTechnique(todoQueue[i]);
							todoQueue.Clear();
						}

						// Actually process the frame here
						if (!cudaAccessExit)
							disparityEstimator.ProcessFrame(processData.inImageL, processData.inImageR, processData.width, processData.height, processData.outImage);

						readyToProcess = true;
						Monitor.PulseAll(this);
					}
				}
				finally
				{
					// Dispose of the disparity estimator, if available
					if (disparityEstimator != null)
					{
						disparityEstimator.Dispose();
						disparityEstimator = null;
					}
				}
			}
		}

		// Fires the Initialized event
		private void OnInitialized()
		{
			if (Initialized != null)
				Initialized(this);
		}

		/// <summary>
		/// Releases unmanaged resources associated with the object
		/// </summary>
		public override void Dispose()
		{
			if(cudaAccessThread != null)
			{
				// Tell the worker thread to exit, and wait until it does
				lock(this)
				{
					cudaAccessExit = true;
					Monitor.PulseAll(this);
				}
				cudaAccessThread.Join();
			}

			base.Dispose();
		}
		/// <summary>
		/// Prepares the node to receive image frames
		/// </summary>
		public override void Init()
		{
			lock(this)
			{
				// Create the underlying disparity estimation object
				if (disparityEstimator == null)
					disparityEstimator = DisparityEstimator.CreateStandard();

				// Initialize and start the spinning worker thread
				readyToProcess = true;
				cudaAccessExit = false;
				cudaAccessThread = new Thread(new ThreadStart(CudaAccessMain));
				cudaAccessThread.Name = "Cuda access thread [Disparities]";
				cudaAccessThread.Start();
			}

			base.Init();
			OnInitialized();
		}

		/// <summary>
		/// Pushes a frame which is a depthmap estimate from the two input frames to the next stage
		/// <para/>The output frame is in the reference frame of the left input "inL"
		/// </summary>
		public override void Process()
		{
			// Get input frames
			Frame inL = GetInputFrame("inL");
			Frame inR = GetInputFrame("inR");
			if (PassOnEndOfStreamFrame(inL) || PassOnEndOfStreamFrame(inR)) 
				return;

			// Get the input bitmap data
			BitmapFrame bitmapL = CastFrameTo<BitmapFrame>(inL);
			BitmapFrame bitmapR = CastFrameTo<BitmapFrame>(inR);

			// Check for same size
			if (bitmapL.Width != bitmapR.Width || bitmapL.Height != bitmapR.Height)
				throw new Exception("Both input frames must have same size.");

			// Check for same format
			if (bitmapL.Format != PixelFormat.RGB32 || bitmapR.Format != PixelFormat.RGB32)
				throw new Exception("Both input frames must have same in RGB32 format.");

			// Check for same sequence number
			if (bitmapL.SequenceNumber != bitmapR.SequenceNumber)
				throw new Exception("Both input frames must have same sequence number.");

			// Copy input images to unsigned integer arrays, as this is what the CUDA code expects
			int w = bitmapL.Width;
			int h = bitmapL.Height;

			// The disparity map generation is done on another thread - all the cuda accesses need to be
			// done on the same thread (but this method is called by multiple threads)
			BitmapFrame outputFrame = new BitmapFrame(w, h, PixelFormat.Float) { SequenceNumber = inL.SequenceNumber };
			lock(this)
			{
				while(!readyToProcess && !cudaAccessExit)
					Monitor.Wait(this);

				if(!cudaAccessExit)
				{
					// Prepare the dataset for the processing thread to work on
					processData.inImageL = bitmapL.Data;
					processData.inImageR = bitmapR.Data;
					processData.width = w; 
					processData.height = h;
					processData.outImage = outputFrame.Data;

					// Notify the other thread of the need to start processing
					readyToProcess = false;
					Monitor.PulseAll(this);
					
					// Wait for the operation to complete
					while(!readyToProcess)
						Monitor.Wait(this);
				}
			}

			// Send off the frame
			PushFrame(outputFrame);
		}

		/// <summary>
		/// Gets a list of currently active 'parameter target' objects (stereo nodes)
		/// </summary>
		public RegisteredCollection<IStereoNode> ActiveParamObjects
		{
			get { return (disparityEstimator == null ? null : disparityEstimator.ActiveParamObjects); }
		}

		/// <summary>
		/// Gets a list of depthmap creation techniques that are compatible with the current options
		/// </summary>
		/// <param name="type">The class of technique (e.g. Cost space aggregator)</param>
		/// <returns>A list of valid techniques to apply</returns>
		public List<TechniqueDescription> GetTechniques(TechniqueType type)
		{
			return (disparityEstimator == null ? new List<TechniqueDescription>() : disparityEstimator.GetTechniques(type));
		}

		/// <summary>
		/// Sets the specified technique to perform a stage in the depth-map generation process
		/// </summary>
		/// <param name="description">
		/// A description of the technique to be set (obtained from the list retrieved by calling <see cref="GetTechniques"/>)
		/// </param>
		public void SetTechnique(TechniqueDescription description)
		{
			if (disparityEstimator != null)
				lock (this)
					todoQueue.Add(description);
		}

		/// <summary>
		/// Adds a 'frame debug handler', which is presented with the results of stereo processing on a single frame (the next one to arrive).
		/// <para>This only lasts for one frame (i.e. 1* handler call), after which the handler is removed.</para>
		/// </summary>
		/// <param name="handler">
		/// An event handler, which will receive a single update of processed frame data for the next frame that is pushed through
		/// </param>
		public void AddFrameDebugHandler(DisparityEstimatorBase.FrameDebugHandler handler)
		{
			if (disparityEstimator != null)
				lock (this)
					disparityEstimator.NextFrameDebug += handler;
		}
	}
}
