#pragma once

#include "UnmanagedParameterTarget.h"

#include "StereoSolverImpls.h"
#include "StereoNodeFactory.h"
#include "StereoNodeFactoryWrap.h"

using namespace System;
using namespace System::Collections::Generic;

namespace Stereo { namespace GpGpuLib {

	public ref struct TechniqueDescription
	{
	public:
		String^ name;
		int index;

		TechniqueDescription(String^ theName, int theIndex) : name(theName), index(theIndex) { }
		virtual String^ ToString() override { return name; }
	};

	/// <summary>
	/// Provides access to the disparity estimation mapping functionality of the library
	/// <p/>Warning: Not thread safe - do not access with multiple threads, even if method invocations
	/// are synchronized. This is because the underlying CUDA has per-thread context, so resources
	/// allocated by one thread are not available to another.
	/// </summary>
	public ref class FlexibleDisparityEstimator
	{
	private:
		StdStereoSolver* solver;

		StereoNodeFactoryListNodeWrap<PreProcStereoNode>^ preFactories;
		StereoNodeFactoryListNodeWrap<CostComputeStereoNode>^ costFactories;
		StereoNodeFactoryListNodeWrap<CostAggregateStereoNode>^ aggregateFactories;
		StereoNodeFactoryListNodeWrap<GlobalOptStereoNode>^ optFactories;
		StereoNodeFactoryListNodeWrap<WtaSolveStereoNode>^ wtaFactories;

		IParameterTarget^ preProcessTarget;
		IParameterTarget^ costComputeTarget;
		IParameterTarget^ aggregateTarget;
		IParameterTarget^ optTarget;
		IParameterTarget^ wtaTarget;

		int preIdx, costIdx, aggIdx, optIdx, wtaIdx;

		void ValidateFactories();

	public:
		FlexibleDisparityEstimator();
		~FlexibleDisparityEstimator();

		/// <summary>
		/// Fills an image with a greyscale depth-disparity map from the two input images
		/// </summary>
		/// <param name="inImageL">The left view (layed out without padding, width-first)</param>
		/// <param name="inImageR">The right view (layed out without padding, width-first)</param>
		/// <param name="width">The width of both of the input images</param>
		/// <param name="height">The height of both of the input images</param>
		/// <param name="ndisparities">The maximum disparity to evaluate</param>
		/// <param name="scaling">The scale factor to generate the map from disparities (commonly = 1/(ndisparities-1))</param>
		/// <param name="outImage">The estimated depth map (should be created beforehand, same dimensions as the two input images)</param>
		void ProcessFrame(array<unsigned int>^ inImageL, array<unsigned int>^ inImageR, 
							int width, int height, int ndisparities, float scaling, array<float>^ outImage);

		List<TechniqueDescription^>^ GetPreprocessTechniques();
		List<TechniqueDescription^>^ GetCostComputeTechniques();
		List<TechniqueDescription^>^ GetCostAggregateTechniques();
		List<TechniqueDescription^>^ GetGlobalOptimizationTechniques();
		List<TechniqueDescription^>^ GetWtaTechniques();

		void SetPreprocessTechnique(TechniqueDescription^ description);
		void SetCostComputeTechnique(TechniqueDescription^ description);
		void SetCostAggregateTechnique(TechniqueDescription^ description);
		void SetGlobalOptimizationTechnique(TechniqueDescription^ description);
		void SetWtaTechnique(TechniqueDescription^ description);

		/// <summary>
		/// The set of mutable parameters relating to the current estimation map generation technique
		/// </summary>
		property List<IParameterTarget^>^ ParameterTargets
		{
			List<IParameterTarget^>^ get() 
			{ 
				List<IParameterTarget^>^ result = gcnew List<IParameterTarget^>();

				if(preProcessTarget != nullptr) result->Add(preProcessTarget);
				if(costComputeTarget != nullptr) result->Add(costComputeTarget);
				if(aggregateTarget != nullptr) result->Add(aggregateTarget);
				if(optTarget != nullptr) result->Add(optTarget);
				if(wtaTarget != nullptr) result->Add(wtaTarget);

				return result;
			}
		}
	};

} }