#pragma once

#include "InputImage.h"
#include "GpuDepthMap.h"

namespace Stereo { namespace GpGpuLib {
using namespace System;

public enum class DepthMapType
{
	Any,
	Single
};

/// <summary>
/// A simple wrapper around the unmanaged depth map for use in managed code
/// </summary>
public ref class DepthMap
{
private:
	GpuDepthMap* depthMap;

public:
	DepthMap()
	{ 
		depthMap = new GpuDepthMap();
	}
	~DepthMap()
	{
		if(depthMap != 0)
			delete depthMap;
		depthMap = 0;
	}

	GpuDepthMap* Get() { return depthMap; }

	property int Width { int get() { return depthMap->GetWidth(); } }
	property int Height { int get() { return depthMap->GetHeight(); } }

	void SizeDepthMap(DepthMapType type, int w, int h)
	{
		depthMap->Create((GpuDepthMapType)((int)type), w, h);
	}
	
	/// <summary>
	/// Swaps the internal GPU memory with that of <paramref name="other"/>
	/// </summary>
	void SwapData(DepthMap^ other)
	{
		depthMap->SwapData(*other->Get());
	}

	/// <summary>
	/// Copies GPU data out into the managed float array given by <paramref name="data"/>
	/// </summary>
	void CopyDataOut(array<float>^ data, int w, int h)
	{
		if(w != depthMap->GetWidth() || h != depthMap->GetHeight())
			throw gcnew Exception(String::Format(
				"The width and/or height of the depth map ({0}x{1}) does not match the output data ({2}x{3}) - cannot copy data", 
				depthMap->GetWidth(), depthMap->GetHeight(), w, h));

		// Pin the managed pointer for the purposes of unmanaged code
		pin_ptr<float> pinData = &data[0];
		depthMap->CopyDataOut(pinData);
	}

	/// <summary>
	/// Copies GPU data out into the managed byte array given by <paramref name="data"/>, treating it internally as an array of floats
	/// </summary>
	void CopyDataOut(array<unsigned char>^ data, int w, int h)
	{
		if(w != depthMap->GetWidth() || h != depthMap->GetHeight())
			throw gcnew Exception(String::Format(
				"The width and/or height of the depth map ({0}x{1}) does not match the output data ({2}x{3}) - cannot copy data", 
				depthMap->GetWidth(), depthMap->GetHeight(), w, h));

		// Pin the managed pointer for the purposes of unmanaged code
		pin_ptr<unsigned char> pinData = &data[0];
		depthMap->CopyDataOut((float*)pinData);
	}
};

} }