// $Id: ManagedAggregators.cpp 65 2010-03-18 17:06:22Z cr333 $
#include "UnmanagedAggregators.h"
#include "ManagedAggregators.h"

using namespace System::Diagnostics;
using namespace Stereo::GpGpuLib;

void ManagedAggregators::RunBoxKernel(CostSpace^ costSpace, int radius, CostSpace^ temp, ExecutionStream^ stream)
{
	GpuCostSpace* gpuCost = costSpace->Get();
	GpuCostSpace* gpuTemp = temp->Get();

	gpuTemp->SizeToMatch(*gpuCost);

	// Run the box aggregation
	RunBoxAggKernel(gpuCost->Get(), gpuCost->GetDepth(), radius, gpuTemp->Get());
	
	// The output is currently in gpuTemp - it needs to be transferred to gpuCost, so the underlying data are swapped
	gpuCost->SwapData(*gpuTemp);
}

void ManagedAggregators::RunShiftKernel(CostSpace^ costSpace, int boxRadius, int minRadius, CostSpace^ temp, ExecutionStream^ stream)
{
	GpuCostSpace* gpuCost = costSpace->Get();
	GpuCostSpace* gpuTemp = temp->Get();

	gpuTemp->SizeToMatch(*gpuCost);

	// Run the shiftable window: composed of a box filter followed by a min-filter
	RunBoxAggKernel(gpuCost->Get(), gpuCost->GetDepth(), boxRadius, gpuTemp->Get());
	RunMinAggKernel(gpuTemp->Get(), gpuCost->GetDepth(), minRadius, gpuCost->Get());

	// The output is currently in gpuTemp - it needs to be transferred to gpuCost, so the underlying data are swapped
	gpuCost->SwapData(*gpuTemp);
}

void ManagedAggregators::RunNaiveDcbKernel(
	CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR,
	unsigned int radius, float sigmaS, float sigmaC, ExecutionStream^ stream)
{
	Debug::Assert(imageL->Width == imageR->Width && imageL->Width == costSpace->Width);
	Debug::Assert(imageL->Height == imageR->Height && imageL->Height == costSpace->Height);
	Debug::Assert(imageL->Pitch == imageR->Pitch);
	Debug::Assert(sigmaS > 0);
	Debug::Assert(sigmaC > 0);

	RunAggregationNaiveDCB(costSpace->GetPointer(), costSpace->Depth,
		imageL->GetPointer(), imageR->GetPointer(), imageL->Pitch,
		imageL->Width, imageL->Height, radius, sigmaS, sigmaC);
}

void ManagedAggregators::RunDcbGridKernel(
	CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, InputImage^ gridTexture,
	float sigmaS, float sigmaC, ExecutionStream^ stream)
{
	Debug::Assert(imageL->Width == imageR->Width && imageL->Width == costSpace->Width);
	Debug::Assert(imageL->Height == imageR->Height && imageL->Height == costSpace->Height);
	Debug::Assert(imageL->Pitch == imageR->Pitch);
	Debug::Assert(sigmaS > 0);
	Debug::Assert(sigmaC > 0);

	// determine size of grid texture and reallocate it if necessary
	unsigned int texWidth = 0, texHeight = 0;
	CalculateGridTextureSize(imageL->Width, imageL->Height, costSpace->Depth, sigmaS, sigmaC, texWidth, texHeight);
	gridTexture->SizeImage(InputImageType::Float2, texWidth, texHeight);

	RunAggregationDCBGrid(costSpace->GetPointer(), costSpace->Depth, imageL->GetPointer(), imageR->GetPointer(), 
		imageL->Pitch, (float2*)gridTexture->GetPointer(), texWidth, texHeight, gridTexture->Pitch, imageL->Width, imageL->Height, sigmaS, sigmaC);
}

void ManagedAggregators::RunTdcbGridKernel(CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, array<InputImage^>^ gridTextures, float sigmaS, float sigmaC, int weighting, float wa, float wb, ExecutionStream^ stream)
{
	Debug::Assert(gridTextures->Length >= 0);

	GpuCostSpace* gpuCost = costSpace->Get();
	GpuImage* gpuImageL = imageL->Get();
	GpuImage* gpuImageR = imageR->Get();

	// determine size of grid texture
	unsigned int texWidth = 0, texHeight = 0;
	CalculateGridTextureSize(gpuImageL->GetWidth(), gpuImageL->GetHeight(), gpuCost->GetDepth(), sigmaS, sigmaC, texWidth, texHeight);

	// make sure that all grids have the correct size
	for(int i = 0; i < gridTextures->Length; i++)
	{
		gridTextures[i]->Get()->Create(IMAGE_TYPE_FLOAT2, texWidth, texHeight);
	}

	// collect grid pointers to pass to runner function
	float2** gpuGrids = new float2*[gridTextures->Length];
	for(int i = 0; i < gridTextures->Length; i++)
	{
		gpuGrids[i] = (float2*)gridTextures[i]->Get()->Get();
	}

	RunAggregationTDCBGrid(gpuCost->Get(), gpuCost->GetDepth(), gpuImageL->Get(), gpuImageR->Get(), 
		gpuImageL->GetPitch(), gpuGrids, gridTextures->Length, texWidth, texHeight, gridTextures[0]->Get()->GetPitch(), gpuImageL->GetWidth(), gpuImageL->GetHeight(), sigmaS, sigmaC, weighting, wa, wb);
}

void ManagedAggregators::RunDcbGrid2Kernel(
	CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR, TiledGrids^ grids,
	float sigmaS, float sigmaC1, float sigmaC2, ExecutionStream^ stream)
{
	Debug::Assert(imageL->Width == imageR->Width && imageL->Width == costSpace->Width);
	Debug::Assert(imageL->Height == imageR->Height && imageL->Height == costSpace->Height);
	Debug::Assert(imageL->Pitch == imageR->Pitch);
	Debug::Assert(grids != nullptr);
	Debug::Assert(sigmaS > 0);
	Debug::Assert(sigmaC1 > 0);
	Debug::Assert(sigmaC2 > 0);

	RunAggregationDCBGrid2(costSpace->GetPointer(), costSpace->Depth, imageL->GetPointer(), imageR->GetPointer(), 
		imageL->Pitch, grids->GetPointer(), imageL->Width, imageL->Height, sigmaS, sigmaC1, sigmaC2);
}

void ManagedAggregators::RunYoonKweonKernel(
	CostSpace^ costSpace, InputImage^ imageL, InputImage^ imageR,
	int radius, float gammaP, float gammaC, ExecutionStream^ stream)
{
	GpuCostSpace* gpuCost = costSpace->Get();
	GpuImage* gpuImageL = imageL->Get();
	GpuImage* gpuImageR = imageR->Get();

	RunAggregationYoonKweon(gpuCost->Get(), gpuCost->GetDepth(), gpuImageL->Get(), gpuImageR->Get(), 
		gpuImageL->GetPitch(), gpuCost->GetWidth(), gpuCost->GetHeight(), radius, gammaP, gammaC);
}
