// $Id: InputImage.h 65 2010-03-18 17:06:22Z cr333 $
#pragma once

#include "GpuImage.h"
#include "ExecutionStream.h"

using namespace System;

namespace Stereo { namespace GpGpuLib {

public enum class InputImageType
{
	Any       = IMAGE_TYPE_ANY,
	Xrgb_uint = IMAGE_TYPE_XRGB32,
    Float2    = IMAGE_TYPE_FLOAT2
};

/// <summary>
/// A simple wrapper around the unmanaged image type for use in managed code
/// </summary>
public ref class InputImage
{
private:
	GpuImage* image;

public:
	InputImage()
	{ 
		image = new GpuImage();
	}
	~InputImage()
	{
		if(image != 0)
			delete image;
		image = 0;
	}

	GpuImage* Get() { return image; }

	property int Width { int get() { return image->GetWidth(); } }
	property int Height { int get() { return image->GetHeight(); } }
	property InputImageType Type { InputImageType get() { return (InputImageType)(int)image->GetType(); } }

internal:
	property int Pitch { int get() { return image->GetPitch(); } }
	unsigned int* GetPointer() { return image->Get(); }

public:

	/// <summary>
	/// Copies the data from <paramref name="source"/> into the input image, using the given cuda stream
	/// </summary>
	void AsyncCopyFrom(InputImage^ source, ExecutionStream^ stream)
	{
		image->AsyncCopyFrom(source->Get(), *stream->Get());
	}

	/// <summary>
	/// Copies image data in from <paramref name="data"/>, in xrgb format
	/// </summary>
	void CopyDataIn(array<unsigned int>^ data, InputImageType type, int w, int h)
	{
		image->Create((GpuImageType)((int)type), w, h);

		// Pin the managed pointer for the purposes of unmanaged code
		pin_ptr<unsigned int> pinData = &data[0];
		image->CopyDataIn(pinData);
	}

	/// <summary>
	/// Copies image data in from <paramref name="data"/>, which is cast internally to an unsigned int array
	/// </summary>
	void CopyDataIn(array<unsigned char>^ data, InputImageType type, int w, int h)
	{
		image->Create((GpuImageType)((int)type), w, h);

		// Pin the managed pointer for the purposes of unmanaged code
		pin_ptr<unsigned char> pinData = &data[0];
		image->CopyDataIn((unsigned int*)pinData);
	}

	/// <summary>
	/// This method is just intended to be used for testing - to output an occlusion map image.
	/// <para>Warning: The implementation includes per-call allocation and freeing of memory</para>
	/// </summary>
	void CopyDataOut(array<float>^ data, int w, int h)
	{
		if(w != image->GetWidth() || h != image->GetHeight())
			throw gcnew Exception(String::Format(
				"The width and/or height of the occlusion map ({0}x{1}) does not match the output data ({2}x{3}) - cannot copy data", 
				image->GetWidth(), image->GetHeight(), w, h));

		// Pin the managed pointer for the purposes of unmanaged code
		pin_ptr<float> pinData = &data[0];
		image->CopyDataOut(pinData);
	}

	/// <summary>
	/// This method is just intended to be used for testing - to output an occlusion map image.
	/// <para>Warning: The implementation includes per-call allocation and freeing of memory</para>
	/// </summary>
	void CopyDataOut(array<System::Byte>^ data, int w, int h)
	{
		if(w != image->GetWidth() || h != image->GetHeight())
			throw gcnew Exception(String::Format(
				"The width and/or height of the occlusion map ({0}x{1}) does not match the output data ({2}x{3}) - cannot copy data", 
				image->GetWidth(), image->GetHeight(), w, h));

		// Pin the managed pointer for the purposes of unmanaged code
		pin_ptr<System::Byte> pinData = &data[0];
		image->CopyDataOut((float*)pinData);
	}
	
	void SizeImage(InputImageType type, int w, int h)
	{
		image->Create((GpuImageType)((int)type), w, h);
	}

	void SizeToMatch(InputImage^ refImage)
	{
		SizeImage(refImage->Type, refImage->Width, refImage->Height);
	}
};

} }