{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module TCPSockIO where
import Thread
import Data.List
import TCP
import TCP.Type.Base
import Control.Monad
import Control.Exception

type SockFD = SocketID

{-----------------------------------------------------------------------
Synchronous TCP Socket I/O operations
-----------------------------------------------------------------------}

buffer_to_chunk buf@(Buffer fptr _ off len) = 
  assert (buffer_ok buf) $ Chunk fptr off len
chunk_to_buffer (Chunk fptr off len) = 
  let buf = Buffer fptr (off+len) off len
   in assert (buffer_ok buf) buf


{-----------------------------------------------------------------------
Create a listening socket on a port

Exception:
  ERROR_SOCK_LISTEN:port
-----------------------------------------------------------------------}
sock_listen :: Int -> CPSMonad SockFD
sock_listen pt = 
  do result <- sys_tcp $ SockListen (fromIntegral pt)
     case result of
      (SockNew fd) -> return fd
      (SockError s) -> sys_throw $ "ERROR_SOCK_LISTEN:=" ++ (show pt) ++ ":" ++ s

{-----------------------------------------------------------------------
Accept a socket connection

Exceptions:
  ERROR_SOCK_ACCEPT
-----------------------------------------------------------------------}
sock_accept :: SockFD -> CPSMonad SockFD
sock_accept sock = do
   result <- sys_tcp $ SockAccept sock
   case result of
      (SockNew client) -> return client
      (SockError s) -> sys_throw $ "ERROR_SOCK_ACCEPT:"++s

{-----------------------------------------------------------------------
Make an active connection 

Exceptions:
  ERROR_SOCK_CONNECT
-----------------------------------------------------------------------}
sock_connect :: String -> Int -> CPSMonad SockFD
sock_connect hostname port = do
   ipaddr <- sys_blio $ dns_lookup hostname
   when (ipaddr == 0) $ sys_throw "ERROR_SOCK_CONNECT:dns lookup failure"
   local_ipaddr <- sys_nbio $ find_local_addr ipaddr
   result <- sys_tcp $ SockConnect (IPAddr local_ipaddr) (TCPAddr ((IPAddr ipaddr), fromIntegral port))
   case result of 
      SockNew sock -> return sock
      SockError s -> sys_throw $ "ERROR_SOCK_CONNECT:" ++ s

{-----------------------------------------------------------------------
Close a socket
-----------------------------------------------------------------------}
sock_close :: SockFD -> CPSMonad Int
sock_close fd = do
  result <- sys_tcp $ SockClose fd
  return 0

{-----------------------------------------------------------------------
Receive data from a socket. Return as soon as 
  (1) some data is available, or (2) EOF is reached.

Returns: A new Chunk object pointing to the received data.
         On EOF, the new Chunk object has a zero length.

Exceptions:
  ERROR_SOCK_READ
-----------------------------------------------------------------------}
sock_recv_any :: SockFD -> CPSMonad Chunk
sock_recv_any sock =
  do result <- sys_tcp $ SockRecv sock
     case result of
       (SockData buf) -> return $ buffer_to_chunk buf
       (SockError s) -> sys_throw $ "ERROR_SOCK_READ:"++s

{-----------------------------------------------------------------------
Send data over a socket.  Keep sending until all data in the buffer
are sent.

Exceptions:
  ERROR_SOCK_WRITE
-----------------------------------------------------------------------}
sock_send_all :: SockFD -> Chunk -> CPSMonad ()
sock_send_all sock chk = 
  do rep <- sys_tcp $ SockSend sock (chunk_to_buffer chk)
     case rep of
       SockOK -> return ()
       SockError s -> sys_throw $ "ERROR_SOCK_WRITE:" ++ s  


{-----------------------------------------------------------------------
Receive data from a socket. Keep receiving until the buffer is full.

Exceptions:
  EOF
  ERROR_SOCK_READ
-----------------------------------------------------------------------}
sock_recv_all :: SockFD -> Chunk -> CPSMonad ()
sock_recv_all sock chk@(Chunk fptr off len) = undefined



{-- Higher-level wrappers ---------------------------------------------}

{-----------------------------------------------------------------------
Send a Haskell string over a socket.

Exceptions:
  ERROR_SOCK_WRITE
-----------------------------------------------------------------------}
sock_send_string :: SockFD -> String -> CPSMonad ()
sock_send_string sock s = do 
   chk <- sys_nbio $ string_to_chunk s
   sock_send_all sock chk

{-----------------------------------------------------------------------
Receive some data from a socket and return the received data as a
Haskell string.  

On EOF, an empty string is returned.

Exceptions:
  ERROR_SOCK_READ
-----------------------------------------------------------------------}
sock_recv_string :: SockFD -> CPSMonad String
sock_recv_string sock = do
   chk <- sock_recv_any sock
   s <- sys_nbio $ chunk_to_string chk
   return s

{-----------------------------------------------------------------------
Write a String to a socket with a newline.

Exception:
  ERROR_SOCK_WRITE
-----------------------------------------------------------------------}
sock_write_line sock s = sock_send_string sock (s++"\r\n")

{-----------------------------------------------------------------------
Read a line from a socket.  CR/LF are trimmed.

There is a buffer in the input arguments as well as in the output, the
buffer represents the received yet unparsed portion of the input.

Exception:
  ERROR_SOCK_READ
  EOF
-----------------------------------------------------------------------}
sock_read_line:: SockFD -> String -> CPSMonad (String, String)
sock_read_line sock buffer =
  case elemIndex '\n' buffer of
    Just idx -> 
       let line = trim $ take idx buffer
           new_buf = drop (idx+1) buffer
        in return (line, new_buf)
    Nothing -> do s <- sock_recv_string sock
                  if length s == 0 then sys_throw "EOF" else return ()
                  sock_read_line sock (buffer++s)
 where
  trim [] = []
  trim s = if last s == '\r' then take (length s - 1) s else s
