{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module AIO where
import Chunk

type AIOFD = CInt
data FILE_MODE = FM_READONLY | FM_WRITEONLY | FM_READWRITE

-----------------------------------------------------------------------------
-- Initializes the global AIO device
-- On error: program will terminate
-----------------------------------------------------------------------------
aio_setup :: IO ()
aio_setup = c_aio_setup

foreign import ccall unsafe "mod_aio.h c_aio_setup" 
  c_aio_setup :: IO ()

-----------------------------------------------------------------------------
-- Open a file.
-- This is a blocking call.
-- On error: -1
-----------------------------------------------------------------------------
aio_file_open :: String -> FILE_MODE -> IO AIOFD
aio_file_open filename mode = 
  do let param = case mode of
                    FM_READONLY -> 1
                    FM_WRITEONLY -> 2
                    FM_READWRITE -> 3
     ptr <- newCString filename
     fd <- c_file_open ptr param
     free ptr
     return fd

foreign import ccall "mod_aio.h c_file_open" 
  c_file_open :: Ptr CChar -> CInt -> IO CInt


-----------------------------------------------------------------------------
-- Close a file.
-- On error: -1
-----------------------------------------------------------------------------
aio_file_close :: AIOFD -> IO Int
aio_file_close fd = 
  do res <- c_file_close fd
     return $ fromIntegral res

foreign import ccall unsafe "mod_aio.h c_file_close" 
  c_file_close :: CInt -> IO CInt


-----------------------------------------------------------------------------
-- Submit a read request.
-- A Haskell memory object is registered as a reference
-- On error: program will terminate
-----------------------------------------------------------------------------
aio_file_read :: AIOFD -> Integer -> Chunk -> a -> IO ()
aio_file_read fd file_offset (Chunk fptr off left) ref =
  do sp <- newStablePtr ref
     let p = castStablePtrToPtr sp
     withForeignPtr fptr $
      \ptr ->
        c_file_read fd 
                    (fromIntegral file_offset) 
                    (ptr `plusPtr` off)
                    left
                    p

foreign import ccall unsafe "mod_aio.h c_file_read" 
  c_file_read :: CInt -> CLLong -> Ptr CChar -> Int -> Ptr () -> IO ()


-----------------------------------------------------------------------------
-- Wait for an AIO event.
-- Input: -1: wait forever
  --       0: return immediately
  --       x: wait at most x milliseconds
-- Output: A list of registered objects, each applied to an integer num_done.
--         For AIO read requests, the integer num_done is the number of bytes read
--             If num_done == 0, it is EOF
--             If num_done < 0, there is an error with the AIO read request
-- On other errors: program will terminate
-----------------------------------------------------------------------------
aio_wait :: Int -> IO [a]
aio_wait timeout =
  do res <- c_file_wait (fromIntegral timeout)
     if res <= 0 then
       do return []
      else
       do num_read <- c_file_getresult 0
          p <- c_file_getref 0
          let (n::Int) = fromIntegral num_read
          let sp = castPtrToStablePtr p
          ref <- deRefStablePtr sp
          freeStablePtr sp
          return [ref n]
                    
foreign import ccall "mod_aio.h c_file_wait" 
  c_file_wait :: CInt  -> IO CInt
foreign import ccall unsafe "mod_aio.h c_file_getresult" 
  c_file_getresult :: CInt -> IO CInt
foreign import ccall unsafe "mod_aio.h c_file_getref" 
  c_file_getref :: CInt -> IO (Ptr ())

