{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module WebCache
( CacheItem (..)
, STMCache
, new_cache
, stm_query_cache
, stm_update_cache
)
where
import Chunk
import Data.Map as Map
import Data.Maybe
import Thread

data CacheItem = CacheItem
    { item_name :: !String
    , item_size :: !Integer
    , item_data :: [Chunk]
    }
 deriving Show
     
empty_item = CacheItem "" 0 []

data Cache = Cache { time_stamp  :: !Int
                   , map_index   :: (Map String Int)
                   , map_data    :: (Map Int CacheItem)
                   , cache_size  :: !Integer
                   , cache_limit :: !Integer
                   }

type STMCache = TVar Cache
instance Show STMCache  where
  show x = "[Cache]"

new_cache :: Integer -> IO STMCache
new_cache limit = 
 atomically $ newTVar (Cache 1 (Map.empty) (Map.empty) 0 limit)

stm_query_cache :: STMCache -> String -> CPSMonad (Maybe CacheItem)
stm_query_cache cache name = 
 sys_nbio $ atomically $ do 
     c <- readTVar cache
     let (new_c, res) = query_cache c name
     writeTVar cache new_c
     return res

stm_update_cache :: STMCache -> CacheItem -> CPSMonad ()
stm_update_cache cache item = 
 sys_nbio $ atomically $ do 
     c <- readTVar cache
     let (new_c) = update_cache c item
     writeTVar cache new_c


query_cache :: Cache -> String -> (Cache, Maybe (CacheItem))
query_cache cache@(Cache stmp map_idx map_dat sz lm) filename =
  if Map.notMember filename map_idx then (cache,Nothing) else
     let new_stmp = stmp+1
         -- retrieve the old stuff
         idx = Map.findWithDefault 0 filename map_idx
         dat = Map.findWithDefault empty_item idx map_dat
         -- update the index to implement MRU
         new_map_idx = Map.update (\_ -> Just new_stmp) filename map_idx
         -- update the data table using new index
         new_map_data = Map.insert new_stmp dat (Map.delete idx map_dat)
       in
         (Cache new_stmp new_map_idx new_map_data sz lm, Just dat)


update_cache :: Cache -> CacheItem -> Cache
update_cache cache dat = 
  if Map.member (item_name dat) (map_index cache) then cache else
     let cache1 = shrink_cache cache $ (cache_limit cache) - (item_size dat)
      in add_to_cache cache1 dat

shrink_cache cache@(Cache stmp map_idx map_dat sz lm) expected_size = 
  if sz==0 then cache else
  if sz < expected_size then cache else
  let (idx,old_dat) = Map.findMin map_dat
      new_map_dat = Map.delete idx map_dat
      new_map_idx = Map.delete (item_name old_dat) map_idx
      new_sz = sz - (item_size old_dat)
   in shrink_cache (Cache stmp new_map_idx new_map_dat new_sz lm) expected_size
      
add_to_cache cache@(Cache stmp map_idx map_dat sz lm) dat@(CacheItem filename size chunks) = 
  let new_stmp = stmp + 1
      new_map_idx = Map.insert filename new_stmp map_idx
      new_map_dat = Map.insert new_stmp dat map_dat
      new_sz = sz + size
    in Cache new_stmp new_map_idx new_map_dat new_sz lm
