{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

import System.Posix
import Data.List
import Numeric
import System
import System.IO
import System.Exit
import Data.Maybe

{-- We borrowed a lot of code from Simon Marlow and Josef
 Svenningsson; their copyright notice is in the following modules:
 HTTPRequest, HTTPResponse and HTTPUtil. --}
import HTTPRequest
import HTTPResponse
import HTTPUtil
import WebCache

import Thread
import FileIO
import Workers

#ifdef USE_TCP
import TCPSockIO
#else
import SockIO
#endif

default_file = "index.html"
buffer_size = 16384

data Server = Server
  { srv_wwwroot     :: !String
  , srv_port        :: !Int
  , srv_n_cpu       :: !Int
  , srv_pool_size   :: !Int
  , srv_cache_limit :: !Integer
  , srv_cache_fmax  :: !Integer
  , srv_cache       :: STMCache
  } deriving Show


-------------- Helper functions ------------------------
parse_int arg = let [(i,rem)] = readDec arg in i

substring s1 s2 = isJust $ findIndex (isPrefixOf s1 ) (tails s2)

--prints fd s = sys_nbio $ putStrLn $ "[" ++ (show fd) ++ "] " ++ s
prints _ _ = return ()

--- Send a file over the socket -----------------------------

send_file server sock fsize filename =
  do -- First, is the file already in the cache?
     res <- stm_query_cache cache filename
     case res of
        Just (CacheItem _ sz chunks) -> do

           -- Yes, file in the cache, send it over the socket.
           prints sock $ "cache hit:" ++ filename ++ " size="++(show sz)
           mapM (sock_send_all sock) chunks >> return ()

        Nothing -> do 

           -- No, file not in the cache.
           fd <- file_open filename FM_READONLY
           prints sock $ "cache miss:" ++ filename ++ " fd=" ++ (show fd) ++ " size=" ++ (show fsize)
           sys_catch (
                 if (fsize <= (srv_cache_fmax server)) then do 
                      -- If the file is small enough, cache it while sending
                      chunks <- use_cache fd [] 0
                      stm_update_cache cache (CacheItem filename fsize chunks)
                 else do 
                      -- Otherwise, simply send it without caching
                      chk <- file_alloc_buffer buffer_size
                      no_cache fd chk 0
               )(\e->do file_close fd
                        sys_throw e
               )
           file_close fd
           return ()
 where
  cache = srv_cache server

  use_cache fd chunks offset =
    do chk <- file_alloc_buffer buffer_size
       chk2@(Chunk _ _ numread) <- file_read_any fd offset chk
       if (numread == 0) then return chunks 
         else do sock_send_all sock chk2
                 use_cache fd (chunks++[chk2]) (offset+(fromIntegral numread))

  no_cache fd chk offset =
    do chk2@(Chunk _ _ numread) <- file_read_any fd offset chk
       if (numread == 0) then return ()
        else do sock_send_all sock chk2
                no_cache fd chk (offset+(fromIntegral numread))

-------------- Misc. HTTP functions ------------------------

find_content_type path =
     if      isSuffixOf ".htm"  path then Just "text/html"
     else if isSuffixOf ".html" path then Just "text/html"
     else if isSuffixOf ".js"   path then Just "application/x-javascript"
     else if isSuffixOf ".gif"  path then Just "image/gif"
     else if isSuffixOf ".jpg"  path then Just "image/jpeg"
     else if isSuffixOf ".jpeg" path then Just "image/jpeg"
     else Nothing

error_page :: ResponseCode -> Response
error_page code 
    = Response {
	respCode        = code,
	respHeaders     = [],
	respCoding      = [],
	respBody        = generateErrorPage "My webserver" "1.0" code,
	respSendBody    = True,
        respContentType = Just "text/html"
      }

--- 1. Read the HTTP request ------------------------------------

read_request :: SockFD -> String -> CPSMonad ([String], String)
read_request sock remain = 
   let read_req sock lines rem =
         do (line, rem2) <- sock_read_line sock rem
            if (line == "") then return (lines, rem2)
                            else read_req sock (lines++[line]) rem2
   in read_req sock [] []

--- 2. Process the HTTP request ---------------------------------

process_request :: Server -> Request -> IO Response
process_request server req@Request{reqCmd=GetReq, reqURI=(AbsPath path)} =
  let realpath = convert_path path in 
   catch
     (do status <- getFileStatus realpath
         if isDirectory status 
           then return (error_page RC_NotFound)
           else let size = fileSize status in
                  return Response { 
                      respCode    = RC_OK,
                      respHeaders     = [],
        	      respCoding      = [],
      	              respBody        = FileBody (fromIntegral size) realpath,
      	              respSendBody    = True,
                      respContentType = find_content_type realpath
                  }  
     )
  (\e -> return (error_page RC_NotFound))
 where
 convert_path s = 
     if substring ".." s then "/illegal" 
     else (srv_wwwroot server)++s++(if (last s)=='/' then default_file else "")

process_request _ _ = return (error_page RC_NotFound)

--- 3. Send the HTTP response -------------------------------------

send_response :: Server -> SockFD -> Response -> CPSMonad ()
send_response server sock (Response{respCode=code,
                           respHeaders=headers,
                           respBody=body,
                           respSendBody=sendBody,
                           respContentType=cType }) =
  do sock_write_line sock (statusLine code)
     sock_write_line sock (serverHeader "Unify server" "0.1")
     case cType of
         Just contentType -> sock_write_line sock (contentTypeHeader contentType)
         Nothing          -> return ()
     case body of
         FileBody fsize _ -> sock_write_line sock (contentLengthHeader fsize)
	 HereItIs str     -> sock_write_line sock (contentLengthHeader (toInteger $ length str))
         _                -> return ()
     mapM (sock_write_line sock) headers
     sock_write_line sock ""
     if sendBody then case body of
         HereItIs s       -> sock_send_string sock s
         FileBody size fn -> send_file server sock size fn
         _                -> return ()
      else return ()

-----------------------------------------------------------------

http_session :: Server -> SockFD -> CPSMonad ()
http_session server sock = do 
  sys_catch ( do 

      -- 1. Read the HTTP request
      (lines,_) <- read_request sock []

      -- 2. Process the HTTP request
      response <- case parseRequest lines of
          Bad code   -> return (error_page code)
          Ok request -> sys_blio $ process_request server request

      -- 3. Send the HTTP response
      send_response server sock response

   )(\e -> prints sock $ "exception:"++e)
  sock_close sock
  return ()

-----------------------------------------------------------------

-- For each accepted connection, fork a thread to run http_session
http_server :: Server -> CPSMonad ()
http_server server = do 
  sys_catch ( do 
      -- Create a server socket
      lis <- sock_listen (srv_port server)

      loop $ do
           -- Accept connection
           sock <- sock_accept lis
                    
           -- Fork an application-level thread to process the HTTP session
           sys_fork_thread $ http_session server sock

   )(\e -> sys_nbio $ putStrLn $ "Server Socket Exception: " ++ e)
  sys_halt
  
-----------------------------------------------------------------

main = do
  args <- getArgs
  if (length args < 6) then do
       putStrLn $ "Usage: executable +RTS -H20M -N{n_cpu} -RTS n_cpu pool_size port wwwroot cache_size cache_fmax"
       putStrLn $ "  n_cpu:      Number of processors"
       putStrLn $ "  pool_size:  Thread pool size for blocking IO operations"
       putStrLn $ "  port:       Which port to listen on"
       putStrLn $ "  wwwroot:    Where the HTML files are located"
       putStrLn $ "  cache_size: Cache size (unit=MB)"
       putStrLn $ "  cache_fmax: Maximal size (unit=MB) of each cached file"
       putStrLn $ "              (files larger than this size will not be cached)"
       putStrLn $ "For example, the arguments"
       putStrLn $ "  +RTS -H20M -N1 -RTS 1 2 8888 '/var/www' 100 10"
       putStrLn $ "will use 1 CPU,  2 threads for blocking OS calls,"
       putStrLn $ "100MB for caching and the maximal cached file size is 10MB."
       exitFailure
    else return ()
  let serv = Server
       { srv_n_cpu       =  parse_int $ args !! 0
       , srv_pool_size   =  parse_int $ args !! 1
       , srv_port        =  parse_int $ args !! 2
       , srv_wwwroot     =              args !! 3
       , srv_cache_limit = (parse_int $ args !! 4) * 1048576
       , srv_cache_fmax  = (parse_int $ args !! 5) * 1048576
       , srv_cache       = undefined
       }
  cache <- new_cache $ srv_cache_limit serv
  let server = serv { srv_cache = cache }
  putStrLn $ show $ server

  default_scheduler (srv_n_cpu server) (srv_pool_size server) [http_server server]
