{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module Workers where
import Thread
import Data.Maybe
import Data.Map as Map hiding (findIndex, map)
import System.Random

#ifdef USE_TCP
import PacketIO
#endif

{-----------------------------------------------------------------------
Some standard event loops.

These event loops are isolated boxes---they need to be assembled in
the user application by interconnecting them using queues (channels).

The scheduling algorithm implemented here is fairly stupid, but you
can easily copy some of these to the application and customize them to
fit your own needs.
-----------------------------------------------------------------------}

{-----------------------------------------------------------------------
Use this loop for nonblocking sockets and pipes.
The application should call epoll_setup before using epoll.
-----------------------------------------------------------------------}
#ifdef USE_EPOLL
worker_epoll epoll_dev = loop $ do
  results <- epoll_wait epoll_dev (-1)
  mapM writeToQueue results
#endif

{-----------------------------------------------------------------------
Use this loop for nonblocking file I/O.
The application should call aio_setup before using AIO.
-----------------------------------------------------------------------}
#ifdef USE_AIO
worker_aio = loop $ do 
  results <- aio_wait (-1)
  mapM writeToQueue results
#endif

{-----------------------------------------------------------------------
Use this loop for blocking I/O.

For heavy blocking I/O, the programmer should launch a pool of OS
threads to run worker_blio.
-----------------------------------------------------------------------}
worker_blio blio_q = loop $ do 
  (io_t, info, ready_q) <- readChan blio_q
  x <- io_t
  writeToQueue ((x,info),ready_q)


{-----------------------------------------------------------------------
The main event loop.

On SMP systems, the programmer should run this loop using multiple OS
threads.
-----------------------------------------------------------------------}
worker_nbio ready_q all_qs blio_q halt_q tcp epoll_dev = loop $ do
    (trace, info) <- fetch_thread ready_q
    exec 10000 trace info
 where

 fetch_thread q = do
   res <- atomically $ do
     empty <- isEmptyTChan q
     if empty then return Nothing else do
        t <- readTChan q
        return $ Just t
   case res of
     Nothing -> do
        q1 <- random_item all_qs
        fetch_thread q1
     Just t -> return t
      
 exec :: Int -> Trace -> ThreadInfo -> IO ()
 exec steps trace info = 
  if steps == 0 then 
     writeToQueue ((trace,info),ready_q) 
  else let c = steps-1 in
  case c `seq` trace of 
  SYS_NBIO io_t             -> do t <- io_t
                                  exec c t info
  SYS_BLIO io_t             -> writeChan blio_q (io_t, info, ready_q)
  SYS_FORK_THREAD new t     -> do writeToQueue ((new,initial_info),ready_q)
                                  exec c t info
  SYS_YIELD trace           -> writeToQueue ((trace,info),ready_q)
  SYS_HALT                  -> writeChan halt_q ()
#ifdef USE_EPOLL
  SYS_EPOLL_WAIT fd ev t    -> epoll_add epoll_dev fd ev ((t, info), ready_q)
#endif
#ifdef USE_EXCEPTION
  SYS_CATCH trace f g       -> exec c trace ((f,g):info)
  SYS_THROW e               -> case info of       
                                (g, _):es_tail   -> exec c (g e) es_tail
                                [] -> putStrLn "Uncatched exception";
  SYS_RET                   -> case info of       
                                ((_, h):es_tail) -> exec c h es_tail
                                [] -> return ()
#else
  SYS_RET                   -> return ()
#endif
#ifdef USE_AIO
  SYS_AIO_READ fd off chk f -> aio_file_read fd off chk (\n->((f n,info), ready_q))
#endif
#ifdef USE_TCP
  SYS_TCP req f             -> do (tr,inf) <-exec_tcp req (\r->((f r,info), ready_q)) tcp
                                  exec c tr inf
#endif
#ifdef USE_MUTEX
  SYS_MUTEX m op f          -> do (tr,inf)<-exec_mutex m op ((f,info),ready_q)
                                  exec c tr inf
#endif


--- Some helper functions -------------------------------------------
loop m = do m; loop m
{-# INLINE loop #-}

loop_until_zero mi = do i <- mi; if i==0 then return () else loop_until_zero mi
{-# INLINE loop_until_zero #-}

writeToQueue (st,ready_q) = atomically $ writeTChan ready_q st
{-# INLINE writeToQueue #-}

rollDice :: Int -> IO Int
rollDice mx = getStdRandom (randomR (0,mx-1))
 
random_item lst = do
  let len = length lst
  idx <- rollDice len
  return $ lst !! idx
  
#ifdef USE_MUTEX
{-----------------------------------------------------------------------
A subroutine for processing mutex system calls.
-----------------------------------------------------------------------}
exec_mutex :: Mutex -> Mutex_OP -> ThreadCont -> IO ThreadState
exec_mutex mutex op cont@(st,_) = 
  modifyMVar mutex $ \(locked,queue) ->
    case op of
     MUTEX_LOCK -> 
       if locked then return ((True, cont:queue), (SYS_RET,initial_info))
                 else return ((True,[]), st)
     MUTEX_UNLOCK->
       if length queue==0 
           then return ((False,[]),st)
           else do let (st1,ready_q) = (last queue)
                   atomically $ writeTChan ready_q st
                   return ((True,(take (length queue - 1) queue)),st)
#endif

#ifdef USE_TCP
------------------------------------------------------------------------
-- TCP stuff

exec_tcp :: SockReq -> (SockRsp -> ThreadCont) 
            -> MVar (Host ThreadCont) -> IO ThreadState
exec_tcp req f mhost = do
  h <- takeMVar mhost
  let (h1, res) = tcp_user_req (req, f) h
  putMVar mhost h1
  collect_result mhost
  case res of
    Just (st,_) -> return st
    Nothing -> return (SYS_RET,[])  

worker_tcp_timer mhost = do
  epoll_dev <- epoll_create 1
  loop $ do
    epoll_wait epoll_dev 100
    h <- takeMVar mhost 
    curr_time <- get_current_time
    h1 <- tcp_timer_check curr_time h
    putMVar mhost $ tcp_timer curr_time h1
    collect_result mhost

worker_tcp_input mhost = loop $ do
    mb <- packet_recv_ip
    case mb of
      Nothing -> return ()
      Just msg -> do
       h <- takeMVar mhost
       putMVar mhost $ tcp_packet_in (fromJust mb) h
       collect_result mhost 

collect_result mhost = do
  h <- takeMVar mhost
  let (h1,ready) = tcp_user_rsp h
      (h2,out)  = tcp_packet_out h1
  mapM writeToQueue ready
  mapM packet_send_ip out
  putMVar mhost $ h2
#endif

-------------------------------------------------------------------------
-- The default scheduler that implements everything:

default_scheduler :: Int -> Int -> [CPSMonad ()] -> IO ()
default_scheduler num_nbio num_blio l = do 
     -- Create the task queues
     ready_qs <- mapM (\_ -> atomically newTChan) [1..num_nbio]
     blio_q <- newChan
     halt_q <- newChan

     -- Initialize the ready queues
     mapM (\(q,task)-> 
              atomically $ writeTChan q (thread_to_trace task, initial_info)
          ) (zip (cycle ready_qs) l)
     --writeList2TChan (head ready_qs) (map (\x-> ((thread_to_trace x),[])) l)

#ifdef USE_EPOLL
     -- Spawn the epoll event loop
     epoll_setup
     epoll_dev <- epoll_create 131072
     forkIO (worker_epoll epoll_dev)
#else
     let epoll_dev = ()
#endif

#ifdef USE_AIO
     -- Spawn the AIO event loop
     aio_setup
     forkIO worker_aio
#endif

#ifdef USE_TCP
     -- Initialize TCP
     packet_setup
     curr_time <- get_current_time
     mhost <- newMVar $ tcp_init_host curr_time (8888:[20000..29999])
     forkIO (worker_tcp_timer mhost)
     forkIO (worker_tcp_input mhost)
#else
     let mhost = ()
#endif

     -- Launch a pool of worker threads to process blocking I/O operations
     mapM (\_ -> forkIO (worker_blio blio_q)) [1..num_blio] 
     -- The main event loop
     mapM (\q -> forkIO (worker_nbio q ready_qs blio_q halt_q mhost epoll_dev)) ready_qs

     -- Wait until termination
     readChan halt_q
