{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module Thread 
(module Thread
,module Chunk
,module NBIO
,module Util
,module Control.Concurrent
,module Control.Concurrent.STM
#ifdef USE_EPOLL
,module Epoll
#endif
#ifdef USE_AIO
,module AIO
#endif
#ifdef USE_TCP
,module TCP
#endif
)
where
import Chunk
import NBIO
import Util
import Control.Concurrent
import Control.Concurrent.STM

#ifdef USE_EPOLL
import Epoll
#endif
#ifdef USE_AIO
import AIO
#endif
#ifdef USE_TCP
import TCP
#endif

--------------------------------------------------------------------------
-- Exception:
-- Each thread has an "exception stack", where each stack frame is a pair:
--   (Exception Continuation, Normal Continuation)
--------------------------------------------------------------------------
#ifdef USE_EXCEPTION
type Exception = String
type ExceptionFrame = (Exception->Trace, Trace)
type ExceptionStack = [ExceptionFrame]
#endif

--------------------------------------------------------------------------
-- Mutex:
-- Each mutex is internally represented as a pair: (Locked, Queued threads)
--------------------------------------------------------------------------
#ifdef USE_MUTEX
type Mutex = MVar (Bool, [ThreadCont])  
data Mutex_OP = MUTEX_LOCK | MUTEX_UNLOCK
#endif

--------------------------------------------------------------------------
-- Thread representation
-- Each thread is represented by its Trace and its exception stack
-- More can be added here, such as threadID, etc...
--------------------------------------------------------------------------
#ifdef USE_EXCEPTION
type ThreadInfo = ExceptionStack
initial_info = []
#else
type ThreadInfo = ()
initial_info = ()
#endif

type CPUQueue = TChan ThreadState
type ThreadState = (Trace, ThreadInfo)
type ThreadCont = (ThreadState, CPUQueue)

--------------------------------------------------------------------------
-- Trace nodes
--------------------------------------------------------------------------
data Trace = 
        SYS_NBIO (IO Trace)  -- Nonblocking IO 
      | SYS_BLIO (IO Trace)  -- Blocking IO
      | SYS_FORK_THREAD Trace Trace
      | SYS_RET
      | SYS_HALT
      | SYS_YIELD Trace
#ifdef USE_EPOLL
      | SYS_EPOLL_WAIT !EpollFD !EPOLL_EVENT Trace
#endif
#ifdef USE_AIO
      | SYS_AIO_READ !AIOFD !Integer !Chunk (Int -> Trace)
#endif
#ifdef USE_EXCEPTION
      | SYS_THROW Exception
      | SYS_CATCH Trace (Exception->Trace) Trace
#endif
#ifdef USE_TCP
      | SYS_TCP !SockReq (SockRsp-> Trace)
#endif
#ifdef USE_MUTEX
      | SYS_MUTEX !Mutex !Mutex_OP Trace
#endif

--------------------------------------------------
-- CPS Monad
--------------------------------------------------
newtype CPSMonad a = CPSMonad ((a-> Trace)-> Trace)
instance Monad CPSMonad where
   m >>= f = cps_bind m f
   return  = cps_return
   {-# INLINE (>>=) #-}
   {-# INLINE return #-}

cps_bind :: CPSMonad a -> (a -> CPSMonad b) -> CPSMonad b
cps_bind (CPSMonad x) f = CPSMonad(\y-> x (\z-> let CPSMonad f' = f z in f' y))
cps_return :: a -> CPSMonad a
cps_return x = CPSMonad(\c -> c x)
{-# INLINE cps_bind #-}
{-# INLINE cps_return #-}

--------------------------------------------------
-- Top-level conversion from threaded code to its trace
--------------------------------------------------

thread_to_trace :: (CPSMonad a) -> Trace
thread_to_trace (CPSMonad f) = f (\c-> SYS_RET)

--------------------------------------------------
-- system calls
--------------------------------------------------
-- Nonblocking IO operation
sys_nbio :: IO a -> CPSMonad a
sys_nbio f = 
    CPSMonad $ \c -> SYS_NBIO( do x <- f; return (c x) )
{-# INLINE sys_nbio #-}

-- Potentially blocking IO operation
sys_blio :: IO a -> CPSMonad a
sys_blio f = 
    CPSMonad $ \c -> SYS_BLIO( do x <- f; return (c x) )
{-# INLINE sys_blio #-}

sys_fork_thread :: CPSMonad a -> CPSMonad ()
sys_fork_thread t = 
    CPSMonad $ \c -> SYS_FORK_THREAD (thread_to_trace t) (c ())

sys_yield :: CPSMonad ()
sys_yield = 
    CPSMonad $ \c -> SYS_YIELD (c ())


sys_halt :: CPSMonad ()
sys_halt = 
    CPSMonad $ \c -> SYS_HALT

#ifdef USE_EPOLL
sys_epoll_wait :: EpollFD -> EPOLL_EVENT -> CPSMonad ()
sys_epoll_wait epoll_fd event = 
    CPSMonad $ \c -> SYS_EPOLL_WAIT epoll_fd event (c ())
#endif

#ifdef USE_TCP
sys_tcp :: SockReq -> CPSMonad SockRsp
sys_tcp req = 
    CPSMonad $ \c -> SYS_TCP req (\rsp -> c rsp)
#endif

#ifdef USE_AIO
sys_aio_read :: AIOFD -> Integer -> Chunk -> CPSMonad Int
sys_aio_read file_fd offset chk = 
    CPSMonad $ \c -> SYS_AIO_READ file_fd offset chk (\num_read -> c num_read)
#endif

#ifdef USE_EXCEPTION
sys_throw :: Exception -> CPSMonad a
sys_throw e =
    CPSMonad $ \c -> SYS_THROW e

sys_catch:: CPSMonad () -> (Exception -> CPSMonad ()) -> CPSMonad ()
sys_catch c1 c2 = 
    CPSMonad $ \c -> SYS_CATCH (thread_to_trace c1)
                               (\e-> let (CPSMonad f) = (c2 e) in f (\_ -> c ()))
                               (c ())
#endif

#ifdef USE_MUTEX
sys_mutex :: Mutex -> Mutex_OP -> CPSMonad ()
sys_mutex mutex op =
   CPSMonad $ \c -> SYS_MUTEX mutex op (c ())
#endif

