{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module FileIO where

import Thread
import Data.IORef

{-----------------------------------------------------------------------
Synchronous File I/O operations

These functions are intended to provide a higher-level interface to
the low-level system calls and nonblocking IO operations.  The
programming model is synchronous and errors are handled using
exceptions.
-----------------------------------------------------------------------}


{-----------------------------------------------------------------------
Open a file
Exception:  
   ERROR_FILE_OPEN:filename
-----------------------------------------------------------------------}
file_open :: String -> FILE_MODE -> CPSMonad AIOFD
file_open filename mode =
 do fd <- sys_blio $ aio_file_open filename mode
    if fd < 0 then sys_throw ("ERROR_FILE_OPEN:"++filename) else return ()
    return fd


{-----------------------------------------------------------------------
Allocate an aligned memory block for AIO
-----------------------------------------------------------------------}
file_alloc_buffer size = sys_nbio $ new_aligned_chunk size


{-----------------------------------------------------------------------
Read a file

Returns: a new Chunk object pointing to the received data chunk.
On EOF, the returned Chunk object has a zero length        

Exception:
  FILE_READ_ERROR
-----------------------------------------------------------------------}

file_read_any :: AIOFD -> Integer -> Chunk -> CPSMonad Chunk
file_read_any fd offset chk@(Chunk fptr off len) = 
  do numread <- sys_aio_read fd offset chk
     if numread<0 then sys_throw "FILE_READ_ERROR" else return ()
     return $ Chunk fptr off numread


{-----------------------------------------------------------------------
To close a file, simply use "sys_nbio $ aio_file_close fd".
It is probably not worth writing a wrapper for it...
-----------------------------------------------------------------------}
file_close fd = sys_nbio $ aio_file_close fd
