{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module TCP.Aux.HostMonad where

import TCP.Type.Base
import TCP.Type.Socket
import Data.Map as Map
import TCP.Aux.Misc
import Control.Exception

newtype HMonad t a = HMonad (Host t -> (a, Host t))

instance Monad (HMonad t) where
    return a = HMonad $ \s -> (a,s)
    x >>= f = bindHMonad x f
    {-# INLINE return #-}
    {-# INLINE (>>=) #-}

bindHMonad :: HMonad t a -> (a -> HMonad t b) -> HMonad t b
bindHMonad (HMonad x) f = 
        HMonad $ \s -> let (res, s') = x s
                           (HMonad z) = f res in z s'
{-# INLINE bindHMonad #-}

get_host :: HMonad t (Host t)
get_host   = HMonad $ \s -> (s,s)

put_host h = HMonad $ \_ -> ((),h)

modify_host f = HMonad $ \s -> ((), f s)
emit_segs  segs = modify_host $ \h -> h { output_queue = (output_queue h)++ segs}
emit_ready threads = modify_host $ \h -> h { ready_list = (ready_list h)++ threads}

{-# INLINE get_host #-}
{-# INLINE put_host #-}
{-# INLINE modify_host #-}
{-# INLINE emit_segs #-}
{-# INLINE emit_ready #-}

-----------------------------------------------------
has_sock :: SocketID -> HMonad t Bool
has_sock sid = do
  h <- get_host   
  return $ Map.member sid (sock_map h)

lookup_sock sid = do
  h <- get_host
  res <- Map.lookup sid (sock_map h)
  return res

delete_sock :: SocketID -> HMonad t ()
delete_sock sid =
  modify_host $ \h -> h {sock_map= Map.delete sid (sock_map h)}

update_sock sid f =
  modify_host $ \h -> h {sock_map= Map.adjust f sid (sock_map h)}

insert_sock :: SocketID -> TCPSocket t -> HMonad t ()
insert_sock sid sock =
  modify_host $ \h -> h { sock_map = Map.insert sid sock (sock_map h) }

{-# INLINE has_sock #-}
{-# INLINE lookup_sock #-}
{-# INLINE delete_sock #-}
{-# INLINE update_sock #-}
{-# INLINE insert_sock #-}

------------------

runHMonad_ :: (HMonad t a) -> (Host t) -> (Host t,a)
runHMonad_ (HMonad m) inithost =
   let (res, finalhost) = m inithost in 
     (finalhost, res)

runHMonad :: (HMonad t a) -> (Host t) -> (Host t)
runHMonad hm inithost =
   let (h, res) = runHMonad_ hm inithost in h
