{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module NBIO

where
import System.IO
import Foreign
import Foreign.C
import Chunk
import Data.Maybe

type NBIO_FD = CInt

-----------------------------------------------------------------------------
-- Creates a listening socket on a port
-- On error: returns a fd<0
-----------------------------------------------------------------------------
nbio_listen  :: Int -> IO NBIO_FD
nbio_listen port = 
  c_sock_listen (fromIntegral port)
foreign import ccall unsafe "mod_nbio.h c_sock_listen" 
  c_sock_listen :: CInt -> IO CInt

-----------------------------------------------------------------------------
-- Create a pair of FIFO pipes (read_fd, write_fd)
-- On error: the program will terminate
-----------------------------------------------------------------------------
nbio_pipe  :: IO (NBIO_FD, NBIO_FD)
nbio_pipe =
  do x <- c_pipe_create
     let rfd  = fromIntegral $ x `shiftL` 32 `shiftR` 32
     let wfd  = fromIntegral $ x `shiftR` 32
     return (rfd,wfd)
foreign import ccall unsafe "mod_nbio.h c_pipe_create"
  c_pipe_create :: IO CLLong

-----------------------------------------------------------------------------
-- Returns the socket immediately, a subsequent EPOLL_WRITE indicates completion
-- On instant error: returns fd<0
-- On connection failure: subsequent read/writes to the socket will fail
-----------------------------------------------------------------------------
nbio_connect :: Word32 -> Int -> IO NBIO_FD
nbio_connect addr port = 
  c_sock_connect addr (fromIntegral port)

foreign import ccall unsafe "mod_nbio.h c_sock_connect" 
  c_sock_connect :: Word32 -> CInt -> IO CInt


-----------------------------------------------------------------------------
-- Returns (Just fd) if successful
--         Nothing   if no more connections to accept, a subsequent EPOLL_READ indicates more connections can be accepted. 
-- On failure:
--   fd<=0, 
-----------------------------------------------------------------------------
nbio_accept  :: CInt -> IO (Maybe NBIO_FD)
nbio_accept server_fd =
  do fd <- c_sock_accept server_fd
     if fd==(-2) then return Nothing
                 else return (Just fd)
foreign import ccall unsafe "mod_nbio.h c_sock_accept" 
  c_sock_accept :: CInt -> IO CInt


-----------------------------------------------------------------------------
-- Returns (Just num_read) if successful
--         Nothing         if nothing to read (should wait for EPOLL_READ)
-- On failure:
--   num_read == -1 if an error has occured
--   num_read == 0 if EOF
-----------------------------------------------------------------------------
nbio_read   :: NBIO_FD -> Chunk -> IO (Maybe Int)
nbio_read fd (Chunk fptr off left) =
  withForeignPtr fptr $
   \(ptr::Ptr CChar) ->
     do res <- c_sock_read fd (ptr `plusPtr` off) (fromIntegral left)
        if res==(-2) then return Nothing
                     else return $ Just $ fromIntegral res
foreign import ccall unsafe "mod_nbio.h c_sock_read" 
  c_sock_read :: CInt -> Ptr CChar -> CInt -> IO CInt


-----------------------------------------------------------------------------
-- Returns (Just num_written) if successful
--         Nothing            if write would block (should wait for EPOLL_WRITE)
-- On failure:
--   num_written == -1 if an error has occured
-----------------------------------------------------------------------------
nbio_write  :: NBIO_FD -> Chunk -> IO (Maybe Int)
nbio_write fd (Chunk fptr off left) =
  withForeignPtr fptr $
   \(ptr::Ptr CChar) ->
     do res <- c_sock_write fd (ptr `plusPtr` off) (fromIntegral left)
        if res==(-2) then return Nothing
                     else return $ Just $ fromIntegral $ res
foreign import ccall unsafe "mod_nbio.h c_sock_write" 
  c_sock_write :: CInt -> Ptr CChar -> CInt -> IO CInt

-----------------------------------------------------------------------------
-- Returns 0 on success, -1 on failure
-----------------------------------------------------------------------------
nbio_close  :: NBIO_FD -> IO Int
nbio_close fd = 
  do res <- c_close fd
     return $ fromIntegral res
foreign import ccall unsafe "mod_nbio.h c_close" 
  c_close :: CInt -> IO CInt


